#' Generate the QC panel of the shiny app
#' @description These are the UI and server components of the QC panel of the 
#' shiny app. It is generated by including 'QC' in the panels.default argument
#' of \code{\link{generateShinyApp}}.
#' @inheritParams DEpanel
#' @return The UI and Server components of the shiny module, that can be used
#' within the UI and Server definitions of a shiny app.
#' @name QCpanel
NULL

#' @rdname QCpanel
#' @export
QCpanelUI <- function(id, metadata){
  ns <- NS(id)
  
  tabPanel(
    'Quality checks',
    tags$h1("Jaccard Similarity Index Heatmap"),
    shinyWidgets::dropdownButton(
      shinyjqui::orderInput(ns('jaccard.annotations'), label = "Show annotations", items = colnames(metadata)),
      sliderInput(ns('jaccard.n.abundant'), label = '# of (most abundant) genes',
                  min = 50, value = 500, max = 5000, step = 50, ticks = FALSE),
      checkboxInput(ns("jaccard.show.values"), label = "Show JSI values", value = FALSE),
      textInput(ns('plotJSIFileName'), 'File name for JSI plot download', value ='JSIPlot.png'),
      downloadButton(ns('downloadJSIPlot'), 'Download JSI Plot'),
      
      status = "info",
      icon = icon("gear", verify_fa = FALSE), 
      tooltip = shinyWidgets::tooltipOptions(title = "Click to see inputs!")
    ),
    plotOutput(ns('jaccard')),
    
    tags$h1("Principal Component Analysis"),
    shinyWidgets::dropdownButton(
      radioButtons(ns('pca.annotation'), label = "Group by",
                   choices = colnames(metadata), selected = colnames(metadata)[ncol(metadata)]),
      sliderInput(ns('pca.n.abundant'), label = '# of (most abundant) genes',
                  min = 50, value = 500, max = 5000, step = 50, ticks = FALSE),
      checkboxInput(ns("pca.show.labels"), label = "Show sample labels", value = FALSE),
      checkboxInput(ns('pca.show.ellipses'),label = "Show ellipses around groups",value=TRUE),
      textInput(ns('plotPCAFileName'), 'File name for PCA plot download', value ='PCAPlot.png'),
      downloadButton(ns('downloadPCAPlot'), 'Download PCA Plot'),
      
      status = "info",
      icon = icon("gear", verify_fa = FALSE), 
      tooltip = shinyWidgets::tooltipOptions(title = "Click to see inputs!")
    ),
    plotOutput(ns('pca')),
    
    tags$h1("MA plots"),
    shinyWidgets::dropdownButton(
      checkboxInput(ns("ma.show.guidelines"), label = "Show guidelines", value = TRUE),
      selectInput(ns('ma.sample1'), 'Sample 1', choices = metadata[, 1], selected = metadata[1, 1]),
      selectInput(ns('ma.sample2'), 'Sample 2', choices = metadata[, 1], selected = metadata[2, 1]),
      textInput(ns('plotMAFileName'), 'File name for MA plot download', value = 'MAPlot.png'),
      downloadButton(ns('downloadMAPlot'), 'Download MA Plot'),
      
      status = "info",
      icon = icon("gear", verify_fa = FALSE), 
      tooltip = shinyWidgets::tooltipOptions(title = "Click to see inputs!")
    ),
    plotOutput(ns('ma'))
  )
}

#' @rdname QCpanel
#' @export
QCpanelServer <- function(id, expression.matrix, metadata, anno){
  # check whether inputs (other than id) are reactive or not
  stopifnot({
    is.reactive(expression.matrix)
    is.reactive(metadata)
    !is.reactive(anno)
  })
  
  moduleServer(id, function(input, output, session){
    observe({
      items <- colnames(metadata())
      include.exclude <- apply(metadata(), 2, function(x){
        l <- length(unique(x))
        (l > 1) & (l < length(x))
      })
      items <- colnames(metadata())[include.exclude]
      items <- items[c(length(items), seq_len(length(items) - 1))]
      shinyjqui::updateOrderInput(session, "jaccard.annotations", items = items)
    })
    jaccard.plot <- reactive({
      meta <- lapply(metadata(), function(x) factor(x, levels = unique(x))) %>% 
        as.data.frame() %>%
        dplyr::arrange(dplyr::across(input[['jaccard.annotations']]))
      
      myplot <- jaccard_heatmap(
        expression.matrix = expression.matrix()[, meta[, 1]],
        metadata = meta,
        top.annotation.ids = match(input[['jaccard.annotations']], colnames(meta)),
        n.abundant = input[['jaccard.n.abundant']], 
        show.values = input[["jaccard.show.values"]],
        show.row.column.names = (nrow(meta) <= 20)
      )
      myplot 
    })
    output[['jaccard']] <- renderPlot(jaccard.plot())
    
    pca.plot <- reactive({
      myplot <- plot_pca(
        expression.matrix = expression.matrix(),
        metadata = metadata(),
        annotation.id = match(input[['pca.annotation']], colnames(metadata())),
        n.abundant = input[['pca.n.abundant']],
        show.labels = input[['pca.show.labels']],
        show.ellipses = input[['pca.show.ellipses']]
      )
      myplot
    })
    output[['pca']] <- renderPlot(pca.plot())
    
    observe({
      updateSelectInput(session, 'ma.sample1', choices = metadata()[, 1], selected = metadata()[1, 1])
      updateSelectInput(session, 'ma.sample2', choices = metadata()[, 1], selected = metadata()[2, 1])
    })
    ma.plot <- reactive({
      gene_id <- NULL; exp1 <- NULL; exp2 <- NULL; l1 <- NULL; l2 <- NULL
      df <- tibble::tibble(
        gene_id = rownames(expression.matrix()),
        gene_name = anno$NAME[match(gene_id, anno$ENSEMBL)],
        exp1 = expression.matrix()[, match(input[['ma.sample1']], colnames(expression.matrix()))],
        exp2 = expression.matrix()[, match(input[['ma.sample2']], colnames(expression.matrix()))],
        l1 = log2(exp1),
        l2 = log2(exp2),
        log2exp = (l1 + l2) / 2,
        log2FC = l1 - l2,
        pval = 1,
        pvalAdj = 1
      )  %>%
        dplyr::filter(exp1 != 0, exp2 != 0)
      myplot <- ma_plot(
        genes.de.results = df,
        alpha = 0.05,
        add.colours = TRUE,
        point.colours = rep(scales::hue_pal()(1), 4),
        add.expression.colour.gradient = FALSE,
        add.guide.lines = input[['ma.show.guidelines']],
        guide.line.colours = rep("gray60", 2),
        add.labels.auto = FALSE,
        add.labels.custom = FALSE,
      )
      myplot
    })
    output[['ma']] <- renderPlot(ma.plot())
    
    output[['downloadJSIPlot']] <- downloadHandler(
      filename = function() { input[['plotJSIFileName']] },
      content = function(file) {
        grDevices::png(file)
        print(jaccard.plot())
        grDevices::dev.off()
        
      }
    )
    
    output[['downloadPCAPlot']] <- downloadHandler(
      filename = function() { input[['plotPCAFileName']] },
      content = function(file) {
        device <- function(..., width, height) grDevices::png(..., width = width, height = height, res = 300, units = "in")
        ggsave(file, plot = pca.plot(), device = device)
      }
    )
    
    output[['downloadMAPlot']] <- downloadHandler(
      filename = function() { input[['plotMAFileName']] },
      content = function(file) {
        device <- function(..., width, height) grDevices::png(..., width = width, height = height, res = 300, units = "in")
        ggsave(file, plot = ma.plot(), device = device)
      }
    )
  })
}

# QCpanelApp <- function(){
#   shinyApp(
#     ui = fluidPage(QCpanelUI('qc', metadata)),
#     server = function(input, output, session){
#       QCpanelServer('qc', expression.matrix, metadata)
#     }
#   )
# }