% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dayavg.R
\name{dayavg}
\alias{dayavg}
\title{Determine daily averages}
\usage{
dayavg(var, infile, outfile, nc34 = 4, overwrite = FALSE, verbose = FALSE)
}
\arguments{
\item{var}{Name of NetCDF variable (character).}

\item{infile}{Filename of input NetCDF file. This may include the directory
(character).}

\item{outfile}{Filename of output NetCDF file. This may include the directory
(character).}

\item{nc34}{NetCDF version of output file. If \code{nc34 = 3} the output file will be
in NetCDFv3 format (numeric). Default output is NetCDFv4.}

\item{overwrite}{logical; should existing output file be overwritten?}

\item{verbose}{logical; if TRUE, progress messages are shown}
}
\value{
A NetCDF file including a time series of daily averages is written.
}
\description{
The function determines daily averages from data of a single CM SAF NetCDF input
file. There is a difference between the operators dayavg and daymean.
The mean is regarded as a statistical function, whereas the average is found
simply by adding the sample members and dividing the result by the sample size.
For example, the mean of 1, 2, miss and 3 is (1 + 2 + 3)/3 = 2,
whereas the average is (1 + 2 + miss + 3)/4 = miss/4 = miss.
If there are no missing values in the sample, the average and mean are identical.
}
\examples{
## Create an example NetCDF file with a similar structure as used by CM
## SAF. The file is created with the ncdf4 package.  Alternatively
## example data can be freely downloaded here: <https://wui.cmsaf.eu/>

library(ncdf4)

## create some (non-realistic) example data

lon <- seq(5, 15, 0.5)
lat <- seq(45, 55, 0.5)
time <- seq(ISOdate(2000, 1, 1), ISOdate(2000, 1, 6), "hours")
origin <- as.Date("1983-01-01 00:00:00")
time <- as.numeric(difftime(time, origin, units = "hour"))
data <- array(250:350, dim = c(21, 21, 121))

## create example NetCDF

x <- ncdim_def(name = "lon", units = "degrees_east", vals = lon)
y <- ncdim_def(name = "lat", units = "degrees_north", vals = lat)
t <- ncdim_def(name = "time", units = "hours since 1983-01-01 00:00:00",
 vals = time, unlim = TRUE)
var1 <- ncvar_def("SIS", "W m-2", list(x, y, t), -1, prec = "short")
vars <- list(var1)
ncnew <- nc_create(file.path(tempdir(),"CMSAF_example_file.nc"), vars)
ncvar_put(ncnew, var1, data)
ncatt_put(ncnew, "lon", "standard_name", "longitude", prec = "text")
ncatt_put(ncnew, "lat", "standard_name", "latitude", prec = "text")
nc_close(ncnew)

## Determine the daily averages of the example CM SAF NetCDF file and
## write the output to a new file.
dayavg(var = "SIS", infile = file.path(tempdir(),"CMSAF_example_file.nc"), 
 outfile = file.path(tempdir(),"CMSAF_example_file_dayavg.nc"))

unlink(c(file.path(tempdir(),"CMSAF_example_file.nc"), 
 file.path(tempdir(),"CMSAF_example_file_dayavg.nc")))
}
\seealso{
Other daily statistics: 
\code{\link{daymax}()},
\code{\link{daymean}()},
\code{\link{daymin}()},
\code{\link{daypctl}()},
\code{\link{dayrange}()},
\code{\link{daysd}()},
\code{\link{daysum}()},
\code{\link{dayvar}()},
\code{\link{ydaymax}()},
\code{\link{ydaymean}()},
\code{\link{ydaymin}()},
\code{\link{ydayrange}()},
\code{\link{ydaysd}()},
\code{\link{ydaysum}()}
}
\concept{daily statistics}
