#' Acrophase Correction
#'
#' @encoding UTF-8
#' @param x An object generated by the \code{cosinor.lm} function.
#' @description Corrects the value of the acrophase parameter of the cosinor model, placing it in the appropriate quadrant.
#' @details The acrophase parameter of a cosinor model is found by solving an equation with inverse tangent of an expression which contains linearized cosinor parameters. However, multiple numeric entities may result in a same value of tangent and just calculating the inverse tangent may result with the wrong value of the acrophase. This function corrects the acrophase from the \code{cosinor.lm} object according to the procedure from Bingham et al. (1982).
#' More specifically, the acrophase is calculated as: \deqn{K + g * arctan \vert\frac{\gamma}{\beta}\vert}
#' where values of \eqn{K} and \eqn{g} depend on the signs of \eqn{\beta} and \eqn{\gamma} and can be derived from the following table:
#' \tabular{cccc}{
#' sign \eqn{\beta} \tab sign \eqn{\gamma} \tab K \tab g \cr
#' + \tab + \tab 0 \tab -1 \cr
#' + \tab - \tab -2\eqn{\pi} \tab 1 \cr
#' - \tab + \tab -\eqn{\pi} \tab 1 \cr
#' - \tab - \tab -\eqn{\pi} \tab -1
#' }
#' @examples
#' fit.temperature<-cosinor.lm(Temperature~time(Time), period = 48, data = temperature_zg)
#' correct.acrophase(fit.temperature)
#' @references Bingham, C., Arbogast, B., Guillaume Cornélissen, G., Lee, J.K. & Halberg, F. (1982). Inferential Statistical Methods for Estimating and Comparing Cosinor Parameters. \emph{Chronobiologia}, \emph{9(4)}, 397-439.
#' @import cosinor ggplot2 matrixStats Hmisc
#' @importFrom graphics par
#' @importFrom stats cor cov lm pf sd
#' @export

correct.acrophase<-function(x){
  coefs<-data.frame(t(x$fit$coefficients))
  if(coefs$rrr>0 & coefs$sss>0){
    acrophase<-0+(-1*atan(abs(coefs$sss/coefs$rrr)))
  }
  else if(coefs$rrr>0 & coefs$sss<0){
    acrophase<-2*-1*pi+(1*atan(abs(coefs$sss/coefs$rrr)))
  }
  else if(coefs$rrr<0 & coefs$sss>0){
    acrophase<-pi*-1+(1*atan(abs(coefs$sss/coefs$rrr)))
  }
  else {
    acrophase<-pi*-1+(-1*atan(abs(coefs$sss/coefs$rrr)))
  }
  return(acrophase)
}

#' Percent Rhythm
#'
#' @param x An object generated by the \code{cosinor.lm} or \code{population.cosinor.lm} function.
#' @description Calculates Percent Rhythm, the measure of the relative strength of a rhythm.
#' @details Percent Rhythm is the coefficient of determination obtained by squaring the correlation between observed and estimated values.
#' @examples
#' fit.temperature<-cosinor.lm(Temperature~time(Time), period = 48, data = temperature_zg)
#' cosinor.PR(fit.temperature)
#'
#' fit.november<-population.cosinor.lm(timecol = 1, firstsubj = 2, lastsubj = 20, data =
#' PANAS_november, period = 7, na.action = "na.exclude")
#' cosinor.PR(fit.november)
#' @import cosinor ggplot2 matrixStats Hmisc
#' @importFrom graphics par
#' @importFrom stats cor cov lm pf sd
#' @export

cosinor.PR<-function(x){
  if (exists("pop.mat", where = x) == T) {
    corr<-rcorr(x$fitted.values,x$emp.mean)
    PR<-data.frame(cbind(corr$r[[2]],corr$r[[2]]^2,corr$P[[2]]))
    colnames(PR)<-c("r","Percent rhythm","p-value")
  }
  else {
    corr<-rcorr(x$fit$fitted.values,x$fit$model[,1])
    PR<-data.frame(cbind(corr$r[[2]],corr$r[[2]]^2,corr$P[[2]]))
    colnames(PR)<-c("r","Percent rhythm","p-value")
  }
  return(PR)
}

#' Rhythm Detection Test
#'
#' @param x An object generated by the \code{cosinor.lm} or \code{population.cosinor.lm} functions.
#' @description Performs the rhythm detection test, a global test for the significance of the estimated model for single cosinor and population-mean cosinor.
#' @details The rhythm detection test, also called the zero-amplitude test, tests the overall significance of the cosinor model. The test is actually an F-ratio and is calculated as following (according to the procedure described in Cornélissen, 2014):
#' \deqn{F = \frac{\frac{\sum(\widehat{Y}_i - \bar{Y})^2}{2}}{\frac{\sum(Y_i - \bar{Y})^2}{N - 3}}} with \eqn{df_1 = 2} and \eqn{df_2 = N - 3}
#' , where \eqn{\widehat{Y}_i} is the \eqn{i}th estimated value, \eqn{Y_i} is the \eqn{i}th observed value, \eqn{\bar{Y}} is the arithmetic mean of observed values and \eqn{N} is the number of timepoints.
#' For the population-mean cosinor model, the test is calculated according to the procedure described in Bingham et al. (1982) as follows:
#' \deqn{F = \frac{k(k-2)}{2(k-1)}\frac{1}{1-(\frac{\widehat{\sigma}_{\beta\gamma}}{\widehat{\sigma}_\beta \widehat{\sigma}_\gamma})^2}[\frac{\beta^2}{\widehat{\sigma}^2_\beta}-2\frac{\widehat{\sigma}_{\beta\gamma}}{\widehat{\sigma}_\beta \widehat{\sigma}_\gamma}\frac{\beta \gamma}{\widehat{\sigma}_\beta \widehat{\sigma}_\gamma}+\frac{\gamma^2}{\widehat{\sigma}^2_\gamma}]} with \eqn{df_1 = 2} and \eqn{df_2 = k - 2}
#' , where \eqn{k} is the number of subjects in the population, \eqn{\widehat{\sigma}_\beta} and \eqn{\widehat{\sigma}_\gamma} are standard deviations of population \eqn{\beta} and \eqn{\gamma} coefficients and \eqn{\widehat{\sigma}_{\beta\gamma}} is the covariance of population \eqn{\beta} and \eqn{\gamma} coefficients.
#' @examples
#' fit.temperature<-cosinor.lm(Temperature~time(Time), period = 48, data = temperature_zg)
#' cosinor.detect(fit.temperature)
#'
#' fit.november<-population.cosinor.lm(timecol = 1, firstsubj = 2, lastsubj = 20, data =
#' PANAS_november, period = 7, na.action = "na.exclude")
#' cosinor.detect(fit.november)
#' @references Cornélissen, G. (2014). Cosinor-Based Rhythmometry. \emph{Theoretical Biology and Medical Modeling}, \emph{11}, Article 16.
#'
#' Bingham, C., Arbogast, B., Guillaume Cornélissen, G., Lee, J.K. & Halberg, F. (1982). Inferential Statistical Methods for Estimating and Comparing Cosinor Parameters. \emph{Chronobiologia}, \emph{9(4)}, 397-439.
#' @import cosinor ggplot2 matrixStats Hmisc
#' @importFrom graphics par
#' @importFrom stats cor cov lm pf sd
#' @export

cosinor.detect<-function(x){
  if(exists("pop.mat", where = x) == T){
    betas<-t(x$pop.mat[2,])
    gammas<-t(x$pop.mat[3,])
    beta<-mean(betas)
    gamma<-mean(gammas)
    sdb<-(sd(betas))
    sdy<-(sd(gammas))
    r<-cor(betas,gammas)
    k=ncol(x$pop.mat)
    frac1<-(k*(k-2))/(2*(k-1))
    frac2<-1/(1-r^2)
    frac3<-beta^2/sdb^2
    frac4<-(beta*gamma)/(sdb*sdy)
    frac5<-gamma^2/sdy^2
    brack<-frac3-(2*r*frac4)+frac5
    Fvalue<-frac1*frac2*brack
    df2<-k-2
    pvalue<-pf(q=Fvalue,df1=2,df2=df2,lower.tail=F)
    detection_test<-cbind(Fvalue,2,k-2,pvalue)
    colnames(detection_test)<-c("F","df1","df2","p")
    rownames(detection_test)<-NULL
  }
  else{
    RSS<-sum((x$fit$residuals)^2)
    MSS<-sum((x$fit$fitted.values-mean(x$fit$fitted.values+x$fit$residuals))^2)
    Fvalue<-(MSS/2)/(RSS/x$fit$df.residual)
    pvalue<-pf(q=Fvalue,df1=2,df2=x$fit$df.residual,lower.tail=F)
    detection_test<-cbind(Fvalue,2,x$fit$df.residual,pvalue)
    colnames(detection_test)<-c("F","df1","df2","p")
  }
  return(detection_test)
}

#' Population-Mean Cosinor
#'
#' @param firstsubj Column number of the first subject in the data frame.
#' @param lastsubj Column number of the last subject in the data frame.
#' @param timecol Column number of the column containing the time data in the data frame.
#' @param period Duration of one cycle of rhythm.
#' @param data data frame containing all the subjects and time data.
#' @param na.action Action to be performed on missing values.
#' @description Calculates the population-mean cosinor.
#' @details According to the procedure described in Cornélissen (2014), to calculate population-mean cosinor, single cosinors are first performed on each subject and linearized parameters are averaged, which allows for calculation of delinearized parameters.
#' @examples
#' population.cosinor.lm(timecol = 1, firstsubj = 2, lastsubj = 20, data = PANAS_november,
#' period = 7, na.action = "na.exclude")
#' @references Cornélissen, G. (2014). Cosinor-Based Rhythmometry. \emph{Theoretical Biology and Medical Modeling}, \emph{11}, Article 16.
#' @return A list containing following objects:
#'  \item{\code{single.cos}}{A list of objects containing all performed single cosinors.}
#'  \item{\code{pop.mat}}{A data frame containing the cosinor parameters of each subject in the population.}
#'  \item{\code{coefficients}}{Delinearized population-mean cosinor coefficients.}
#'  \item{\code{emp.mean}}{Empirical mean of the data across all timepoints.}
#'  \item{\code{fitted.values}}{Estimated values of the rhythm caclculated using the cosinor model.}
#'  \item{\code{residuals}}{The difference between empirical mean and the fitted values.}
#' @import cosinor ggplot2 matrixStats Hmisc
#' @importFrom graphics par
#' @importFrom stats cor cov lm pf sd
#' @export

population.cosinor.lm<-function(firstsubj,lastsubj,timecol,period,data,na.action){
  population.cosinor<-list()
  cosinors<-list()
  popmat<-data.frame(matrix(nrow=3,ncol=lastsubj-firstsubj+1))
  coefficients<-data.frame(matrix(nrow=1,ncol=3))
  colnames(coefficients)<-c("MESOR","Amplitude","Acrophase")
  subject<-integer()
  amplitude<-integer()
  acrophase<-integer()
  fitted.values<-vector()
  residuals<-vector()
  for (subject in firstsubj:lastsubj) {
    colnames(data)[subject]<-"Subjy"
    colnames(data)[subject-1]<-"done"
    colnames(data)[timecol]<-"Time"
    cosinors[[subject]]<-cosinor.lm(Subjy~time(Time),period=period,data=data,na.action=na.action)
    popmat[[subject]]<-cosinors[[subject]]$fit$coefficients
  }
  population.cosinor[[1]]<-cosinors
  population.cosinor[[2]]<-popmat
  names(population.cosinor)<-c("single.cos","pop.mat")
  population.cosinor$single.cos[[1]]<-NULL
  population.cosinor$pop.mat<-population.cosinor$pop.mat[,-1]
  coefs<-rowMeans(popmat, na.rm = T)
  MESOR<-coefs[[1]]
  beta<-coefs[[2]]
  gamma<-coefs[[3]]
  amplitude<-sqrt(beta^2+gamma^2)
  if(beta>0 & gamma>0){
    acrophase<-0+(-1*atan(abs(gamma/beta)))
  }
  else if(beta>0 & gamma<0){
    acrophase<-2*-1*pi+(1*atan(abs(gamma/beta)))
  }
  else if(beta<0 & gamma>0){
    acrophase<-pi*-1+(1*atan(abs(gamma/beta)))
  }
  else {
    acrophase<-pi*-1+(-1*atan(abs(gamma/beta)))
  }
  coefficients[,1]<-MESOR
  coefficients[,2]<-amplitude
  coefficients[,3]<-acrophase
  population.cosinor[[3]]<-coefficients
  names(population.cosinor)<-c("single.cos","pop.mat","coefficients")
  data2<-data[,-timecol]
  emp.mean<-rowMeans(data2, na.rm=T)
  for (n in 1:nrow(data)){
    fitted.values<-c(fitted.values,MESOR+(amplitude*cos(((2*pi*data[n,timecol])/period)+acrophase)))
  }
  residuals<-emp.mean-fitted.values
  population.cosinor[[4]]<-emp.mean
  population.cosinor[[5]]<-fitted.values
  population.cosinor[[6]]<-residuals
  names(population.cosinor)<-c("single.cos","pop.mat","coefficients","emp.mean","fitted.values","residuals")
  print(coefficients)
  return(population.cosinor)
}

#' Periodogram
#'
#' @param data data frame containing all the subjects and time data.
#' @param timecol Column number of the column containing the time data in the data frame.
#' @param firstsubj Column number of the first subject in the data frame.
#' @param lastsubj Column number of the last subject in the data frame. Same as \code{firstsubj} if performing single cosinor analysis.
#' @param na.action Action to be performed on missing values.
#' @description Estimates the best-fitting period using iterative cosinor procedure.
#' @details Iterative cosinor procedure is performed as described in Klemfuss & Clopton (1993). Cosinor is performed iteratively with the period (\eqn{\tau}) increased by 1 in each iteration. Percent Rhythm is calculated in each iteration, which allows for an estimation of the best fitting period. A periodogram can be plotted, which shows Percent Rhythm (coefficient of determination) for each period.
#' @note The range of periods included in iterations starts from 3 (sinusoidality of the curve is not achieved for \eqn{\tau} < 3) and ends with the number of timepoints in the data.
#' @seealso \code{\link{cosinor.PR}}
#' @examples
#' periodogram(timecol = 1, firstsubj = 2, lastsubj = 20, data = PANAS_november, na.action =
#' "na.exclude")
#'
#' periodogram(timecol = 1, firstsubj = 2, lastsubj = 2, data = temperature_zg)
#' @references Klemfuss, H. & Clopton, P. L. (1993). Seeking Tau: A Comparison of Six Methods. \emph{Journal of Interdisciplinary Cycle Research}, \emph{24(1)}, 1-16.
#' @import cosinor ggplot2 matrixStats Hmisc
#' @importFrom graphics par
#' @importFrom stats cor cov lm pf sd
#' @export

periodogram<-function(data, timecol, firstsubj, lastsubj, na.action){
  periodogram<-matrix()
  if (lastsubj - firstsubj == 0) {
    colnames(data)[timecol]<-"Time"
    colnames(data)[firstsubj]<-"Subjy"
    for (i in 3:nrow(data)) {
      cosinor<-cosinor.lm(Subjy~time(Time),data=data,na.action=na.action,period=i)
      periodogram[[i]]<-cosinor.PR(cosinor)[[2]]
    }
  }
  else {
    for (i in 3:nrow(data)){
      cosinor<-population.cosinor.lm(data = data, timecol = timecol, firstsubj = firstsubj, lastsubj = lastsubj, na.action = na.action, period = i)
      periodogram[[i]]<-cosinor.PR(cosinor)[[2]]
    }
  }
  periods<-as.numeric(periodogram)
  periodogram<-data.frame(periodogram)
  rows<-(nrow(periodogram))
  plot<-ggplot(periodogram,aes(x=1:rows))+
    geom_point(aes(y=periodogram)) +
    geom_line(aes(y=periodogram)) +
    labs(x = "Period", y = "Coefficient of determination")
  best<-(which(periods == max(periods,na.rm=T)))
  print(paste("The best fitting period is",best))
  return(plot)
}

#' Comparison of Cosinor Parameters of Two Populations
#'
#' @param pop1 An object created by \code{population.cosinor.lm} executed on the first population.
#' @param pop2 An object created by \code{population.cosinor.lm} executed on the second population.
#' @description Runs the tests that compare MESORs, amplitudes and acrophases of two different populations.
#' @details Bingham et al. (1982) describe tests for comparing population MESORs, amplitudes and acrophases. These tests are esentially F-ratios with \eqn{df_1 = m - 1} and \eqn{df_2 = K - m}, where \eqn{m} is the number of populations and \eqn{K} is the total number of subjects. The tests for MESOR, amplitude and acrophase differences respectively are calculated as follows:
#' \deqn{F_M = \frac{\sum_{j = 1}^{m}k_j(\widehat{M}_j - \widehat{M})^2}{(m-1)\widehat{\sigma}_M^2}}
#' \deqn{F_\phi = \frac{\frac{\sum_{j = 1}^{m}k_j A_j^2 * sin^2(\widehat{\phi}_j - \tilde{\phi})}{m - 1}} {\widehat{\sigma}_\beta^2 sin^2\tilde{\phi} + 2\widehat{\sigma}_{\beta \gamma} cos\tilde{\phi}sin\tilde{\phi} + \widehat{\sigma}_\gamma^2 cos^2\tilde{\phi}}}
#' \deqn{F_A = \frac{\frac{\sum_{j = 1}^{m}(\widehat{A}_j - \widehat{A})^2}{m - 1}}{\widehat{\sigma}^2_\beta cos^2\widehat{\phi} - 2\widehat{\sigma}_{\beta \gamma}cos\widehat{\phi}sin\widehat{\phi} + \widehat{\sigma}^2_\gamma sin^2 \widehat{\phi}}}
#' where \eqn{\widehat{M}}, \eqn{\widehat{A}} and \eqn{\widehat{\phi}} are weighted averages of parameters across populations calculated as:
#' \deqn{\widehat{M} = \frac{\sum_{j = 1}^{m}k_j\widehat{M}_j}{K}}
#' \deqn{\widehat{A} = \frac{\sum_{j = 1}^{m}k_j\widehat{A}_j}{K}}
#' \deqn{\widehat{\phi} = \frac{\sum_{j = 1}^{m}k_j\widehat{\phi}_j}{K}}
#' \eqn{\tilde{\phi}} is derived from the following expression:
#' \deqn{tan 2\tilde{\phi} = \frac{\sum_{j = 1}^{m}k_j\widehat{A}^2_j sin 2\widehat{\phi}_j}{\sum_{j = 1}^{m}k_j\widehat{A}^2_j cos 2\widehat{\phi}_j}}
#' where \eqn{2\tilde{\phi}} lies between \eqn{-\frac{\pi}{2}} and \eqn{\frac{\pi}{2}} if the denomanator is positive or between \eqn{\frac{\pi}{2}} and \eqn{\frac{3\pi}{2}} if the denominator is negative, \eqn{k_j} is the number of subjects in the \eqn{j}th population, \eqn{\widehat{M}_j}, \eqn{\widehat{A}_j} and \eqn{\widehat{\phi}_j} are the cosinor parameters of the \eqn{j}th population and \eqn{\widehat{\sigma}_\beta},\eqn{\widehat{\sigma}_\gamma} and \eqn{\widehat{\sigma}_{\beta \gamma}} are the estimates of pooled standard deviations (and covariance) calculated as following:
#' \deqn{\widehat{\sigma}_\beta = \frac{\sum_{j = 1}^{m} (k_j - 1)\widehat{\sigma}_{\beta_j}}{K - m}}
#' \deqn{\widehat{\sigma}_\gamma = \frac{\sum_{j = 1}^{m} (k_j - 1)\widehat{\sigma}_{\gamma_j}}{K - m}}
#' \deqn{\widehat{\sigma}_{\beta \gamma} = \frac{\sum_{j = 1}^{m} (k_j - 1)\widehat{\sigma}_{{\beta_j} {\gamma_j}}}{K - m}}
#' where \eqn{\widehat{\sigma}_{\beta_j}}, \eqn{\widehat{\sigma}_{\gamma_j}} and \eqn{\widehat{\sigma}_{{\beta_j} {\gamma_j}}} are the standard devations and covariance of \eqn{\beta} and \eqn{\gamma} parameters in the \eqn{j}th population.
#' @note These tests should only be performed on independent samples. If the acrophases of two populations are significantly different, the results of the amplitude difference test are not reliable and should not be interpreted. While it's possible to perform tests which compare more than two populations, this function can only compare two populations.
#' @examples
#' fit.extraverts<-population.cosinor.lm(timecol = 1, firstsubj = 2, lastsubj = 25, data =
#' PA_extraverts, period = 12)
#' fit.introverts<-population.cosinor.lm(timecol = 1, firstsubj = 2, lastsubj = 30, data =
#' PA_introverts, period = 12)
#' cosinor.poptests(pop1 = fit.extraverts, pop2 = fit.introverts)
#' @references Bingham, C., Arbogast, B., Guillaume Cornélissen, G., Lee, J.K. & Halberg, F. (1982). Inferential Statistical Methods for Estimating and Comparing Cosinor Parameters. \emph{Chronobiologia}, \emph{9(4)}, 397-439.
#' @import cosinor ggplot2 matrixStats Hmisc
#' @importFrom graphics par
#' @importFrom stats cor cov lm pf sd
#' @export

cosinor.poptests<-function(pop1,pop2){
  mesors1<-list()
  mesors2<-list()
  amplitudes1<-list()
  amplitudes2<-list()
  acrophases1<-list()
  acrophases2<-list()
  betas1<-list()
  betas2<-list()
  gammas1<-list()
  gammas2<-list()
  k1<-length(pop1$single.cos)
  k2<-length(pop2$single.cos)
  K<-(k1+k2)
  for (i in 1:k1){
    mesors1[[i]]<-pop1$single.cos[[i]]$coefficients[[1]]
    amplitudes2[[i]]<-pop1$single.cos[[i]]$coefficients[[2]]
    betas1[[i]]<-pop1$single.cos[[i]]$fit$coefficients[[2]]
    gammas1[[i]]<-pop1$single.cos[[i]]$fit$coefficients[[3]]
    acrophases1[[i]]<-correct.acrophase(pop1$single.cos[[i]])
  }
  for (i in 1:k2){
    mesors2[[i]]<-pop2$single.cos[[i]]$coefficients[[1]]
    amplitudes2[[i]]<-pop2$single.cos[[i]]$coefficients[[2]]
    betas2[[i]]<-pop2$single.cos[[i]]$fit$coefficients[[2]]
    gammas2[[i]]<-pop2$single.cos[[i]]$fit$coefficients[[3]]
    acrophases2[[i]]<-correct.acrophase(pop2$single.cos[[i]])
  }
  mesors1<-matrix(mesors1)
  mesors2<-matrix(mesors2)
  betas1<-matrix(betas1)
  betas2<-matrix(betas2)
  gammas1<-matrix(gammas1)
  gammas2<-matrix(gammas2)
  amplitudes1<-matrix(amplitudes1)
  amplitudes2<-matrix(amplitudes2)
  acrophases1<-matrix(acrophases1)
  acrophases2<-matrix(acrophases2)
  M = ((k1*pop1$coefficients[[1]])+(k2*pop2$coefficients[[1]]))/K
  A = ((k1*pop1$coefficients[[2]])+(k2*pop2$coefficients[[2]]))/K
  FI = ((k1*pop1$coefficients[[3]])+(k2*pop2$coefficients[[3]]))/K
  BETA = ((k1*(pop1$coefficients[[2]]*cos(pop1$coefficients[[3]])))+(k2*(pop2$coefficients[[2]]*cos(pop2$coefficients[[3]]))))/K
  GAMMA = ((k1*(-1*pop1$coefficients[[2]]*sin(pop1$coefficients[[3]])))+(k2*(-1*pop2$coefficients[[2]]*sin(pop2$coefficients[[3]]))))/K
  TM = (k1*((pop1$coefficients[[1]]-M)^2)) + (k2*((pop2$coefficients[[1]]-M)^2))
  tann<-((k1*(pop1$coefficients[[2]]^2))*sin(2*pop1$coefficients[[3]]))+((k2*(pop2$coefficients[[2]]^2))*sin(2*pop2$coefficients[[3]]))
  tand<-((k1*(pop1$coefficients[[2]]^2))*cos(2*pop1$coefficients[[3]]))+((k2*(pop2$coefficients[[2]]^2))*cos(2*pop2$coefficients[[3]]))
  if (tand > 0) {
    twofi = atan(tann/tand)
  }
  else {
    twofi = atan(tann/tand) + pi
  }
  FITILDE = twofi/2
  varm1<-rowVars(as.numeric(mesors1), dim. = c(1,k1))
  varm2<-rowVars(as.numeric(mesors2), dim. = c(1,k2))
  varb1<-rowVars(as.numeric(betas1), dim. = c(1,k1))
  varb2<-rowVars(as.numeric(betas2), dim. = c(1,k2))
  vary1<-rowVars(as.numeric(gammas1), dim. = c(1,k1))
  vary2<-rowVars(as.numeric(gammas2), dim. = c(1,k2))
  covby1<-cov(as.numeric(betas1),as.numeric(gammas1))
  covby2<-cov(as.numeric(betas2),as.numeric(gammas2))
  varm<-(((k1-1)*varm1)/(K-2))+(((k2-1)*varm2)/(K-2))
  varb<-(((k1-1)*varb1)/(K-2))+(((k2-1)*varb2)/(K-2))
  vary<-(((k1-1)*vary1)/(K-2))+(((k2-1)*vary2)/(K-2))
  covby<-(((k1-1)*covby1)/(K-2))+(((k2-1)*covby2)/(K-2))
  FM = TM/varm
  acrn<-(pop1$coefficients[[2]]^2+((sin(pop1$coefficients[[3]]-FITILDE))^2)) + (pop2$coefficients[[2]]^2+((sin(pop2$coefficients[[3]]-FITILDE))^2))
  acrd1<-varb*((sin(FITILDE))^2)
  acrd2<-2*covby*cos(FITILDE)*sin(FITILDE)
  acrd3<-vary*((cos(FITILDE))^2)
  acrd<-acrd1-acrd2+acrd3
  FFI<-acrn/acrd
  ampn<-((pop1$coefficients[[2]]-A)^2) + ((pop2$coefficients[[2]]-A)^2)
  ampd1<-varb*((cos(FI))^2)
  ampd2<-2*covby*cos(FI)*sin(FI)
  ampd3<-vary*((sin(FI))^2)
  ampd<-ampd1-ampd2+ampd3
  FA<-ampn/ampd
  df1=1
  df2=K-2
  PM<-pf(q=FM,df1=df1,df2=df2,lower.tail=F)
  PA<-pf(q=FA,df1=df1,df2=df2,lower.tail=F)
  PFI<-pf(q=FFI,df1=df1,df2=df2,lower.tail=F)
  resm<-cbind(FM,df1,df2,PM,pop1$coefficients[[1]],pop2$coefficients[[1]])
  resa<-cbind(FA,df1,df2,PA,pop1$coefficients[[2]],pop2$coefficients[[2]])
  resfi<-cbind(FFI,df1,df2,PFI,pop1$coefficients[[3]],pop2$coefficients[[3]])
  results<-rbind(resm,resa,resfi)
  colnames(results)<-c("F","df1","df2","p", "1st population", "2nd population")
  row.names(results)<-c("MESOR", "Amplitude", "Acrophase")
  if (PFI < 0.05) {
    warning("Results of population amplitude difference test are not reliable due to different acrophases.")
  }
  return(results)
}

#' Serial Sections
#'
#' @param data data frame containing all the subjects and time data.
#' @param timecol Column number of the column containing the time data in the data frame.
#' @param firstsubj Column number of the first subject in the data frame.
#' @param lastsubj Column number of the last subject in the data frame. Same as \code{firstsubj} if performing single cosinor analysis.
#' @param period Duration of one cycle of rhythm.
#' @param na.action Action to be performed on missing values.
#' @param interval Length of an interval (number of timepoints) on which cosinor analyses will be ran.
#' @param increment A number indicating for how much timepoints should the interval be displaced throughout the data. Note that the value of the increment cannot be higher than the value of the interval.
#' @description Performs serial section analysis of rhythmic data and fits non-stationary cosinor models.
#' @details Cornélissen (2014) describes procedures for rhythmometric analysis of non-stationary data. First, an interval of an user-specified length (\eqn{I}) is chosen and usual cosinor analysis (i.e. single cosinor or population-mean cosinor) is performed on the interval. The interval is then displaced throughout the data by an user-specified increment (\eqn{\Delta t}) and cosinor analysis is then performed on the new interval. Intervals can be overlapping (\eqn{\Delta t}<I) or non-overlapping (\eqn{\Delta t}=I). A rhythm detection test is also calculated in each interval. After values of cosinor parameters have been obtained for each interval, they can be plotted, along with the \emph{p}-values from the rhythm detection test. Trends of growth of the MESOR, amplitude and the acrophase can be estimated using linear regression. Estimated trends of the model can then be imputed into the cosinor equation to fit the non-stationary model to data.
#' @note The value of increment cannot be higher than the value of the interval. While it's possible to estimate the trends of cosinor parameters using nonlinear regression, this function only includes estimates obtained using linear regression.
#' @examples
#' ssections(timecol = 1, firstsubj = 2, lastsubj = 20, data = PANAS_november, period = 7,
#' interval = 14, increment = 1, na.action = "na.exclude")
#' @references Cornélissen, G. (2014). Cosinor-Based Rhythmometry. \emph{Theoretical Biology and Medical Modeling}, \emph{11}, Article 16.
#' @return A list containing following objects:
#'  \item{\code{coefficients}}{Cosinor coefficients in each of the intervals.}
#'  \item{\code{lm.coeffs}}{Linear regression coefficients for MESOR, amplitude and acrophase.}
#'  \item{\code{lm.fitted}}{Values of MESOR, amplitude and acrophase estimated by linear regression.}
#'  \item{\code{fitted.values}}{Estimated values of the non-stationary cosinor model.}
#'  \item{\code{emp.mean}}{Empirical mean of the data across all timepoints.}
#'  \item{\code{p-values}}{\emph{p}-values from the rhythm detection test in each interval.}
#'  \item{\code{cosinors}}{A list containing all cosinor objects calculated in each interval.}
#'  \item{\code{plots}}{A list containing following plots: \itemize{\item{\code{Chronogram}}{ A graphical representation of the observed and fitted values. Percent Rhythm is shown in the caption.} \item{\code{MESOR}}{ A graphical representation of the trend of the MESOR across intervals.} \item{\code{Amplitude}}{ A graphical representation of the trend of the amplitude across intervals.} \item{\code{Acrophase}}{ A graphical representation of the trend of the acrophase across intervals.} \item{\code{p-values}}{ A graphical representation of the trend of the \emph{p}-values across intervals. The usual .05 cutoff is shown.}}}
#' @import cosinor ggplot2 matrixStats Hmisc
#' @importFrom graphics par
#' @importFrom stats cor cov lm pf sd
#' @export

ssections<-function(data,timecol,firstsubj,lastsubj,period,na.action,interval,increment){
  i<-1
  j<-1
  q<-0
  cosinors<-list()
  coefficients<-data.frame(matrix(nrow=3,ncol=1))
  lmcoeffs<-data.frame(matrix(nrow=2,ncol=1))
  pvalues<-list()
  cor.acrs<-list()
  acro4<-list()
  acro5<-list()
  acro6<-list()
  interm.coefs<-list()
  sections<-list()
  fitted.values<-list()
  emp.mean<-list()
  plots<-list()
  while (j <= nrow(data)){
    j=i+interval-1
    if(length(i:j) == interval & j <= nrow(data)){
      q=q+1
    }
    i=i+increment
  }
  lmfitted<-data.frame(matrix(nrow=q,ncol=1))
  if (increment>interval){
    stop("Value of increment cannot be higher than the value of interval.")
  }
  if (lastsubj - firstsubj == 0){
    colnames(data)[timecol]<-"Time"
    colnames(data)[firstsubj]<-"Subjy"
    for (k in c(1:q)){
      cosinors[[k]]<-cosinor.lm(Subjy~time(Time),period=period,na.action=na.action,data=data[c(k:(k+interval-1)),])
      pvalues[[k]]<-cosinor.detect(cosinors[[k]])[[4]]
      coefficients[[k]]<-cosinors[[k]]$coefficients
      cor.acrs[[k]]<-correct.acrophase(cosinors[[k]])
      k=k+increment
    }
    coefficients<-data.frame(t(coefficients))
  }
  else {
    for (k in c(1:q)){
      cosinors[[k]]<-population.cosinor.lm(firstsubj = firstsubj, lastsubj = lastsubj,timecol = timecol,period=period,na.action=na.action,data=data[c(k:(k+interval-1)),])
      pvalues[[k]]<-cosinor.detect(cosinors[[k]])[[4]]
      coefficients[[k]]<-t(cosinors[[k]]$coefficients)
      cor.acrs[[k]]<-cosinors[[k]]$coefficients$Acrophase
      k=k+increment
    }
  }
  acro2<-as.numeric(cor.acrs)-2*pi
  acro3<-as.numeric(cor.acrs)+2*pi
  cor.acrs<-as.numeric(cor.acrs)
  if (lastsubj-firstsubj != 0){
    coefficients<-t(coefficients)
  }
  coefficients<-cbind(coefficients,acro2,cor.acrs,acro3)
  coefficients<-data.frame(coefficients)
  colnames(coefficients)<-c("MESOR","Amplitude","Acrophase","acro2","cor.acrs","acro3")
  for (l in c(1:nrow(coefficients))){
    if (coefficients[l,4] >= -2*pi & coefficients[l,4] <= 0)
    {
      coefficients[l,3]<-coefficients[l,4]
    }
    else if (coefficients[l,5] >= -2*pi & coefficients[l,5] <= 0)
    {
      coefficients[l,3]<-coefficients[l,5]
    }
    else if (coefficients[l,6] >= -2*pi & coefficients[l,6] <= 0)
    {
      coefficients[l,3]<-coefficients[l,6]
    }
  }
  cor.acrs<-data.frame(t(cor.acrs))
  acro4[[1]]<-cor.acrs[[1]]
  for (m in c(2:nrow(coefficients))){
    if(abs(as.numeric(cor.acrs[[m]])-as.numeric(acro4[[(m-1)]]) < 1*pi)){
      acro4[[m]]<-cor.acrs[[m]]
    }
    else if(abs(as.numeric(acro2[[m]])-as.numeric(acro4[[(m-1)]]) < 1*pi)){
      acro4[[m]]<-acro2[[m]]
    }
    else {
      acro4[[m]]<-acro3[[m]]
    }
  }
  acro4<-data.frame(acro4)
  acro4<-t(acro4)
  acro5<-acro4+(2*pi)
  acro6<-acro4-(2*pi)
  coefficients<-cbind(coefficients,acro4,acro5,acro6)
  colnames(coefficients)<-c("MESOR","Amplitude","Acrophase","acro2","cor.acrs","acro3","acro4","acro5","acro6")
  for (n in c(1:nrow(coefficients))){
    if (coefficients[n,7] >= -2*pi & coefficients[n,7] <= 0)
    {
      coefficients[n,3]<-coefficients[n,7]
    }
    else if (coefficients[n,8] >= -2*pi & coefficients[n,8] <= 0)
    {
      coefficients[n,3]<-coefficients[n,8]
    }
    else if (coefficients[n,9] >= -2*pi & coefficients[n,9] <= 0)
    {
      coefficients[n,3]<-coefficients[n,9]
    }
  }
  acrophases<-data.frame(coefficients[,c(7,8,9)])
  coefficients<-data.frame(coefficients[,c(-4,-5,-6,-7,-8,-9)])
  row.names(coefficients)<-c(1:nrow(coefficients))
  lmcoeffs[[1]]<-lm(coefficients$MESOR~c(1:nrow(coefficients)))$coefficients
  lmcoeffs[[2]]<-lm(coefficients$Amplitude~c(1:nrow(coefficients)))$coefficients
  lmcoeffs[[3]]<-lm(acrophases[,1]~c(1:nrow(coefficients)))$coefficients
  lmcoeffs[[4]]<-lm(acrophases[,2]~c(1:nrow(coefficients)))$coefficients
  lmcoeffs[[5]]<-lm(acrophases[,3]~c(1:nrow(coefficients)))$coefficients
  lmfitted[[1]]<-lm(coefficients$MESOR~c(1:nrow(coefficients)))$fitted.values
  lmfitted[[2]]<-lm(coefficients$Amplitude~c(1:nrow(coefficients)))$fitted.values
  lmfitted[[3]]<-lm(acrophases[,1]~c(1:nrow(coefficients)))$fitted.values
  lmfitted[[4]]<-lm(acrophases[,2]~c(1:nrow(coefficients)))$fitted.values
  lmfitted[[5]]<-lm(acrophases[,3]~c(1:nrow(coefficients)))$fitted.values
  for (o in c(1:nrow(lmfitted))){
    if (lmfitted[o,3] >= -2*pi & lmfitted[o,3] <= 0)
    {
      lmfitted[l,3]<-lmfitted[l,3]
    }
    else if (lmfitted[o,4] >= -2*pi & lmfitted[o,4] <= 0)
    {
      lmfitted[o,3]<-lmfitted[o,4]
    }
    else if (lmfitted[o,5] >= -2*pi & lmfitted[o,5] <= 0)
    {
      lmfitted[o,3]<-lmfitted[o,5]
    }
  }
  acrolms<-data.frame(lmfitted[,c(3,4,5)])
  lmfitted<-data.frame(lmfitted[,c(-4,-5)])
  colnames(lmfitted)<-c("MESOR","Amplitude","Acrophase")
  if (lmcoeffs[[3]][[1]] >= 2*pi & lmcoeffs[[3]][[1]] <=0){
    lmcoeffs[[3]]<-lmcoeffs[[3]]
  }
  else if (lmcoeffs[[4]][[1]] >= 2*pi & lmcoeffs[[4]][[1]] <=0){
    lmcoeffs[[3]]<-lmcoeffs[[4]]
  }
  else if (lmcoeffs[[5]][[1]] >= 2*pi & lmcoeffs[[5]][[1]] <=0){
    lmcoeffs[[3]]<-lmcoeffs[[5]]
  }
  lmcoeffs<-data.frame(lmcoeffs[,c(-4,-5)])
  sections[[1]]<-coefficients
  sections[[2]]<-lmcoeffs
  sections[[3]]<-lmfitted
  names(sections)<-c("coefficients","lm.coeffs","lm.fitted")
  colnames(sections$lm.coeffs)<-c("MESOR","Amplitude","Acrophase")
  row.names(sections$lm.coeffs)<-c("b","a")
  for (p in 1:nrow(data)){
    fitted.values<-c(fitted.values,((sections$lm.coeffs$MESOR[[2]]*p)+sections$lm.coeffs$MESOR[[1]])+(((sections$lm.coeffs$Amplitude[[2]]*p)+sections$lm.coeffs$Amplitude[[1]])*cos(((2*pi*data[p,timecol])/period)+((sections$lm.coeffs$Acrophase[[2]]*p)+sections$lm.coeffs$Acrophase[[1]]))))
  }
  sections[[4]]<-data.frame(t(data.frame(fitted.values)))
  names(sections)<-c("coefficients","lm.coeffs","lm.fitted","fitted.values")
  data2<-data.frame(data[,-timecol])
  emp.mean<-rowMeans(data2, na.rm = T)
  sections[[5]]<-data.frame(emp.mean)
  sections[[6]]<-data.frame(pvalues)
  sections[[7]]<-cosinors
  names(sections)<-c("coefficients","lm.coeffs","lm.fitted","fitted.values","emp.mean","p-values", "cosinors")
  PR<-cor(sections[[4]],sections[[5]])^2
  plots[[1]]<-ggplot(data.frame(emp.mean),aes(x=c(1:nrow(data.frame(emp.mean)))))+
    geom_line(aes(y=t(data.frame(fitted.values)), linetype = "Estimated"), size = 1) +
    geom_line(aes(y=data.frame(emp.mean), linetype = "Observed"), size = 1.3)+
    labs(x = "Time", y = "Value", title = "Chronogram", caption = paste("PR = ",PR))+
    scale_linetype_manual("",values=c("Estimated"=2,"Observed"=1))
  plots[[2]]<-ggplot(coefficients,aes(x=c(1:nrow(coefficients))))+
    geom_line(aes(y=lmfitted[,1], linetype = "Estimated"), size = 1) +
    geom_line(aes(y=coefficients[,1], linetype = "Observed"), size = 1.3)+
    labs(x = "Section", y = "Value", title = "MESOR")+
    scale_linetype_manual("",values=c("Estimated"=2,"Observed"=1))
  plots[[3]]<-ggplot(coefficients,aes(x=c(1:nrow(coefficients))))+
    geom_line(aes(y=lmfitted[,2], linetype = "Estimated"), size = 1) +
    geom_line(aes(y=coefficients[,2], linetype = "Observed"), size = 1.3)+
    labs(x = "Section", y = "Value", title = "Amplitude")+
    scale_linetype_manual("",values=c("Estimated"=2,"Observed"=1))
  plots[[4]]<-ggplot(acrophases,aes(x=c(1:nrow(acrophases))))+
    geom_line(aes(y=acrolms[,1], linetype = "Estimated"), size = 1) +
    geom_line(aes(y=acrolms[,2], linetype = "Estimated"), size = 1) +
    geom_line(aes(y=acrolms[,3], linetype = "Estimated"), size = 1) +
    geom_line(aes(y=acrophases[,1], linetype = "Observed"), size = 1.3)+
    geom_line(aes(y=acrophases[,2], linetype = "Observed"), size = 1.3)+
    geom_line(aes(y=acrophases[,3], linetype = "Observed"), size = 1.3)+
    labs(x = "Section", y = "Value", title = "Acrophase")+
    scale_linetype_manual("",values=c("Estimated"=2,"Observed"=1))+
    ylim(-2*pi,0)
  pvalues<-data.frame(t(data.frame(pvalues)))
  plots[[5]]<-ggplot(pvalues,aes(x=c(1:q)))+
    geom_line(aes(y=pvalues, linetype = "Observed"), size = 1.3)+
    geom_point(aes(y=pvalues), size = 1.3)+
    geom_line(aes(y=0.05, linetype = "Cutoff"), size = 0.7)+
    labs(x = "Section", y = "Value", title = "p-values")+
    scale_linetype_manual("",values=c("Cutoff"=2,"Observed"=1)) +
    theme(legend.position="none")
  sections[[8]]<-plots
  names(sections)<-c("coefficients","lm.coeffs","lm.fitted","fitted.values","emp.mean","p-values", "cosinors", "plots")
  par(ask=T)
  return(sections)
}

#' Daily air temperature in Zagreb
#'
#' A dataset containing the air temperature in Zagreb on July 1, 2015.
#' @details Measurements were taken every 30 minutes throughout the whole day.
#'
#' @format A data frame with 48 rows and 2 variables:
#' \describe{
#'   \item{Time}{Time of measurement in hours.}
#'   \item{Temperature}{Air temperature in degrees Celsius.}
#'   }
#' @source Croatian Meteorological and Hydrological Service, \url{http://www.meteo.hr}
"temperature_zg"

#' Self-reported mood
#'
#' A dataset containing the responses of 19 subjects on the shortened version of the PANAS questionnaire (Watson, Clark & Tellegen, 1988) in November 2015.
#' @details Measurements were taken every day after 8 PM.
#'
#' @format A data frame with 30 rows and 20 variables:
#' \describe{
#'   \item{Time}{Day of measurement.}
#'   \item{Subject 1, Subject 2, Subject 3, Subject 4, Subject 5, Subject 6, Subject 7, Subject 8, Subject 9, Subject 10, Subject 11, Subject 12, Subject 13, Subject 14, Subject 15, Subject 16, Subject 17, Subject 18, Subject 19}{Results of 19 subjects on the PANAS questionnaire.}
#'   }
#' @source Mutak, A. (2016, May). \emph{Kako struktura tjedna utječe na naše raspoloženje?} [\emph{How Does the Structure of the Week Impact Our Mood?}]. Study presented at the 5th \emph{Psihozij} conference, Zagreb, Croatia.
#' @note The data contained in this dataset has been reduced compared to the original data that included more subjects. This dataset contains only the subjects that have responded to the PANAS questionnaire on more than 85\% of the timepoints in both of the research cycles (July and November).
#' @references Watson, D., Clark, L. A. & Tellegen, A. (1988). Development and Validation of Brief Measures of Positive and Negative Affect: The PANAS Scales. \emph{Journal of Personality and Social Psychology}, \emph{54(6)}, 1063-1070.
"PANAS_november"

#' Self-reported mood of extraverts
#'
#' A dataset containing the responses of 24 subjects on the Positive Affect scale of the shortened version of the PANAS questionnaire (Watson, Clark & Tellegen, 1988) in January 2017.
#' @details Measurements were taken at 10 AM, 12 PM, 2 PM, 4 PM, 6 PM and 8 PM \eqn{\pm} 30 minutes in the period of January 16 - 22, 2017. The data contained in this dataset has been averaged for each hour across 7 days of measurement.
#'
#' @format A data frame with 6 rows and 25 variables:
#' \describe{
#'   \item{Time}{Time of measurement in hours.}
#'   \item{Subject 1, Subject 2, Subject 3, Subject 4, Subject 5, Subject 6, Subject 7, Subject 8, Subject 9, Subject 10, Subject 11, Subject 12, Subject 13, Subject 14, Subject 15, Subject 16, Subject 17, Subject 18, Subject 19, Subject 20, Subject 21, Subject 22, Subject 23, Subject 24}{Results of 24 subjects on the PA scale of the PANAS questionnaire.}
#'   }
#' @source Mutak, A., Pavlović, M. & Zibar, K. (2017, May). \emph{Postoje li razlike između introverata i ekstraverata u cirkadijurnim ritmovima raspoloženja?} [\emph{Are There Differences Between Introverts and Extraverts in Circadian Mood Rhythms?}]. Study presented at the 3rd \emph{Regionalni susret studenata psihologije - SPIRI} [\emph{Regional Meeting of Psychology Students - SPIRI}] conference, Rijeka, Croatia.
#' @references Watson, D., Clark, L. A. & Tellegen, A. (1988). Development and Validation of Brief Measures of Positive and Negative Affect: The PANAS Scales. \emph{Journal of Personality and Social Psychology}, \emph{54(6)}, 1063-1070.
"PA_extraverts"

#' Self-reported mood of introverts
#'
#' A dataset containing the responses of 29 subjects on the Positive Affect scale of the shortened version of the PANAS questionnaire (Watson, Clark & Tellegen, 1988) in January 2017.
#' @details Measurements were taken at 10 AM, 12 PM, 2 PM, 4 PM, 6 PM and 8 PM \eqn{\pm} 30 minutes in the period of January 16 - 22, 2017. The data contained in this dataset has been averaged for each hour across 7 days of measurement.
#'
#' @format A data frame with 6 rows and 30 variables:
#' \describe{
#'   \item{Time}{Time of measurement in hours.}
#'   \item{Subject 1, Subject 2, Subject 3, Subject 4, Subject 5, Subject 6, Subject 7, Subject 8, Subject 9, Subject 10, Subject 11, Subject 12, Subject 13, Subject 14, Subject 15, Subject 16, Subject 17, Subject 18, Subject 19, Subject 20, Subject 21, Subject 22, Subject 23, Subject 24, Subject 25, Subject 26, Subject 27, Subject 28, Subject 29}{Results of 29 subjects on the PA scale of the PANAS questionnaire.}
#'   }
#' @source Mutak, A., Pavlović, M. & Zibar, K. (2017, May). \emph{Postoje li razlike između introverata i ekstraverata u cirkadijurnim ritmovima raspoloženja?} [\emph{Are There Differences Between Introverts and Extraverts in Circadian Mood Rhythms?}]. Study presented at the 3rd \emph{Regionalni susret studenata psihologije - SPIRI} [\emph{Regional Meeting of Psychology Students - SPIRI}] conference, Rijeka, Croatia.
#' @references Watson, D., Clark, L. A. & Tellegen, A. (1988). Development and Validation of Brief Measures of Positive and Negative Affect: The PANAS Scales. \emph{Journal of Personality and Social Psychology}, \emph{54(6)}, 1063-1070.
"PA_introverts"
