% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evaluate.R
\name{evaluate}
\alias{evaluate}
\title{Evaluate your model's performance}
\usage{
evaluate(
  data,
  target_col,
  prediction_cols,
  type = "gaussian",
  id_col = NULL,
  id_method = "mean",
  models = NULL,
  apply_softmax = TRUE,
  cutoff = 0.5,
  positive = 2,
  metrics = list(),
  include_predictions = TRUE,
  parallel = FALSE
)
}
\arguments{
\item{data}{Data frame with predictions, targets and (optionally) an ID column.
 Can be grouped with \code{\link[dplyr]{group_by}}.

 \subsection{Multinomial}{
 When \code{type} is \code{"multinomial"}, the predictions should be passed as
 one column per class with the probability of that class. The columns should
 have the name of their class, as they are named in the target column. E.g.:

 \tabular{rrrrr}{
  \strong{class_1} \tab \strong{class_2} \tab
  \strong{class_3} \tab \strong{target}\cr
  0.269 \tab 0.528 \tab 0.203 \tab class_2\cr
  0.368 \tab 0.322 \tab 0.310 \tab class_3\cr
  0.375 \tab 0.371 \tab 0.254 \tab class_2\cr
  ... \tab ... \tab ... \tab ...}
 }
 \subsection{Binomial}{
 When \code{type} is \code{"binomial"}, the predictions should be passed as
 one column with the probability of class being
 the second class alphabetically
 (1 if classes are 0 and 1). E.g.:

 \tabular{rrrrr}{
  \strong{prediction} \tab \strong{target}\cr
  0.769 \tab 1\cr
  0.368 \tab 1\cr
  0.375 \tab 0\cr
  ... \tab ...}
 }
 \subsection{Gaussian}{
 When \code{type} is \code{"gaussian"}, the predictions should be passed as
 one column with the predicted values. E.g.:

 \tabular{rrrrr}{
  \strong{prediction} \tab \strong{target}\cr
  28.9 \tab 30.2\cr
  33.2 \tab 27.1\cr
  23.4 \tab 21.3\cr
  ... \tab ...}
 }}

\item{target_col}{Name of the column with the true classes/values in \code{data}.

 When \code{type} is \code{"multinomial"}, this column should contain the class names,
 not their indices.}

\item{prediction_cols}{Name(s) of column(s) with the predictions.

 When evaluating a classification task,
 the column(s) should contain the predicted probabilities.}

\item{type}{Type of evaluation to perform:

 \code{"gaussian"} for regression (like linear regression).

 \code{"binomial"} for binary classification.

 \code{"multinomial"} for multiclass classification.}

\item{id_col}{Name of ID column to aggregate predictions by.

 N.B. Current methods assume that the target class/value is constant within the IDs.

 N.B. When aggregating by ID, some metrics (such as those from model objects) are excluded.}

\item{id_method}{Method to use when aggregating predictions by ID. Either \code{"mean"} or \code{"majority"}.

 When \code{type} is \code{gaussian}, only the \code{"mean"} method is available.

 \subsection{mean}{
 The average prediction (value or probability) is calculated per ID and evaluated.
 This method assumes that the target class/value is constant within the IDs.
 }
 \subsection{majority}{
 The most predicted class per ID is found and evaluated. In case of a tie,
 the winning classes share the probability (e.g. \code{P = 0.5} each when two majority classes).
 This method assumes that the target class/value is constant within the IDs.
 }}

\item{models}{Unnamed list of fitted model(s) for calculating R^2 metrics and information criterion metrics.
 May only work for some types of models.

 When only passing one model, remember to pass it in a list (e.g. \code{list(m)}).

 N.B. When \code{data} is grouped, provide one model per group in the same order as the groups.

 N.B. When aggregating by ID (i.e. when \code{id_col} is not \code{NULL}),
 it's not currently possible to pass model objects,
 as these would not be aggregated by the IDs.

 N.B. Currently, \strong{Gaussian only}.}

\item{apply_softmax}{Whether to apply the softmax function to the
 prediction columns when \code{type} is \code{"multinomial"}.

 N.B. \strong{Multinomial models only}.}

\item{cutoff}{Threshold for predicted classes. (Numeric)

N.B. \strong{Binomial models only}.}

\item{positive}{Level from dependent variable to predict.
 Either as character or level index (1 or 2 - alphabetically).

 E.g. if we have the levels \code{"cat"} and \code{"dog"} and we want \code{"dog"} to be the positive class,
 we can either provide \code{"dog"} or \code{2}, as alphabetically, \code{"dog"} comes after \code{"cat"}.

 Used when calculating confusion matrix metrics and creating ROC curves.

 N.B. Only affects the evaluation metrics.

 N.B. \strong{Binomial models only}.}

\item{metrics}{List for enabling/disabling metrics.

  E.g. \code{list("RMSE" = FALSE)} would remove RMSE from the results,
  and \code{list("Accuracy" = TRUE)} would add the regular accuracy metric
  to the classification results.
  Default values (TRUE/FALSE) will be used for the remaining metrics available.

  Also accepts the string \code{"all"}.

  N.B. Currently, disabled metrics are still computed.}

\item{include_predictions}{Whether to include the predictions
in the output as a nested tibble. (Logical)}

\item{parallel}{Whether to run evaluations in parallel,
when \code{data} is grouped with \code{\link[dplyr]{group_by}}.}
}
\value{
----------------------------------------------------------------

 \subsection{Gaussian Results}{

 ----------------------------------------------------------------

 Tibble containing the following metrics by default:

 Average \strong{RMSE}, \strong{MAE}, \strong{r2m},
 \strong{r2c}, \strong{AIC}, \strong{AICc}, and \strong{BIC}.

 N.B. Some of the metrics will only be returned if model
 objects were passed, and will be \code{NA} if they could not be
 extracted from the passed model objects.

 Also includes:

 A nested tibble with the \strong{Predictions} and targets.

 A nested tibble with the model \strong{Coefficients}. The coefficients
 are extracted from the model object with \code{\link[broom:tidy]{broom::tidy()}} or
 \code{\link[stats:coef]{coef()}} (with some restrictions on the output).
 If these attempts fail, a default coefficients tibble filled with \code{NA}s is returned.
 }

 ----------------------------------------------------------------

 \subsection{Binomial Results}{

 ----------------------------------------------------------------

 Tibble with the following evaluation metrics, based on a
 confusion matrix and a ROC curve fitted to the predictions:

 ROC:

 \strong{AUC}, \strong{Lower CI}, and \strong{Upper CI}

 Confusion Matrix:

 \strong{Balanced Accuracy},
 \strong{F1},
 \strong{Sensitivity},
 \strong{Specificity},
 \strong{Positive Prediction Value},
 \strong{Negative Prediction Value},
 \strong{Kappa},
 \strong{Detection Rate},
 \strong{Detection Prevalence},
 \strong{Prevalence}, and
 \strong{MCC} (Matthews correlation coefficient).

 Other available metrics (disabled by default, see \code{metrics}):
 \strong{Accuracy}.

 Also includes:

 A nested tibble with the \strong{predictions} and targets.

 A nested tibble with the sensativities and specificities from the \strong{ROC} curve.

 A nested tibble with the \strong{confusion matrix}.
 The \code{Pos_} columns tells you whether a row is a
 True Positive (TP), True Negative (TN), False Positive (FP), or False Negative (FN),
 depending on which level is the "\code{positive}" class. I.e. the level you wish to predict.
 }

 ----------------------------------------------------------------

 \subsection{Multinomial Results}{

 ----------------------------------------------------------------

 For each class, a \emph{one-vs-all} binomial evaluation is performed. This creates
 a \strong{Class Level Results} tibble containing the same metrics as the binomial results
 described above, along with the \strong{Support} metric, which is simply a
 count of the class in the target column. These metrics are used to calculate the macro metrics
 in the output tibble. The nested class level results tibble is also included in the output tibble,
 and would usually be reported along with the macro and overall metrics.

 The output tibble contains the macro and overall metrics.
 The metrics that share their name with the metrics in the nested
 class level results tibble are averages of those metrics
 (note: does not remove \code{NA}s before averaging).
 In addition to these, it also includes the \strong{Overall Accuracy} metric.

 Other available metrics (disabled by default, see \code{metrics}):
 \strong{Accuracy}, \strong{Weighted Balanced Accuracy}, \strong{Weighted Accuracy},
 \strong{Weighted F1}, \strong{Weighted Sensitivity}, \strong{Weighted Sensitivity},
 \strong{Weighted Specificity}, \strong{Weighted Pos Pred Value},
 \strong{Weighted Neg Pred Value}, \strong{Weighted AUC}, \strong{Weighted Lower CI},
 \strong{Weighted Upper CI}, \strong{Weighted Kappa}, \strong{Weighted MCC},
 \strong{Weighted Detection Rate}, \strong{Weighted Detection Prevalence}, and
 \strong{Weighted Prevalence}.

 Note that the "Weighted" metrics are weighted averages, weighted by the \code{Support}.

 Also includes:

 A nested tibble with the \strong{Predictions} and targets.

 A nested tibble with the multiclass \strong{Confusion Matrix}.
 }

 \strong{Class Level Results}

 Besides the binomial evaluation metrics and the \code{Support} metric,
 the nested class level results tibble also contains:

 A nested tibble with the sensativities and specificities from the \strong{ROC} curve.

 A nested tibble with the \strong{Confusion Matrix} from the one-vs-all evaluation.
 The \code{Pos_} columns tells you whether a row is a
 True Positive (TP), True Negative (TN), False Positive (FP), or False Negative (FN),
 depending on which level is the "positive" class. In our case, \code{1} is the current class
 and \code{0} represents all the other classes together.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("maturing")}

 Evaluate your model's predictions
 on a set of evaluation metrics.

 Create ID-aggregated evaluations by multiple methods.

 Currently supports regression and classification
 (binary and multiclass). See \code{type}.
}
\details{
Packages used:

 \strong{Gaussian}:

 r2m : \code{\link[MuMIn:r.squaredGLMM]{MuMIn::r.squaredGLMM}}

 r2c : \code{\link[MuMIn:r.squaredGLMM]{MuMIn::r.squaredGLMM}}

 AIC : \code{\link[stats:AIC]{stats::AIC}}

 AICc : \code{\link[MuMIn:AICc]{MuMIn::AICc}}

 BIC : \code{\link[stats:BIC]{stats::BIC}}

 \strong{Binomial} and \strong{Multinomial}:

 Confusion matrix and related metrics:
 \code{\link[caret:confusionMatrix]{caret::confusionMatrix}}

 ROC and related metrics: \code{\link[pROC:roc]{pROC::roc}}

 MCC: \code{\link[mltools:mcc]{mltools::mcc}}
}
\examples{
\donttest{
# Attach packages
library(cvms)
library(dplyr)

# Load data
data <- participant.scores

# Fit models
gaussian_model <- lm(age ~ diagnosis, data = data)
binomial_model <- glm(diagnosis ~ score, data = data)

# Add predictions
data[["gaussian_predictions"]] <- predict(gaussian_model, data,
                                          type = "response",
                                          allow.new.levels = TRUE)
data[["binomial_predictions"]] <- predict(binomial_model, data,
                                          allow.new.levels = TRUE)

# Gaussian evaluation
evaluate(data = data, target_col = "age",
         prediction_cols = "gaussian_predictions",
         models = list(gaussian_model),
         type = "gaussian")

# Binomial evaluation
evaluate(data = data, target_col = "diagnosis",
         prediction_cols = "binomial_predictions",
         type = "binomial")

# Multinomial

# Create a tibble with predicted probabilities
data_mc <- multiclass_probability_tibble(
    num_classes = 3, num_observations = 30,
    apply_softmax = TRUE, FUN = runif,
    class_name = "class_")

# Add targets
class_names <- paste0("class_", c(1,2,3))
data_mc[["target"]] <- sample(x = class_names,
                              size = 30, replace = TRUE)

# Multinomial evaluation
evaluate(data = data_mc, target_col = "target",
         prediction_cols = class_names,
         type = "multinomial")

# ID evaluation

# Gaussian ID evaluation
# Note that 'age' is the same for all observations
# of a participant
evaluate(data = data, target_col = "age",
         prediction_cols = "gaussian_predictions",
         id_col = "participant",
         type = "gaussian")

# Binomial ID evaluation
evaluate(data = data, target_col = "diagnosis",
         prediction_cols = "binomial_predictions",
         id_col = "participant",
         id_method = "mean", # alternatively: "majority"
         type = "binomial")

# Multinomial ID evaluation

# Add IDs and new targets (must be constant within IDs)
data_mc[["target"]] <- NULL
data_mc[["id"]] <- rep(1:6, each = 5)
id_classes <- tibble::tibble(
    "id" = 1:6,
    target = sample(x = class_names, size = 6, replace = TRUE)
)
data_mc <- data_mc \%>\%
    dplyr::left_join(id_classes, by = "id")

# Perform ID evaluation
evaluate(data = data_mc, target_col = "target",
         prediction_cols = class_names,
         id_col = "id",
         id_method = "mean", # alternatively: "majority"
         type = "multinomial")

# Training and evaluating a multinomial model with nnet

# Create a data frame with some predictors and a target column
class_names <- paste0("class_", 1:4)
data_for_nnet <- multiclass_probability_tibble(
    num_classes = 3, # Here, number of predictors
    num_observations = 30,
    apply_softmax = FALSE,
    FUN = rnorm,
    class_name = "predictor_") \%>\%
    dplyr::mutate(class = sample(
        class_names,
        size = 30,
        replace = TRUE))

# Train multinomial model using the nnet package
mn_model <- nnet::multinom(
    "class ~ predictor_1 + predictor_2 + predictor_3",
    data = data_for_nnet)

# Predict the targets in the dataset
# (we would usually use a test set instead)
predictions <- predict(mn_model, data_for_nnet,
                       type = "probs") \%>\%
    dplyr::as_tibble()

# Add the targets
predictions[["target"]] <- data_for_nnet[["class"]]

# Evaluate predictions
evaluate(data = predictions, target_col = "target",
         prediction_cols = class_names,
         type = "multinomial")
}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
