#' Filter loci in a genlight \{adegenet\} object based on sequence tag length
#'
#' SNP datasets generated by DArT typically have sequence tag lengths ranging from 20 to 69 base pairs.
#' 
#' @param x -- name of the genlight object containing the SNP data [required]
#' @param lower -- lower threshold value below which loci will be removed [default 20]
#' @param upper -- upper threshold value above which loci will be removed [default 69]
#' @param verbose -- verbosity: 0, silent or fatal errors; 1, begin and end; 2, progress log ; 3, progress and results summary; 5, full report [default 2, unless specified using gl.set.verbosity]
#' @return Returns a genlight object retaining loci with a sequence tag length in the range specified by the lower and upper threshold.
#' @export
#' @author Arthur Georges (Post to \url{https://groups.google.com/d/forum/dartr})
#' @examples
#' # SNP data
#'   gl.report.taglength(testset.gl)
#'   result <- gl.filter.taglength(testset.gl,lower=60)
#'   gl.report.taglength(result)
#' # Tag P/A data
#'   gl.report.taglength(testset.gs)
#'   result <- gl.filter.taglength(testset.gs,lower=60)
#'   gl.report.taglength(result)

gl.filter.taglength <- function(x, lower=20, upper=69, verbose=NULL) {

# TRAP COMMAND, SET VERSION
  
  funname <- match.call()[[1]]
  build <- "Jacob"
  
# SET VERBOSITY
  
  if (is.null(verbose)){ 
    if(!is.null(x@other$verbose)){ 
      verbose <- x@other$verbose
    } else { 
      verbose <- 2
    }
  } 
  
  if (verbose < 0 | verbose > 5){
    cat(paste("  Warning: Parameter 'verbose' must be an integer between 0 [silent] and 5 [full report], set to 2\n"))
    verbose <- 2
  }
  
# FLAG SCRIPT START
  
  if (verbose >= 1){
    if(verbose==5){
      cat("Starting",funname,"[ Build =",build,"]\n")
    } else {
      cat("Starting",funname,"\n")
    }
  }
  
# STANDARD ERROR CHECKING
  
  if(class(x)!="genlight") {
    stop("Fatal Error: genlight object required!\n")
  }
  
  if (all(x@ploidy == 1)){
    cat("  Processing Presence/Absence (SilicoDArT) data\n")
  } else if (all(x@ploidy == 2)){
    cat("  Processing a SNP dataset\n")
  } else {
    stop("Fatal Error: Ploidy must be universally 1 (fragment P/A data) or 2 (SNP data)!\n")
  }
  
# FUNCTION SPECIFIC ERROR CHECKING
  
  if(length(x@other$loc.metrics$TrimmedSequence) != nLoc(x)) {
    stop("Fatal Error: Data must include Trimmed Sequences for each loci in a column called 'TrimmedSequence' in the @other$loc.metrics slot.\n")
  }
  if (upper < lower){
    cat("  Warning: Parameter 'upper' must be greater than parameter 'lower', swapping\n")
    tmp <- upper
    upper <- lower
    lower <- tmp
  }
  if (lower < 0 | lower > 250){
    cat("  Warning: Parameter 'verbose' must be an integer between 0 and 250 , set to 20\n")
    lower <- 20
  }
  if (upper < 0 | upper > 250){
    cat("  Warning: Parameter 'upper' must be an integer between 0 and 250 , set to 69\n")
    upper <- 69
  }
    
# DO THE JOB
  
  n0 <- nLoc(x)
  if (verbose > 2) {cat("Initial no. of loci =", n0, "\n")}

  tags <- x@other$loc.metrics$TrimmedSequence
  nchar.tags <- nchar(as.character(tags))
  
    # Remove SNP loci with rdepth < threshold
    if (verbose > 1){cat("  Removing loci with taglength <",lower,"and >",upper,"\n")}
    index <- (nchar.tags>=lower & nchar.tags<= upper)
    x2 <- x[, index]
    # Remove the corresponding records from the loci metadata
    x2@other$loc.metrics <- x@other$loc.metrics[index,]
    if (verbose > 2) {cat ("  No. of loci deleted =", (n0-nLoc(x2)),"\n")}
    
  # REPORT A SUMMARY
  if (verbose > 2) {
    cat("Summary of filtered dataset\n")
    cat(paste("  Sequence Tag Length >=",lower,"and Sequence Tag Length <=",upper,"\n"))
    cat(paste("  No. of loci:",nLoc(x2),"\n"))
    cat(paste("  No. of individuals:", nInd(x2),"\n"))
    cat(paste("  No. of populations: ", length(levels(factor(pop(x2)))),"\n"))
  }  
  
# ADD TO HISTORY
  nh <- length(x2@other$history)
  x2@other$history[[nh + 1]] <- match.call()
    
# FLAG SCRIPT END

  if (verbose > 0) {
    cat("Completed:",funname,"\n")
  }
    
  return(x2)
  
}
