#' Download Asset From A Databrary Session.
#'
#' @description Databrary stores file types (assets) of many types. This
#' function downloads assets in a data frame generated by list_session_assets().
#'
#' @param session_df A data frame as generated by list_session_assets_2().
#' @param session_id An integer. Slot/session number where target file is
#' stored. Default is the unique session_id from `session_df`. This makes piped
#' workflows possible.
#' @param target_dir A character string. Directory to save the downloaded file.
#' Default is directory named after the session_id.
#' @param vb A logical value. If TRUE provides verbose output. Default is FALSE.
#' @param overwrite A logical value. Overwrite existing data or not. Default is TRUE.
#' @param rq A list in the form of an `httr2` request object. Default is NULL.
#'
#' @returns Full file names to the downloaded assets or NULL.
#'
#' @examples
#' \donttest{
#' \dontrun{
#' download_session_assets_fr_df() # Downloads all of the files from session
#' 9807 in Databrary volume 1.
#'
#' # Just the CSVs
#' v1 <- list_session_assets()
#' v1_csv <- dplyr::filter(v1, format_extension == "csv")
#' download_session_assets_fr_df(v1_csv, vb = TRUE)
#' }
#' }
#' @export
download_session_assets_fr_df <-
  function(session_df = list_session_assets(),
           session_id = unique(session_df$session_id),
           target_dir = file.path(session_id),
           overwrite = TRUE,
           vb = FALSE,
           rq = NULL) {
    # Check parameters
    assertthat::assert_that(is.data.frame(session_df))
    
    assertthat::assert_that("session_id" %in% names(session_df))
    
    assertthat::assert_that(length(target_dir) == 1)
    assertthat::assert_that(is.character(target_dir))
    if (!dir.exists(target_dir)) {
      if (vb) {
        message("Target directory not found: ", target_dir)
        message("Creating: ", target_dir)
      }
      dir.create(target_dir, recursive = TRUE)
    } else {
      if (vb)
        message("Target directory exists: ", target_dir)
      if (overwrite) {
        if (vb)
          message("`overwrite` is TRUE. Overwriting directory: ", target_dir)
      } else {
        if (vb)
          message("`overwrite` is FALSE. Cannot continue.")
        return(NULL)
      }
    }
    assertthat::is.writeable(target_dir)
    
    assertthat::assert_that(length(vb) == 1)
    assertthat::assert_that(is.logical(vb))
    
    assertthat::assert_that(is.null(rq) |
                              ("httr2_request" %in% class(rq)))
    
    fns <-
      file.path(target_dir,
                paste0(session_df$asset_name, ".", session_df$format_extension))
    if (vb)
      message(
        "Downloading n=",
        length(fns),
        " files from session_id ",
        session_id,
        " to /",
        target_dir
      )
    purrr::walk2(
      session_df$asset_id,
      fns,
      download_session_asset_fr_df,
      session_id = session_id,
      target_dir = target_dir,
      overwrite = overwrite,
      vb = vb,
      rq = rq,
      .progress = TRUE
    )
    fns
  }
