context("SystemMetadata tests")
test_that("dataone library loads", {
	expect_true(require(dataone))
})
test_that("SystemMetadata constructors", {
    library(datapack)
    sysmeta <- new("SystemMetadata")
    expect_that(sysmeta@serialVersion, equals(1))
    expect_true(is.na(sysmeta@identifier))
    sysmeta <- new("SystemMetadata", identifier="TestId", formatId="text/csv")
    expect_that(sysmeta@identifier, equals("TestId"))
    expect_that(sysmeta@formatId, equals("text/csv"))
})
test_that("XML SystemMetadata parsing works", {
  library(dataone)
  library(XML)
  testid <- "doi:10.xxyy/AA/tesdoc123456789"
  sysmeta <- new("SystemMetadata")
  expect_that(sysmeta@serialVersion, equals(1))
  doc <- xmlParseDoc(system.file("testfiles/sysmeta.xml", package="dataone"), asText=FALSE)
  expect_match(xmlValue(xmlRoot(doc)[["identifier"]]), testid)
  xml <- xmlRoot(doc)
  #getEncoding(doc)
  sysmeta <- parseSystemMetadata(sysmeta, xmlRoot(xml))
  expect_match(sysmeta@identifier, testid)
  expect_that(nrow(sysmeta@accessPolicy), equals(5))
  expect_match(as.character(sysmeta@accessPolicy$permission[[1]]), "read")
  expect_true(sysmeta@archived)
  csattrs <- xmlAttrs(xml[["checksum"]])
  expect_match(sysmeta@checksumAlgorithm, csattrs[[1]])
  expect_true(grep("urn:node:KNB", sysmeta@preferredNodes) > 0)
  expect_true(grep("urn:node:mnUNM1", sysmeta@preferredNodes) > 0)
  expect_true(grep("urn:node:BADNODE", sysmeta@blockedNodes) > 0)
})

test_that("XML SystemMetadata serialization works", {
    library(dataone)
    testid <- "doi:10.xxyy/AA/tesdoc123456789"
    sysmeta <- new("SystemMetadata")
    expect_that(sysmeta@serialVersion, equals(1))
    xml <- xmlParseDoc(system.file("testfiles/sysmeta.xml", package="dataone"), asText=FALSE)
    expect_match(xmlValue(xmlRoot(xml)[["identifier"]]), testid)
    sysmeta <- parseSystemMetadata(sysmeta, xmlRoot(xml))
    expect_match(sysmeta@identifier, testid)
    expect_true(sysmeta@archived)
    xml <- serializeSystemMetadata(sysmeta)
    #cat(xml)
    expect_match(xml, "<d1:systemMetadata")
    expect_match(xml, "<blockedMemberNode>urn:node:BADNODE</blockedMemberNode>")
    expect_match(xml, "<preferredMemberNode>urn:node:KNB</preferredMemberNode>")
    expect_match(xml, "<subject>public</subject>")
    expect_match(xml, "<permission>read</permission>")
    expect_match(xml, "<subject>CN=Subject2,O=Google,C=US,DC=cilogon,DC=org</subject>")
    expect_match(xml, "<permission>changePermission</permission>")
    sysmeta@obsoletes <- ""
    sysmeta <- new("SystemMetadata")
    xml <- serializeSystemMetadata(sysmeta)
    foundObsoletes <- grep("<obsoletes>", xml, invert=TRUE)
    expect_true(as.logical(foundObsoletes))
    # TODO: check tree equivalence with original XML document
})
test_that("SystemMetadata XML constructor works", {
    library(dataone)
    library(XML)
    testid <- "doi:10.xxyy/AA/tesdoc123456789"
    doc <- xmlParseDoc(system.file("testfiles/sysmeta.xml", package="dataone"), asText=FALSE)
    expect_match(xmlValue(xmlRoot(doc)[["identifier"]]), testid)
    xml <- xmlRoot(doc)
    sysmeta <- SystemMetadata(xmlRoot(xml))
    expect_match(sysmeta@identifier, testid)
    expect_that(nrow(sysmeta@accessPolicy), equals(5))
    expect_match(as.character(sysmeta@accessPolicy$permission[[1]]), "read")
    expect_true(sysmeta@archived)
    csattrs <- xmlAttrs(xml[["checksum"]])
    expect_match(sysmeta@checksumAlgorithm, csattrs[[1]])
    expect_true(grep("urn:node:KNB", sysmeta@preferredNodes) > 0)
    expect_true(grep("urn:node:mnUNM1", sysmeta@preferredNodes) > 0)
    expect_true(grep("urn:node:BADNODE", sysmeta@blockedNodes) > 0)
})
test_that("SystemMetadata validation works", {
    library(dataone)
    sysmeta <- new("SystemMetadata", identifier="foo", formatId="text/csv", size=59, checksum="jdhdjhfd", rightsHolder="ff")
    isValid <- validate(sysmeta)
    expect_true(isValid)
    isValid <- validObject(sysmeta)
    expect_true(isValid)
    sysmeta <- new("SystemMetadata", identifier="foo", checksum="jdhdjhfd", rightsHolder="ff")
    errors <- validate(sysmeta)
    expect_that(length(errors), equals(2))
})

test_that("SystemMetadata accessPolicy can be constructed and changed", {
    library(dataone)
    apolicy=data.frame(list("public", "read"))
    colnames(apolicy) <- c("subject", "permission")
    sysmeta <- new("SystemMetadata", identifier="foo", formatId="text/csv", size=59, checksum="jdhdjhfd", rightsHolder="ff", accessPolicy=apolicy)
    expect_that(sysmeta@serialVersion, equals(1))
    expect_that(nrow(sysmeta@accessPolicy), equals(1))
    expect_match(as.character(sysmeta@accessPolicy$permission[[1]]), "read")
    
    sysmeta <- addAccessRule(sysmeta, "foo", "write")
    expect_that(nrow(sysmeta@accessPolicy), equals(2))
    expect_match(as.character(sysmeta@accessPolicy$permission[[2]]), "write")
    
    apolicy=data.frame(subject=c("bar", "baz"), permission= c("changePermission", "write"))
    sysmeta <- addAccessRule(sysmeta, apolicy)
    expect_that(nrow(sysmeta@accessPolicy), equals(4))
    expect_match(as.character(sysmeta@accessPolicy$permission[[4]]), "write")
    expect_match(as.character(sysmeta@accessPolicy$subject[[4]]), "baz")
})
