#' List the rulesets approximating a model generated by DataRobot Prime
#'
#' This function wll return list of rulesets that could be used to approximate the specified model.
#' Rulesets are created using the RequestApproximation function. If model hasn't been approximated
#' yet, will return empty list
#'
#' @inheritParams DeleteProject
#' @param modelId Unique alphanumeric identifier for the model of interest.
#' @return A list of lists with one element for each ruleset. If there are no rulesets created for
#'   a model then an empty list is returned
#' If the group is not empty,a list is returned with the following elements:
#' \itemize{
#'   \item projectId. Character string giving the unique identifier for the project.
#'   \item rulesetId. Integer number giving the identifier for the ruleset.
#'   \item score. Score of ruleset (using project leaderboard metric).
#'   \item parentModelId. Character string giving the unique identifier for the parent model.
#'   \item ruleCount. integer: number of rules in ruleset.
#'   \item modelId. Character string giving the unique identifier for a model using the ruleset.
#'     May be NULL if no model using the ruleset has been created yet.
#' }
#' @examples
#' \dontrun{
#'   projectId <- "59a5af20c80891534e3c2bde"
#'   modelId <- "5996f820af07fc605e81ead4"
#'   GetRulesets(projectId, modelId)
#' }
#' @export
GetRulesets <- function(project, modelId) {
  projectId <- ValidateProject(project)
  routeString <- UrlJoin("projects", projectId, "models", modelId, "primeRulesets")
  rawResponse <- DataRobotGET(routeString, simplifyDataFrame = FALSE)
  lapply(rawResponse, as.dataRobotGetRuleset)
}

as.dataRobotGetRuleset <- function(inList) {
  elements <- c("projectId",
                "rulesetId",
                "score",
                "parentModelId",
                "ruleCount",
                "modelId")
  ApplySchema(inList, elements)
}
