% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/delaunay.R
\name{delaunay}
\alias{delaunay}
\title{Delaunay tessellation}
\usage{
delaunay(points, elevation = FALSE, constraints = NULL, quick3d = FALSE)
}
\arguments{
\item{points}{numeric matrix which stores the points, one point per row}

\item{elevation}{if points are three-dimensional and \code{elevation=TRUE},
then the function performs an elevated two-dimensional Delaunay
triangulation, using the \code{z} coordinates of the points for the
elevations; see the example}

\item{constraints}{\emph{for 2D only}, some edges to perform a constrained
Delaunay triangulation, given as an integer matrix with two columns (each
row provides the indices of the two points forming the edge);
\code{NULL} for no constraint}

\item{quick3d}{Boolean, for 3D only; if \code{FALSE}, there is more 
information in the output about the Delaunay tessellation; see the 
\strong{Value} section for details}
}
\value{
The Delaunay tessellation.
\itemize{
  \item \strong{If the dimension is 2} and \code{constraints=NULL},
        the returned value is a list with three fields:
        \code{faces}, \code{edges} and \code{area}. The \code{faces} field
        contains an integer matrix with three columns; each row represents a
        triangle whose each vertex is given by the index (row number) of
        this point in the \code{points} matrix. The \code{edges} field
        also contains an integer matrix with three columns. The two first
        integers of a row are the indices of the two points which form the
        edge. The third column, named \code{border}, only contains some
        zeros and some ones; a border (exterior) edge is labelled by a
        \code{1}. The \code{area} field contains only a number: the area
        of the triangulated region (that is, the area of the convex hull of
        the points).
  \item \strong{If the dimension is 2} and \code{constraints} is not
        \code{NULL}, the returned value is a list with
        four fields: \code{faces}, \code{edges}, \code{constraints}, and
        \code{area}. The \code{faces} field
        contains an integer matrix with three columns; each row represents a
        triangle whose each vertex is given by the index (row number) of
        this point in the \code{points} matrix. The \code{edges} field
        is a dataframe with four columns. The first two columns provide
        the edges of the triangulation; they are given by row, the two
        integers of a row are the indices of the two points which form the
        edge. Each integer of the third column is the index of the face
        the corresponding edge belongs to. The fourth column,
        named \code{border}, only contains some
        zeros and some ones; a border edge is labelled by a
        \code{1}.
        The \code{constraints} field is an integer matrix with
        two columns, it represents the constraint edges.
        Finally, the \code{area} field contains only a number: the area
        of the triangulated region.
  \item \strong{If the dimension is 3}, the returned value is a list with
        four fields: \code{cells}, \code{facets}, \code{edges}, and
        \code{volume}. The \code{cells} field represents the tetrahedra
        which form the tessellation. The \code{facets} field represents
        the faces of these tetrahedra, some triangles. The \code{edges}
        field represents the edges of these triangles. The \code{volume}
        field provides only one number, the volume of the tessellation,
        in other words the volume of the convex hull of the given points.
        If \code{quick3d=TRUE}, then \code{cells}, \code{facets} and 
        \code{edges} are integer matrices with four, three, and two 
        columns respectively; each integer is a vertex index. 
        If \code{quick3d=FALSE}, the \code{cells} field is a list of lists. 
        Each sublist is composed of three fields: \code{cell} provides the 
        indices of the four vertices of the corresponding tetrahedron, 
        \code{faces} provides the indices of the four faces of the 
        tetrahedron, that is to say the row number of the \code{facets} 
        field which represents this face, and finally there is a 
        \code{volume} field which provides the volume of the tetrahedron. 
        The \code{facets} field is an integer matrix with four columns. 
        The three first integers of a row are the indices of the points 
        which form the corresponding facet. The fourth column, named 
        \code{onhull} is composed of zeros and ones only, and a \code{1} 
        means that the corresponding facet lies on the convex hull of the 
        points. The \code{edges} field contains an integer matrix with 
        three columns. Each row represents an edge, given by the two
        indices of the points which form this edge, and the third integer,
        in the column named \code{onhull} is a \code{0/1} indicator of
        whether the edge lies on the convex hull. Finally the \code{volume}
        field provides only one number, the volume of the tessellation (i.e.
        the volume of the convex hull of the points).
  \item \strong{If} \code{elevation=TRUE}, the returned value is a list with
        five fields: \code{mesh}, \code{edges}, \code{faceVolumes},
        \code{volume} and \code{area}. The \code{mesh} field is an object of
        class \code{mesh3d}, ready for plotting with the \strong{rgl}
        package. The \code{edges} field provides the indices of the edges,
        given as an integer matrix with two columns. The \code{faceVolumes}
        field is a numeric vector, it provides the volumes under the faces
        that can be found in the \code{mesh} field. The \code{volume} field
        provides the sum of these volumes, that is to say the total volume
        under the triangulated surface. Finally, the \code{area} field
        provides the sum of the areas of all triangles, thereby
        approximating the area of the triangulated surface.
}
}
\description{
Delaunay tessellation of a set of 2D or 3D points.
}
\examples{
library(delaunay)
# elevated Delaunay triangulation ####
f <- function(x, y){
  2 * exp(-(x^2 + y^2)) # integrate to 2pi
}
x <- y <- seq(-4, 4, length.out = 50)
grd <- transform(expand.grid(x = x, y = y), z = f(x, y))
del <- delaunay(as.matrix(grd), elevation = TRUE)
# `del` is a list; its first component is a mesh representing the surface:
mesh <- del[["mesh"]]
library(rgl)
open3d(windowRect = c(50, 50, 562, 562))
shade3d(mesh, color = "limegreen")
wire3d(mesh)
# in `del` you can also found the volume under the surface, which should
#   approximate the integral of the function:
del[["volume"]]
}
