% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PF.R
\name{PF_get_score_n_hess}
\alias{PF_get_score_n_hess}
\title{Approximate Observed Information Matrix and Score Vector}
\usage{
PF_get_score_n_hess(object, debug = FALSE, use_O_n_sq = FALSE)
}
\arguments{
\item{object}{object of class \code{\link{PF_EM}}.}

\item{debug}{\code{TRUE} if debug information should be printed to the
console.}

\item{use_O_n_sq}{\code{TRUE} if the method from Poyiadjis et al. (2011)
should be used.}
}
\value{
A list with the following functions as elements
\item{run_particle_filter}{function to run particle filter as with
\code{\link{PF_forward_filter}}.}
\item{set_parameters}{function to set the parameters in the model.
The first argument is a vectorized version of \eqn{F} matrix and \eqn{Q}
matrix. The second argument is the fixed effect coefficients.}
\item{set_n_particles}{sets the number of particles to use in
\code{run_particle_filter} and \code{get_get_score_n_hess} when
\code{use_O_n_sq} is \code{TRUE}.}
\item{get_get_score_n_hess}{approximate the observed information
matrix and score vector. The argument toggles whether or not to approximate
the observed information matrix. The last particle cloud
from \code{run_particle_filter} is used when \code{use_O_n_sq} is
\code{FALSE}.}
}
\description{
Returns a list of functions to approximate the observed information matrix
and score vector.
}
\details{
The score vector and observed information matrix are computed
with the (forward)
particle filter. This comes at an \eqn{O(d^2)} variance where \eqn{d}
is the number of periods. Thus, the approximation may be poor for long
series. The score vector can be used to perform stochastic gradient
descent.

If \code{use_O_n_sq} is \code{TRUE} then the method in Poyiadjis et al. (2011)
is used. This may only have a variance which is linear in the number of
time periods. However, the present implementation is \eqn{O(N^2)} where
\eqn{N} is the number of particles. The method uses a particle filter as
in Section 3.1
of Lin et al. (2005). There is no need to call
\code{run_particle_filter} unless one wants a new approximation of the
log-likelihood as a separate filter is run with \code{get_get_score_n_hess}
when \code{use_O_n_sq} is \code{TRUE}.
}
\section{Warning}{

The function is still under development so the output and API may change.
}

\examples{
\donttest{
library(dynamichazard)
.lung <- lung[!is.na(lung$ph.ecog), ]
# standardize
.lung$age <- scale(.lung$age)

# fit model
set.seed(43588155)
pf_fit <- PF_EM(
  fixed = Surv(time, status == 2) ~ ph.ecog + age,
  random = ~ 1, model = "exponential",
  data = .lung, by = 50, id = 1:nrow(.lung),
  Q_0 = as.matrix(1), Q = as.matrix(.5^2), type = "VAR",
  max_T = 800, Fmat = as.matrix(.5),
  control = PF_control(
    N_fw_n_bw = 250, N_first = 2000, N_smooth = 500, covar_fac = 1.1,
    nu = 6, n_max = 1000L, eps = 1e-4, averaging_start = 200L,
    n_threads = 2))

# compute score and observed information matrix
comp_obj <- PF_get_score_n_hess(pf_fit)
comp_obj$set_n_particles(N_fw = 10000L, N_first = 10000L)
comp_obj$run_particle_filter()
(o1 <- comp_obj$get_get_score_n_hess())

# O(N^2) method with lower variance as a function of time
comp_obj <- PF_get_score_n_hess(pf_fit, use_O_n_sq = TRUE)
comp_obj$set_n_particles(N_fw = 2500L, N_first = 2500L)
(o2 <- comp_obj$get_get_score_n_hess())

# approximations may have large variance
o3 <- replicate(10L, {
  runif(1)
  pf_fit$seed <- .Random.seed
  comp_obj <- PF_get_score_n_hess(pf_fit)
  comp_obj$set_n_particles(N_fw = 10000L, N_first = 10000L)
  comp_obj$run_particle_filter()
  comp_obj$get_get_score_n_hess()
}, simplify = FALSE)
sapply(o3, function(x) x$score)
sapply(o3, function(x) sqrt(diag(solve(x$obs_info))))
}
}
\references{
Cappe, O. and Moulines, E. (2005) Recursive Computation of the Score and
Observed Information Matrix in Hidden Markov Models.
\emph{IEEE/SP 13th Workshop on Statistical Signal Processing}.

Cappe, O., Moulines, E. and Ryden, T. (2005) Inference in Hidden Markov
Models (Springer Series in Statistics). Springer-Verlag.

Doucet, A., and Tadić, V. B. (2003) Parameter Estimation in General
State-Space Models Using Particle Methods.
\emph{Annals of the Institute of Statistical Mathematics}, \strong{55(2)},
409–422.

Lin, M. T., Zhang, J. L., Cheng, Q. and Chen, R. (2005) Independent
Particle Filters. \emph{Journal of the American Statistical Association},
\strong{100(472)}, 1412-1421.

Poyiadjis, G., Doucet, A. and Singh, S. S. (2011) Particle Approximations of
the Score and Observed Information Matrix in State Space Models with
Application to Parameter Estimation. \emph{Biometrika}, \strong{98(1)},
65--80.
}
\seealso{
See the examples at https://github.com/boennecd/dynamichazard/tree/master/examples.
}
