% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/joint_model.R
\name{joint_model}
\alias{joint_model}
\title{Specify and fit joint model using count data from traditional surveys and
eDNA PCR data}
\usage{
joint_model(
  data,
  cov = NULL,
  family = "poisson",
  p10_priors = c(1, 20),
  q = FALSE,
  phi_priors = NULL,
  multicore = FALSE,
  initial_values = NULL,
  n_chain = 4,
  n_warmup = 500,
  n_iter = 3000,
  thin = 1,
  adapt_delta = 0.9,
  verbose = TRUE,
  seed = NULL
)
}
\arguments{
\item{data}{A list containing data necessary for model fitting. Valid tags
are \code{pcr_n}, \code{pcr_k}, \code{count}, \code{count_type}, and \code{site_cov}. \code{pcr_n} and
\code{pcr_k} are matrices or data frames with first dimension equal to the
number of sites (i) and second dimension equal to the maximum number of
eDNA samples at a given site (m). \code{pcr_n} contains the total number of
PCR replicates per site and eDNA sample, and \code{pcr_k} contains the total
number of positive PCR detections per site and eDNA sample. \code{count} is a
matrix or data frame of traditional survey count data, with first
dimension equal to the number of sites (i) and second dimension equal to
the maximum number of traditional survey replicates at a given site (j).
\code{count_type} is an optional matrix or data frame of integers indicating
gear type used in corresponding count data, with first dimension equal to
the number of sites (i) and second dimension equal to the maximum number
of traditional survey replicates at a given site. Values should be
integers beginning with 1 (referring to the first gear type) to n (last
gear type). \code{site_cov} is an optional matrix or data frame of site-level
covariate data, with first dimension equal to the number of sites (i).
\code{site_cov} should include informative column names. Empty cells should
be NA and will be removed during processing. Sites, i, should be consistent
in all PCR, count, and site covariate data.}

\item{cov}{A character vector indicating the site-level covariates to include
in the model. Default value is NULL.}

\item{family}{The distribution class used to model traditional survey count
data. Options include poisson ('poisson'), negative binomial ('negbin'),
and gamma ('gamma'). Default value is 'poisson'.}

\item{p10_priors}{A numeric vector indicating beta distribution
hyperparameters (alpha, beta) used as the prior distribution for the eDNA
false positive probability (p10). Default vector is c(1,20).}

\item{q}{A logical value indicating whether to estimate gear scaling
coefficients, q, for traditional survey gear types (TRUE) or to not
estimate gear scaling coefficients, q, for traditional survey gear types
(FALSE). Default value is FALSE.}

\item{phi_priors}{A numeric vector indicating gamma distribution
hyperparameters (shape, rate) used as the prior distribution for phi, the
overdispersion in the negative binomial distribution for traditional survey
gear data. Used when family = 'negbin.' If family = 'negbin', then
default vector is c(0.25,0.25), otherwise, default is NULL.}

\item{multicore}{A logical value indicating whether to parallelize chains
with multiple cores. Default is FALSE.}

\item{initial_values}{A list of lists of initial values to use in MCMC. The
length should equal the number of MCMC chains. Initial values can be
provided for parameters: beta, p10 (log-scale), mu, q, alpha. If no
initial values are provided, default random values are drawn.}

\item{n_chain}{Number of MCMC chains. Default value is 4.}

\item{n_warmup}{A positive integer specifying the number of warm-up MCMC
iterations. Default value is 500.}

\item{n_iter}{A positive integer specifying the number of iterations for each
chain (including warmup). Default value is 3000.}

\item{thin}{A positive integer specifying the period for saving samples.
Default value is 1.}

\item{adapt_delta}{Numeric value between 0 and 1 indicating target average
acceptance probability used in \code{rstan::sampling}. Default value is 0.9.}

\item{verbose}{Logical value controlling the verbosity of output (i.e.,
warnings, messages, progress bar). Default is TRUE.}

\item{seed}{A positive integer seed used for random number generation in
MCMC. Default is NULL, which means the seed is generated from 1 to the
maximum integer supported by R.}
}
\value{
A list of:
\itemize{
\item a model object of class \code{stanfit} returned by \code{rstan::sampling}
\item initial values used in MCMC
}
}
\description{
This function implements a Bayesian model that integrates data from paired
eDNA and traditional surveys, as described in Keller et al. (2022)
<doi.org/10.1002/eap.2561>. The model estimates parameters including
the expected species catch rate and the probability of false positive eDNA
detection. This function allows for optional model variations, like inclusion
of site-level covariates that scale the sensitivity of eDNA sampling relative
to traditional sampling, as well as estimation of gear scaling coefficients
that scales the relative catchability of multiple traditional gear types.
Model is implemented using Bayesian inference using the \code{rstan} package,
which uses Hamiltonian Monte Carlo to simulate the posterior distributions.
See more examples in the
\href{https://ednajoint.netlify.app/}{Package
Vignette}.
}
\note{
Before fitting the model, this function checks to ensure that the
model specification is possible given the data files. These checks include:
\itemize{
\item  All tags in data are valid (i.e., include pcr_n, pcr_k, count,
count_type, and site_cov).
\item  Dimensions of pcr_n and pcr_k are equal, and dimensions of count and
count_type are equal (if count_type provided).
\item  Number of sites in PCR and count data are equal.
\item  All data are numeric (i.e., integer or NA).
\item  Empty data cells (NA) match in pcr_n and pcr_k and in count and
count_type.
\item  family is either 'poisson', 'negbin', or 'gamma'.
\item  p10_priors and phi_priors (if used) is a vector of two numeric values.
\item  site_cov has same number of rows as pcr_n and count, if present
\item  site_cov is numeric, if present
\item  cov values match column names in site_cov, if present
}

If any of these checks fail, the function returns an error message.
}
\examples{
\donttest{
# Ex. 1: Implementing the joint model

# Load data
data(goby_data)

# Examine data in list
names(goby_data)

# Note that the surveyed sites (rows) should match in all data
dim(goby_data$pcr_n)[1]
dim(goby_data$count)[1]

# Fit a basic model with paired eDNA and traditional survey data.
# Count data is modeled using a poisson distribution.
fit <- joint_model(data = goby_data, family = "poisson",
                   p10_priors = c(1, 20),
                   multicore = FALSE)

# Ex. 2: Implementing the joint model with site-level covariates

# With the same data, fit a model including 'Filter_time' and 'Salinity'
# site-level covariates
# These covariates will scale the sensitivity of eDNA sampling relative to
# traditional surveys
# Count data is modeled using a poisson distribution.
fit_cov <- joint_model(data = goby_data, cov = c('Filter_time','Salinity'),
                       family = "poisson", p10_priors = c(1, 20),
                       multicore = FALSE)


# Ex. 3: Implementing the joint model with multiple traditional gear types

# Load data
data(green_crab_data)

# Examine data in list
names(green_crab_data)

# Note that the surveyed sites (rows) should match in all data
dim(green_crab_data$pcr_n)[1]
dim(green_crab_data$count)[1]

# Fit a model estimating a gear scaling coefficient for traditional survey
# gear types.
# This model does not assume all traditional survey methods have the same
# catchability.
# Count data is modeled using a negative binomial distribution.
fit_q <- joint_model(data = green_crab_data, cov = NULL, family = "negbin",
                     p10_priors = c(1, 20), q = TRUE,
                     phi_priors = c(0.25, 0.25),
                     multicore = FALSE, initial_values = NULL,
                     n_chain = 4, n_warmup = 500,
                     n_iter = 3000, thin = 1, adapt_delta = 0.9,
                     verbose = TRUE, seed = 123)
}

}
