% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/obj_find.r
\name{obj_find}
\alias{obj_find}
\title{Find an object in the workspace including user-defined environments}
\usage{
obj_find(obj, envir = NULL, envmap = NULL, globalsearch = TRUE, n = 0,
  return_address = FALSE, include_functions = FALSE, silent = TRUE)
}
\arguments{
\item{obj}{object to be searched given as the object itself or as a character string. If given as an object,
expressions are accepted (see details on how expressions are handled).}

\item{envir}{environment where the search for \code{obj} should be carried out.
It defaults to \code{NULL} which means \code{obj} is searched in the calling environment
(i.e. in the environment calling this function), unless \code{globalsearch=TRUE} in which case
it is searched in the whole workspace.}

\item{envmap}{data frame containing a lookup table with name-address pairs of environment names and
addresses to be used when searching for environment \code{env}. It defaults to \code{NULL} which means that the
lookup table is constructed on the fly with the environments defined in the \code{envir} environment
--if not \code{NULL}--, or in the whole workspace if \code{envir=NULL}.
See the details section for more information on its structure.}

\item{globalsearch}{when \code{envir=NULL} it specifies whether the search for \code{obj} should be done
globally, i.e. in the whole workspace, or just within the calling environment.}

\item{n}{non-negative integer indicating the number of levels to go up from the calling function environment
to evaluate \code{obj}. It defaults to 0 which implies that \code{obj} is evaluated in the environment
of the calling function (i.e. the function that calls \code{obj_find()}).}

\item{return_address}{whether to return the address of the environments where the object is found in addition
to their names.}

\item{include_functions}{whether to include funtion execution environments as environments where the object
is searched for. Set this flag to \code{TRUE} with caution because there may be several functions where the
same object is defined, for instance functions that are called as part of the object searching process!}

\item{silent}{run in silent mode? If not, the search history is shown,
listing all the environments that are searched for object \code{obj}. It defaults to TRUE.}
}
\value{
The return value depends on the value of parameter \code{return_address}: when \code{FALSE}
(the default) it returns an array containing the names of the environments where the object \code{obj}
is found; when \code{TRUE} it returns a list with two attributes: \code{"env_full_names"} and
\code{"env_addresses"} with respectively the environment names and addresses where the object is found.
}
\description{
Look for an object in the whole workspace including all environments defined within it
(possibly recursively) and return ALL the environment(s) where the object is found.
User-defined environments are also searched.
Note that both the "recursive search" and the "user-defined environments search" makes this function
quite different from functions \link{find} and \link{exists} of the base package.
Optionally, the search can be limited to a specified environment, as opposed to carrying it out in the whole workspace.
Still, all user-defined environments defined inside the specified environment are searched.
}
\details{
An object is found in an environment if it is reachable from within that environment. An object is considered
reachable in an environment if either one of the following occurs:
\itemize{
\item it exists in the given environment
\item it exists in a user-defined environment defined inside the given environment or in any environment
recursively defined inside them
}

Note that \code{obj_find} differs from base functions \code{find} and \code{exists} in that \code{obj_find}
searches for the object inside user-defined environments within any given environment in a \emph{recursive} way.

In particular, compared to:
\itemize{
\item{\code{find}:} \code{obj_find} searches for objects inside user-defined environments while \code{find} is not
able to do so (see examples).
\item{\code{exists}:} \code{obj_find} \emph{never} searches for objects in the parent environment of \code{envir}
when \code{envir} is not \code{NULL}, as is the case with the \code{exists} function when its \code{inherits}
parameter is set to \code{TRUE} (the default).
If it is wished to search for objects in parent environments, simply set \code{envir=NULL}
and \code{globalsearch=TRUE}, in which case the object will be searched in the whole workspace
and the environments where it is found will be returned.
}

When the object is found, an array containing the names of all the environments where the object is found is
returned.

When \code{envir} is not \code{NULL} attached packages are not included in the search for \code{obj},
unless of course \code{envir} is itself a package environment.

When given as an object, \code{obj} can be an expression. If the expression is an attribute of a list
or an array element, the object contained therein is searched for.
Ex: if \code{alist$var = "x"} then object \code{x} is searched.

If \code{envmap} is passed it should be a data frame providing an address-name pair lookup table
of environments and should contain at least the following columns:
\itemize{
\item{\code{location}} for user-defined environments, the name of the environment where the environment
is located; otherwise \code{NA}.
\item{\code{pathname}} the full \emph{environment path} to reach the environment separated by \code{$}
(e.g. \code{"env1$env$envx"})
\item{\code{address}} the 8-digit (32-bit architectures) thru 16-digit (64-bit architectures) memory address
of the environment given in \code{pathname} enclosed in < > (e.g. \code{"<0000000007DCFB38>"}
(64-bit architectures))
Be ware that Linux Debian distributions may have a 12-digit memory address representation.
So the best way to know is to check a memory address by calling e.g. `address("x")`.
}
Passing an \code{envmap} lookup table is useful for speedup purposes, in case several calls to this
function will be performed in the context of an unchanged set of defined environments.
Such \code{envmap} data frame can be created by calling \link{get_env_names}.
Use this parameter with care, as the matrix passed may not correspond to the actual mapping of existing
environments to their addresses and in that case results may be different from those expected.
}
\examples{
# Define a variable in the global environment
x = 4
# Create new environments, some nested
env1 = new.env()
with(env1, envx <- new.env())
env1$x = 3
env1$envx$x = 2
env1$y = 5

# Look for objects (crawling environments recursively)
obj_find(x)                  # "env1" "env1$envx" "R_GlobalEnv"
obj_find("x")                # "env1" "env1$envx" "R_GlobalEnv"
obj_find("x", envir=env1)    # "env1" "envx" (as the search is limited to the env1 environment)
obj_find("y")                # "env1"
obj_find(nonexistent)        # NULL (note that NO error is raised even if the object does not exist)
}

