#' Set or get variable label
#' 
#' These functions set/get/drop variable labels. For utilizing labels in base R 
#' see \link{f}, \link{names2labels}, \link{values2labels}, \link{unlab}. For
#' value labels see \link{val_lab}.
#' \itemize{
#' \item{\code{var_lab}}{ returns variable label or NULL if label doesn't 
#' exist.} 
#' \item{\code{var_lab<-}}{ set variable label.} 
#' \item{\code{set_var_lab}}{ returns variable with label.}
#' \item{\code{unvr}}{ drops variable label.} 
#' }
#' @param x Variable. In the most cases it is numeric vector.
#' @param value A character scalar - label for the variable x.
#' @return \code{var_lab} return variable label. If label doesn't exist it return
#'   NULL . \code{var_lab<-} and \code{set_var_lab} return variable (vector x)
#'   of class "labelled" with attribute "label" which equals submitted value.
#' @details Variable label is stored in attribute "label" (\code{attr(x,"label")}). For
#'   preserving from dropping this attribute during some operations (such as \code{c})
#'   variable class is set to "labelled". There are special methods of
#'   subsetting and concatenation for this class. To drop variable label use 
#'   \code{var_lab(var) <- NULL} or \code{unvr(var)}.
#' @export  
#' @examples
#' data(mtcars)
#' mtcars = modify(mtcars,{
#'                 var_lab(mpg) = "Miles/(US) gallon"
#'                 var_lab(cyl) = "Number of cylinders"
#'                 var_lab(disp) = "Displacement (cu.in.)"
#'                 var_lab(hp) = "Gross horsepower"
#'                 var_lab(drat) = "Rear axle ratio"
#'                 var_lab(wt) = "Weight (lb/1000)"
#'                 var_lab(qsec) = "1/4 mile time"
#'                 var_lab(vs) = "V/S"
#'                 var_lab(am) = "Transmission (0 = automatic, 1 = manual)"
#'                 var_lab(gear) = "Number of forward gears"
#'                 var_lab(carb) = "Number of carburetors"
#' })
#' 
#' # note: we exclude dependent variable 'mpg' from conversion to use its short name in formula
#' summary(lm(mpg ~ ., data = n2l(mtcars, exclude = "mpg")))
#' 
#' data(mtcars)
#' 
#' var_lab(mtcars$am) = "Transmission"
#' val_lab(mtcars$am) = c(automatic = 0, manual=1)
#' 
#' \dontrun{
#' plot(f(mtcars$am))
#' }
#' 
#' fre(mtcars$am)
#' 
var_lab=function(x){
    UseMethod("var_lab")
}

#' @export
var_lab.default=function(x){
    attr(x,"label", exact = TRUE)
}

#' @export
var_lab.data.frame=function(x)
    ## Drop this function???
    ## mainly for multiple choice questions
    ## if there is no label on the data.frame itself
    ## we return label from the first variable 
{
    res=var_lab.default(x)
    if (is.null(res)){
        all_labs=lapply(x,var_lab)
        all_labs=all_labs[!sapply(all_labs,is.null)]
        if (length(all_labs)>0) res=all_labs[[1]] else res=NULL
    }
    res
}

#' @rdname var_lab
#' @export
"var_lab<-"=function(x,value){
    set_var_lab(x,value)
}

#' @rdname var_lab
#' @export
set_var_lab=function(x,value){
    UseMethod("set_var_lab")
}

#' @export
set_var_lab.list = function(x,value){
    for (each in seq_along(x)) var_lab(x[[each]]) = value
    x
}

#' @export
set_var_lab.data.frame = function(x,value){
    for (each in seq_along(x)) var_lab(x[[each]]) = value
    x
}


#' @export
set_var_lab.default = function(x,value){
    if (length(value)==0){
        attr(x,"label")=NULL
        return(x)
    }
    attr(x,"label")=value
    class(x)=union("labelled",class(x))
    x
}





#'@rdname var_lab
#' @export
unvr=function(x){
    UseMethod("unvr")
}

#' @export
unvr.default=function(x){
    set_var_lab(x,NULL)
}

#' @export
unvr.data.frame=function(x){
    for (each in seq_along(x)) x[[each]] = unvr(x[[each]])
    x
}

#' @export
unvr.list=function(x){
    for (each in seq_along(x)) x[[each]] = unvr(x[[each]])
    x
}


############# value labels #######################

#' Set or get value labels
#' 
#' These functions set/get/drop value labels. Duplicated values are not allowed.
#' If argument \code{x} is data.frame or list then labels applied to all 
#' elements of data.frame/list. To drop value labels, use \code{val_lab(var) <- 
#' NULL} or \code{unvl(var)}. \code{make_labels} converts text from the form 
#' that usually used in questionnaires to named vector. See examples. For 
#' utilizing labels in base R see \link{f}, \link{names2labels}, 
#' \link{values2labels}, \link{unlab}, \link{dichotomy}. For variable labels see
#' \link{var_lab}.
#' \itemize{
#' \item{\code{val_lab}}{ returns value labels or NULL if labels doesn't 
#' exist.} 
#' \item{\code{val_lab<-}}{ set value labels.}
#' \item{\code{set_val_lab}}{ returns variable with value labels.} 
#' \item{\code{add_val_lab<-}}{ add value labels to already existing value labels.} 
#' \item{\code{unvl}}{ drops value labels.}
#' \item{\code{make_labels}}{ makes named vector from text for usage as value labels.}
#' \item{\code{ml_left} and \code{ml_right}}{ are shortcuts for \code{make_labels}
#' with \code{code_postion} 'left' and 'right' accordingly.}
#' }
#' @param x Variable(s). Vector/data.frame/list.
#' @param value Named vector. Names of vector are labels for the
#'   appropriate values of variable x.
#' @param add Logical. Should we add value labels to old labels or replace it?
#'   Deafult is FALSE - we completely replace old values. If TRUE new value
#'   labels will be combined with old value labels.
#' @param text text that should be converted to named vector
#' @param code_position Possible values "left" or "right" - position of numeric code in
#' \code{text}.
#' @return \code{val_lab} return value labels (named vector). If labels doesn't 
#'   exist it return NULL . \code{val_lab<-} and \code{set_val_lab} return 
#'   variable (vector x) of class "labelled" with attribute "labels" which
#'   contains value labels. \code{make_labels} return named vector for usage as
#'   value labels.
#' @details Value labels are stored in attribute "labels" 
#'   (\code{attr(x,"labels")}). We set variable class to "labelled" for preserving
#'   labels from dropping during some operations (such as \code{c} and \code{`[`}).
#'   There are special methods of subsetting and concatenation for this class.
#' @export
#' @examples
#' # toy example
#' set.seed(123)
#' # score - evaluation of tested product
#' 
#' score = sample(-1:1,20,replace = TRUE)
#' var_lab(score) = "Evaluation of tested brand"
#' val_lab(score) = c("Dislike it" = -1,
#'                    "So-so" = 0,
#'                    "Like it" = 1    
#'                    )
#' 
#' # frequency of product scores                                      
#' fre(score)
#' 
#' # brands - multiple response question
#' # Which brands do you use during last three months? 
#' 
#' brands = t(replicate(20,sample(c(1:5,NA),4,replace = FALSE)))
#'
#' var_lab(brands) = "Used brands"
#' val_lab(brands) = make_labels("
#'                               1 Brand A
#'                               2 Brand B
#'                               3 Brand C
#'                               4 Brand D
#'                               5 Brand E
#'                               ")
#' 
#' 
#' # percentage of used brands
#' fre(brands)
#' 
#' # percentage of brands within each score
#' cro(brands, score)
#' 
#' 
#' aggregate(dichotomy(brands) ~ f(score), FUN = mean)
#' 
#' # customer segmentation by used brands
#' kmeans(dichotomy(brands),3)
#' 
#' # model of influence of used brands on evaluation of tested product 
#' summary(lm(score ~ dichotomy(brands)))
#' 
#' ## make labels from text copied from questionnaire
#' 
#' age = c(1, 2, 1, 2)
#' 
#' val_lab(age) = make_labels("
#'  1. 18 - 26
#'  2. 27 - 35
#' ")
#' 
#' f(age)
#' 
#' # or, if in original codes is on the right side
#' 
#' products = 1:8
#' 
#' val_lab(products) = ml_right("
#'  Chocolate bars    1
#'  Chocolate sweets (bulk)	2
#'  Slab chocolate(packed)	3
#'  Slab chocolate (bulk)	4
#'  Boxed chocolate sweets	5
#'  Marshmallow/pastilles in chocolate coating	6
#'  Marmalade in chocolate coating	7
#'  Other	8
#' ")
#' 
#' f(products)
val_lab=function(x){
    UseMethod("val_lab")
}

#' @export
val_lab.data.frame=function(x)
{
    
    res=val_lab.default(x)
    # strange code below for case when we consider data.frame as multiple response question
    if (is.null(res)){
        all_labs=lapply(x, val_lab)
        all_labs=all_labs[!sapply(all_labs,is.null)]
        if (length(all_labs)>0) res=do.call(combine_labels, all_labs) else res=NULL
    }
    res
}

#' @export
val_lab.default=function(x){
    attr(x,"labels", exact = TRUE)
}

#####################

#' @export
#' @rdname val_lab 
"val_lab<-"=function(x, value){
    set_val_lab(x, value, add = FALSE)
}

#####################

#' @export
#' @rdname val_lab 
set_val_lab = function(x, value, add = FALSE){
    UseMethod("set_val_lab")
}

#' @export
#' @rdname val_lab
add_val_lab = function(x, value) set_val_lab(x, value, add = TRUE) 

#' @export
set_val_lab.default = function(x,value, add = FALSE){
    stopif(anyDuplicated(value),"duplicated values in labels: ",paste(value[duplicated(value)],collapse=" "))
    names_vallab = names(value)
    # This warning was removed because it was generated too often for third party *.sav files.
    # if (anyDuplicated(names_vallab)){
    #     duplicates = duplicated(names_vallab)
    #     warning(paste0("duplicated labels: ", paste(names_vallab[duplicates], collapse = ",")))
    # }
    if (add) value = combine_labels(value,val_lab(x))
    if (length(value)==0) value=NULL else value=sort(value)
    attr(x,"labels")=value
    class(x)=union("labelled",class(x))
    x
}

#' @export
set_val_lab.data.frame = function(x,value, add = FALSE){
    for (each in seq_along(x)) x[[each]] = set_val_lab(x[[each]], value, add = add)
    x
}

#' @export
set_val_lab.list = function(x,value, add = FALSE){
    for (each in seq_along(x)) x[[each]] = set_val_lab(x[[each]], value, add = add)
    x
}

#######

#' @export
#' @rdname val_lab 
"add_val_lab<-"=function(x,value){
    set_val_lab(x,value, add = TRUE)
}

######



#' @export
#' @rdname val_lab
unvl=function(x){
    set_val_lab(x,NULL)
}


#' @export
#' @rdname val_lab
make_labels=function(text, code_position=c("left","right")){
    split="\n"
    if (length(text)>1) text = paste(text,collapse=split) 
    res = unlist(strsplit(text,split=split))
    res = res[!is.na(res)]
    res = gsub("^([\\s\\t]+)|([\\s\\t]+)$","",res,perl = TRUE)
    res = res[res!=""]
    code_position = match.arg(code_position)
    if (code_position == "left") {
        pattern = "^(-*)([\\d\\.]+)([\\.\\s\\t]*)(.*?)$"
        code_pattern = "\\1\\2"
        label_pattern = "\\4"
    } else {
        pattern = "^(.*?)([\\s\\t]*)(-*)([\\d\\.]+)$"
        code_pattern = "\\3\\4"
        label_pattern = "\\1"
        
    }
    stopif(!all(grepl(pattern, res,perl=TRUE)), "Incorrect pattern for labels:\n", paste(res[!grepl(pattern, res,perl=TRUE)], collapse = "\n"))
    code=as.numeric(gsub(pattern,code_pattern,res,perl=TRUE))
    #     if (!any(abs(floor(code)-code)>0)) code = as.integer(code)
    lab=gsub(pattern,label_pattern,res,perl=TRUE)
    code = code[!(lab %in% "")]
    lab = lab[!(lab %in% "")]
    if(length(lab)>0){
        structure(code,names=lab)
    } else {
        NULL
    }   
}

#' @export
#' @rdname val_lab
ml_left = function(text) make_labels(text = text, code_position = "left")
#' @export
#' @rdname val_lab
ml_right = function(text) make_labels(text = text, code_position = "right")

#' Drop variable label and value labels
#' 
#' \code{unlab} returns variable x without variable labels and value labels
#' 
#' @param x Variable(s). Vector/data.frame/list.
#' @return \code{unlab} returns original variable x without variable label, value labels and class.
#' @seealso \code{\link{unvr}} \code{\link{unvl}}
#' @export 
#' @examples
#' raw_var = rep(1:2,5)
#' var_with_lab = set_var_lab(raw_var,"Income")
#' val_lab(var_with_lab) = c("Low"=1,"High"=2)
#' identical(raw_var,unlab(var_with_lab)) # should be TRUE
unlab=function(x){
    UseMethod("unlab")
}

#' @export
unlab.default=function(x){
    var_lab(x) = NULL
    val_lab(x) = NULL
    class(x) = setdiff(class(x),"labelled")
    x
}

#' @export
unlab.data.frame=function(x){
    for (each in seq_along(x)) x[[each]] = unlab(x[[each]])
    x
}

#' @export
unlab.list=function(x){
    for (each in seq_along(x)) x[[each]] = unlab(x[[each]])
    x
}

########



combine_labels = function(...){
    args = list(...)
    new_lab = Reduce(`%u%`, args)
    if(length(new_lab)>0) sort(new_lab) else new_lab
}

labelled_and_unlabelled = function(uniqs,vallab){
    if (length(uniqs)>0) {
        uniqs=uniqs[!is.na(uniqs)]
        names(uniqs) = uniqs
    }    
    vallab = vallab %u% uniqs
    if (length(vallab)>1) sort(vallab) else vallab
}

as.labelled = function(x){
    UseMethod("as.labelled")
}

as.labelled.default = function(x){
    x
}

as.labelled.factor = function(x){
    labels = seq_along(levels(x))
    names(labels) = levels(x)
    x = as.numeric(x)
    val_lab(x) = labels
    x
    
}
