
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU Library General
# Public License along with this library; if not, write to the
# Free Foundation, Inc., 59 Temple Place, Suite 330, Boston,
# MA  02111-1307  USA

# Copyrights (C)
# for this R-port:
#   1999 - 2008, Diethelm Wuertz, Rmetrics Foundation, GPL
#   Diethelm Wuertz <wuertz@itp.phys.ethz.ch>
#   www.rmetrics.org
# for the code accessed (or partly included) from other R-ports:
#   see R's copyright and license files
# for the code accessed (or partly included) from contributed R-ports
# and other sources
#   see Rmetrics's copyright file


################################################################################
# FUNCTION:            GENERALIZED DISTRIBUTION:
#  nigFit               Fits parameters of a normal inverse Gaussian density
#  .nigFit.mle          max Log-likelihood Estimation
#  .nigFit.gmm          gemeralized method of moments estimation
#  .nigFit.mps          maximum product spacings estimation
################################################################################


nigFit <- function(x, alpha = 1, beta = 0, delta = 1, mu = 0, 
    method = c("mle", "gmm", "mps"), scale = TRUE, doplot = TRUE, 
    span = "auto", trace = TRUE, title = NULL, description = NULL, ...)
{
    # A function implemented by Diethelm Wuertz
    
    # FUNCTION: 
    
    # MLE:
    fit = .nigFit.mle(x = x, alpha = alpha, beta = beta, delta = delta, 
        mu = mu , scale = scale, doplot = doplot, span = span, 
        trace = trace, title = title, description = description, ...)
        
    # Return Value:
    fit
}


# ------------------------------------------------------------------------------


.nigFit.mle <-
    function(x, alpha = 1, beta = 0, delta = 1, mu = 0, 
    scale = TRUE, doplot = TRUE, add = FALSE, span = "auto", trace = TRUE,
    title = NULL, description = NULL, ...)
{
    # A function implemented by Diethelm Wuertz

    # Description:
    #   Fits parameters of a NIG using maximum log-likelihood  

    # Example:
    #   set.seed(4711); x = rnig(500); mle = .nigFit.mle(x)@fit$estimate; mle
    
    # FUNCTION:

    # Transform:
    x.orig = x
    x = as.vector(x)
    if (scale) {
        SD = sd(x)
        x = x / SD }

    # Settings:
    CALL = match.call()

    # Parameter Estimation:
    llh = function(x, y = x, trace) {
        if (abs(x[2]) >= x[1]) return(1e99)
        f = -sum(dnig(y, x[1], x[2], x[3], x[4], log = TRUE))
        # Print Iteration Path:
        if (trace) {
            cat("\n Objective Function Value:  ", -f)
            cat("\n Parameter Estimates:       ", x[1], x[2], x[3], x[4], "\n")
        }
        f }
    eps = 1e-10
    BIG = 1000
    r = nlminb(start = c(alpha, beta, delta, mu), objective = llh,
        lower = c(eps, -BIG, eps, -BIG), upper = BIG, y = x, trace = trace)
    names(r$par) <- c("alpha", "beta", "delta", "mu")

    # Add Title and Description:
    if (is.null(title)) title = "Normal Inverse Gaussian Parameter Estimation"
    if (is.null(description)) description = description()

    # Rescale Result:
    if (scale) {
        r$par = r$par / c(SD, SD, 1/SD, 1/SD)
        r$objective = llh(r$par, y = as.vector(x.orig), trace = trace)
    }
    fit = list(estimate = r$par, minimum = -r$objective, code = r$convergence)

    # Optional Plot:
    if (doplot) {
        x = as.vector(x.orig)
        if (span == "auto") span = seq(min(x), max(x), length = 501)
        z = density(x, n = 100, ...)
        x = z$x[z$y > 0]
        y = z$y[z$y > 0]
        y.points = dnig(span, r$par[1], r$par[2], r$par[3], r$par[4])
        ylim = log(c(min(y.points), max(y.points)))
        if (add) {
            lines(x = span, y = log(y.points), col = "steelblue")
        } else {
            plot(x, log(y), xlim = c(span[1], span[length(span)]),
                ylim = ylim, type = "p", xlab = "x", ylab = "log f(x)", ...)
            title("NIG Parameter Estimation")
            lines(x = span, y = log(y.points), col = "steelblue")
        }
    }

    # Return Value:
    new("fDISTFIT",
        call = as.call(CALL),
        model = "Normal Inverse Gaussian Distribution",
        data = as.data.frame(x.orig),
        fit = fit,
        title = as.character(title),
        description = description() )
}


# ------------------------------------------------------------------------------


.nigFit.gmm <-
    function(x, 
    scale = TRUE, doplot = TRUE, add = FALSE, span = "auto", trace = TRUE,
    title = NULL, description = NULL, ...)
{
    # A function implemented by Diethelm Wuertz

    # Description:
    #   Fits parameters of a NIG using GMM estimator  

    # Example:
    #   set.seed(4711); x = rnig(500); gmm = .nigFit.gmm(x)@fit$estimate; gmm
    
    # FUNCTION:

    # Transform:
    x.orig = x
    x = as.vector(x)
    if (scale) {
        SD = sd(x)
        x = x / SD }

    # Settings:
    CALL = match.call()

    # Parameter Estimation:
    nig.gmm <- function(Theta, x) {
        # Parameters:
        alpha = Theta[1]
        beta  = Theta[2]
        delta = Theta[3]
        mu    = Theta[4] 
        names(Theta) <- c("alpha", "beta", "delta", "mu")
        # Trace:
        if (TRUE) print(Theta)
        # Moments:
        m1 <- x   - .ghMuMoments(1, alpha, beta, delta, mu, lambda = -0.5)
        m2 <- x^2 - .ghMuMoments(2, alpha, beta, delta, mu, lambda = -0.5)
        m3 <- x^3 - .ghMuMoments(3, alpha, beta, delta, mu, lambda = -0.5)
        m4 <- x^4 - .ghMuMoments(4, alpha, beta, delta, mu, lambda = -0.5)
        # Result:
        f <- cbind(m1, m2, m3, m4)
        return(f)
    }
    r <- .gmm(g = nig.gmm, x = x, t0 = c(1, 0, 1, 0)) 
    names(r$par) <- c("alpha", "beta", "delta", "mu")
    
    # Add Title and Description:
    if (is.null(title)) title = "Normal Inverse Gaussian Parameter Estimation"
    if (is.null(description)) description = description()

    # Rescale Result:
    if (scale) {
        r$par = r$par / c(SD, SD, 1/SD, 1/SD)
        r$objective = NA
    }
    fit = list(estimate = r$par)

    # Optional Plot:
    if (doplot) {
        x = as.vector(x.orig)
        if (span == "auto") span = seq(min(x), max(x), length = 51)
        z = density(x, n = 100, ...)
        x = z$x[z$y > 0]
        y = z$y[z$y > 0]
        y.points = dnig(span, r$par[1], r$par[2], r$par[3], r$par[4])
        ylim = log(c(min(y.points), max(y.points)))
        if (add) {
            lines(x = span, y = log(y.points), col = "steelblue")
        } else {
            plot(x, log(y), xlim = c(span[1], span[length(span)]),
                ylim = ylim, type = "p", xlab = "x", ylab = "log f(x)", ...)
            title("NIG Parameter Estimation")
            lines(x = span, y = log(y.points), col = "steelblue")
        }
    }

    # Return Value:
    new("fDISTFIT",
        call = as.call(CALL),
        model = "Normal Inverse Gaussian Distribution",
        data = as.data.frame(x.orig),
        fit = fit,
        title = as.character(title),
        description = description() )
}


# ------------------------------------------------------------------------------


.nigFit.mps <-
    function(x, alpha = 1, beta = 0, delta = 1, mu = 0, 
    scale = TRUE, doplot = TRUE, add = FALSE, span = "auto", trace = TRUE,
    title = NULL, description = NULL, ...)
{
    # A function implemented by Diethelm Wuertz

    # Description:
    #   Fits parameters of a NIG using maximum product spacings  

    # Example:
    #   set.seed(4711); x = rnig(500); mps = .nigFit.mps(x)@fit$estimate; mps
    
    # FUNCTION:

    # Transform:
    x.orig = x
    x = as.vector(x)
    if (scale) {
        SD = sd(x)
        x = x / SD }

    # Settings:
    CALL = match.call()

    # Parameter Estimation:
    mps = function(x, y = x, trace) {
        if (abs(x[2]) >= x[1]) return(1e99)
        
        DH = diff(c(0, na.omit(.pnigC(sort(y), x[1], x[2], x[3], x[4])), 1))
        f = -mean(log(DH[DH > 0]))*length(y)

        # Print Iteration Path:
        if (trace) {
            cat("\n Objective Function Value:  ", -f)
            cat("\n Parameter Estimates:       ", x[1], x[2], x[3], x[4], "\n")
        }
        f }
    eps = 1e-10
    BIG = 1000
    r = nlminb(start = c(alpha, beta, delta, mu), objective = mps,
        lower = c(eps, -BIG, eps, -BIG), upper = BIG, y = x, trace = trace)
    names(r$par) <- c("alpha", "beta", "delta", "mu")

    # Add Title and Description:
    if (is.null(title)) title = "NIG MPS Parameter Estimation"
    if (is.null(description)) description = description()

    # Result:
    if (scale) {
        r$par = r$par / c(SD, SD, 1/SD, 1/SD)
        r$objective = mps(r$par, y = as.vector(x.orig), trace = trace)
    }
    fit = list(estimate = r$par, minimum = -r$objective, code = r$convergence)

    # Optional Plot:
    if (doplot) {
        x = as.vector(x.orig)
        if (span == "auto") span = seq(min(x), max(x), length = 501)
        z = density(x, n = 100, ...)
        x = z$x[z$y > 0]
        y = z$y[z$y > 0]
        y.points = dnig(span, r$par[1], r$par[2], r$par[3], r$par[4])
        ylim = log(c(min(y.points), max(y.points)))
        if (add) {
            lines(x = span, y = log(y.points), col = "steelblue")
        } else {
            plot(x, log(y), xlim = c(span[1], span[length(span)]),
                ylim = ylim, type = "p", xlab = "x", ylab = "log f(x)", ...)
            title("NIG Parameter Estimation")
            lines(x = span, y = log(y.points), col = "steelblue")
        }
    }

    # Return Value:
    new("fDISTFIT",
        call = as.call(CALL),
        model = "Normal Inverse Gaussian Distribution",
        data = as.data.frame(x.orig),
        fit = fit,
        title = as.character(title),
        description = description() )
}


################################################################################

   