#' Read a Teknologic Acoustic Receiver File
#'
#' This function takes a raw acoustic detection file generated by a Teknologic
#' JSATS receiver and processes it into a dataframe which can be used by
#' the filtering functions in this package. This is called within read_jsats().
#'
#' @param path the path to the folder containing the desired file
#' @param file the path of the desired file
#' @param timezone the Olsen Named time zone, default is "America/Los_Angeles"
#' @returns A dataframe converting the raw detection data into rows of detections
#' @export
#' @examples
#' # Read in a Teknologic file
#' path = system.file("extdata/", package = "filteRjsats")
#' read_tekno(path = path, file = "2015-6007211361217.SUM",
#' timezone = "America/Los_Angeles")
read_tekno <- function(path, file, timezone="America/Los_Angeles"){

  TEK <- read.csv(file.path(path,file),
                  colClasses = c(character(),character(),character(),
                                 character(),numeric(),numeric(),
                                 numeric(), numeric(), numeric(),
                                 numeric(), numeric(), numeric(),
                                 numeric(), logical()),
                  skip = 8,
                  header = TRUE)

  TEK$Tag_Hex = as.character(substr(TEK$TagCode,4,7)) #Extract hex format of tag code
  TEK <- TEK[TEK$valid == 1,] # filter out "invalid" detections X = 0
  TEK$ReceiverSN <- gsub("-","",TEK$Serial.Number) #Retrieve Serial Number
  TEK$Tag_Decimal = broman::hex2dec(TEK$Tag_Hex) # Convert hex to dec format
  TEK$DateTime_Local = lubridate::mdy_hms(TEK$Date.Time, tz = timezone)
  TEK$Volt = TEK$vBatt
  TEK$Freq = 416.666 + (TEK$Freq/1000)
  TEK$Thres = TEK$Thresh
  TEK$SigStr = TEK$snr
  TEK$Make = rep("Tekno", length(TEK$DateTime_Local))
  TEK <- dplyr::select(.data =  TEK, ReceiverSN, Make, DateTime_Local,
                       Tag_Decimal, Tag_Hex, Volt, SigStr, Freq, Thres)
  TEK
}
