% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/first.R
\name{first}
\alias{first}
\title{Factor Importance Ranking and Selection using Total (Sobol') indices}
\usage{
first(
  X,
  y,
  n.knn = 2,
  n.mc = nrow(X),
  mc.option = "random",
  rescale = TRUE,
  sparse = FALSE,
  parl = NULL,
  verbose = FALSE
)
}
\arguments{
\item{X}{a matrix or data frame containing the inputs.}

\item{y}{a vector containing the outputs.}

\item{n.knn}{number of nearest-neighbors for the inner loop conditional variance estimation. See \code{\link{totalsobol.knn}} for details.}

\item{n.mc}{number of Monte Carlo samples for the outer loop expectation estimation. See \code{\link{totalsobol.knn}} for details.}

\item{mc.option}{option for the Monte Carlo samples. The options are random(default)/twinning. See \code{\link{totalsobol.knn}} for details.}

\item{rescale}{a logical indicating whether continuous inputs is rescaled to zero mean unit variance.}

\item{sparse}{a logical indicating whether to run the fast version of forward selection that integrates with the effect sparsity principle.}

\item{parl}{number of cores on which to parallelize the computation. If \code{NULL}, then no parallelization is done.}

\item{verbose}{a logical indicating whether to display intermediate results of running \code{first}.}
}
\value{
A numeric vector of the factor importance, with zero indicating that the factor is not important to the prediction of the response.
}
\description{
\code{first} implements the \emph{\strong{model-independent}} factor importance and selection procedure proposed in Huang and Joseph (2024).
The importance measure is based on total Sobol' indices from global sensitivity analysis.
Factor importance computation and selection are performed directly from the noisy data.
Parallel computations are available to accelerate the estimation.
For categorical data inputs, please convert them to factor type before calling the function.
}
\details{
\code{first} provides factor importance ranking and selection directly from scattered data without any model fitting.
Factor importance is based on total Sobol' indices (Sobol, 1993),
and its connection to the approximation error introduced by excluding the factor of interest is shown in Huang and Joseph (2024).
The estimation of the total Sobol' indices is carried out using \code{\link{totalsobol.knn}}.
Integrating it with forward selection and backward elimination allows for factor selection.
Please see Huang and Joseph (2024) for the details of the algorithm.

\code{n.knn=2} nearest-neighbors is recommended for integer/numeric output, and \code{n.knn=3} is suggested for binary output.
For numeric inputs, it is recommended to standardize them via setting the argument \code{rescale=TRUE}.
Categorical inputs are transformed via one-hot encoding for the nearest-neighbor search.
To speed up the nearest-neighbor search, k-d tree from the \pkg{FNN} package is used.
Also, parallel computation is also supported via the \pkg{parallel} package.

For very large dimensional problems, we also provide the fast forward selection method (\code{sparse=TRUE})
that is based on the effect sparsity principle, but it sacrifices some accuracy.
See Appendix of Huang and Joseph (2024) for more details.
For large datasets, we support the use of subsamples for further acceleration.
Use argument \code{n.mc} to specify the number of subsamples.
Two options (argument \code{mc.option}) are available for finding the subsamples: random and twinning (Vakayil and Joseph, 2022).
Twinning is able to find subsamples that better represent the big data, i.e.,
providing a more accurate estimation, but at a slightly higher computational cost.
For more details, please see the \pkg{twinning} package.
}
\examples{
ishigami <- function(x) {
  x <- -pi + 2*pi*x
  y <- sin(x[1]) + 7*sin(x[2])^2 + 0.1*x[3]^4*sin(x[1])
  return (y)
} 
set.seed(123)
n <- 1000
p <- 6
X <- matrix(runif(n*p), ncol=p)
y <- apply(X,1,ishigami) + rnorm(n)
imp <- first(X, y, n.knn=2, rescale=FALSE, verbose=TRUE)
print(round(imp,3)) # Only first 3 factors are important

}
\references{
Huang, C., & Joseph, V. R. (2024). Factor Importance Ranking and Selection using Total Indices. arXiv preprint arXiv:2401.00800.

Sobol', I. Y. M. (1990). On sensitivity estimation for nonlinear mathematical models. Matematicheskoe modelirovanie, 2(1), 112-118.

Vakayil, A., & Joseph, V. R. (2022). Data twinning. Statistical Analysis and Data Mining: The ASA Data Science Journal, 15(5), 598-610.
}
\author{
Chaofan Huang \email{chaofan.huang@gatech.edu} and V. Roshan Joseph \email{roshan@gatech.edu}
}
