/*  Routines for manipulating B-splines.  These are intended for use with
 *  S or S-PLUS or R.
 *
 *     Copyright (C) 2015 Michel Grzebyk.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * These functions are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, a copy is available at
 *  http://www.r-project.org/Licenses/
 *
 * The routines are loosely based on the pseudo-code in Schumacher (Wiley,
 * 1981) and the CMLIB library DBSPLINES.
 */

#include <R.h>
#include <Rinternals.h>

#ifdef ENABLE_NLS
#include <libintl.h>
#define _(String) dgettext ("splines", String)
#else
#define _(String) (String)
#endif




SEXP predict_linex_spline_basis(SEXP knots, SEXP order, SEXP Matrices, SEXP linexinf, SEXP linexsup, SEXP orderextrapol,
                                SEXP intercept, SEXP xvals, SEXP outerok)
{
/* predict ie evaluate linear combination of the  B-spline basis functions at xvals, 
using matrices generated by packag::orthogonalsplinbasis */
/* 
 knots : vector of ordered unreplicated INTERIOR knots 
Matrices : a vectorized array of dim order X nbases X number_of_intervales(knots) 
  where nbases is the number of bases of the non integrated, non derived splines
the values in Matrices are scaled by the coef of the spline bases 
order : order of the splines (see package orthogonalsplinbasis
intercept : wehtehr first basis is included
xvals : vector values at which bases are computed
linexinf, linexsup : matrix of dim order X nbases for the extrapolation
orderextrapol : order of extrapolation outside of the knots. coef are 0.0 in linexinf, linexsup when line number > orderextrapol.
theoritically, order > (>=?) orderextrapol
 */
	R_len_t i, k, nknots, theorder, theorderextrapol, nbases, nintervals, nx, rorderlinexinf, rorderlinexsup;
	R_len_t theinterval, firstbasis, mfl;
	double *rknots, *rMatrices, *rAddMatrices, *rlinexinf, *rlinexsup, *raddlinexinf, *raddlinexsup, *rxvals, *rpredict;
	SEXP predict;
	SEXP dims;
	double temppredict, U, u, rlinexextrainf, rlinexextrasup;
	
	
	PROTECT(knots = coerceVector(knots, REALSXP));
	PROTECT(order = coerceVector(order, INTSXP));
	PROTECT(intercept = coerceVector(intercept, INTSXP));
	PROTECT(Matrices = coerceVector(Matrices, REALSXP));
	PROTECT(linexinf = coerceVector(linexinf, REALSXP));
	PROTECT(linexsup = coerceVector(linexsup, REALSXP));
	PROTECT(orderextrapol = coerceVector(orderextrapol, INTSXP));
	PROTECT(xvals = coerceVector(xvals, REALSXP));
	PROTECT(outerok = coerceVector(outerok, LGLSXP));


	rknots = REAL(knots); 
	nknots = length(knots);
	theorder = INTEGER(order)[0];
	theorderextrapol = INTEGER(orderextrapol)[0];

	dims = getAttrib(Matrices, R_DimSymbol);
	if( LENGTH(dims) < 3 ){
		error("'Matrices' must be an array with 3 dim");   
	}
	nbases = INTEGER(dims)[1];
	nintervals = INTEGER(dims)[2];
	
	rxvals = REAL(xvals); 
	nx = length(xvals);
		
	
	firstbasis = (INTEGER(intercept)[0]==0);
	rMatrices = REAL(Matrices);
	rlinexinf = REAL(linexinf);
	rlinexsup = REAL(linexsup);
		
	PROTECT(predict = allocVector(REALSXP, nx));
	rpredict = REAL(predict);

/*
	U = (double *) R_alloc( theorder, sizeof(double));
*/
/* outer values are linearly extrapolated 
	oo = asLogical(outerok);
	if(oo == NA_LOGICAL) {
		error("'outer.ok' must be TRUE or FALSE");    
	} else  if (oo) {
		outer_val = 0.0;
	} else {
		outer_val = R_NaN;
	}
*/

/* first add cols of each matrix */
	rAddMatrices = (double *) R_alloc( theorder * nintervals, sizeof(double));
	for (i = 0; i < nintervals; i++) {
		for (int j = 0; j < theorder ; j++) {
			rAddMatrices[theorder*i + j] = 0.0;
			for (k = firstbasis; k < nbases; k++) {
				rAddMatrices[theorder*i + j] += rMatrices[theorder*nbases*i+ theorder*k + j];
			}
		}
	}

	raddlinexinf = (double *) R_alloc( theorder , sizeof(double));
	rorderlinexinf = 0;
	for (int j = 0; j <= theorderextrapol ; j++) {
		raddlinexinf[j] = 0.0;
		for (k = firstbasis; k < nbases; k++) {
			raddlinexinf[j] += rlinexinf[theorder*k + j];
		}
		if( raddlinexinf[j] != 0.0 ){
			rorderlinexinf = j;
		}
	}
	if (raddlinexinf[rorderlinexinf]<0){
			rlinexextrainf = R_PosInf;
	} else if (raddlinexinf[rorderlinexinf]>0){
		rlinexextrainf = R_NegInf;
	} else {
		rlinexextrainf = 0.0;
	}

	raddlinexsup = (double *) R_alloc( theorder , sizeof(double));
	rorderlinexsup = 0;
	for (int j = 0; j <= theorderextrapol ; j++) {
		raddlinexsup[j] = 0.0;
		for (k = firstbasis; k < nbases; k++) {
			raddlinexsup[j] += rlinexsup[theorder*k + j];
		}
		if( raddlinexsup[j] != 0.0 ){
			rorderlinexsup = j;
		}
	}
	if (raddlinexsup[rorderlinexsup]<0){
		rlinexextrasup = R_NegInf;
	} else if (raddlinexsup[rorderlinexsup]>0){
		rlinexextrasup = R_PosInf;
	} else {
		rlinexextrasup = 0.0;
	}


/*
	U[0]=1.0;
*/
	for(i = 0; i < nx; i++) {
		if (R_finite(rxvals[i])) {
		    theinterval= 1;
		    mfl = 0;
/* find the interval within the range of all the knots (which include boundaries) 
   of rxvals[i], rightmost_close=TRUE, all_inside = FALSE */ 
		    theinterval = findInterval(rknots, nknots, rxvals[i], 1, 0 , theinterval, &mfl );
		    if ((theinterval == 0) & (theorderextrapol >= 0) ) { 
/* before first boundary knot, polynomial extrapolation */
			    u = (rxvals[i]-rknots[0]);
			    temppredict = raddlinexinf[0];
			    U = 1.0;
			    for (int j = 1; j <= theorderextrapol ; j++) {
				    U *= u;
				    temppredict += U * raddlinexinf[j];
			    }
			    rpredict[i] = temppredict;
		    } else if ( (theinterval == nknots) & (theorderextrapol >= 0)  ) {
/* after last boundary knot, linear */
			    u = (rxvals[i]-rknots[nknots-1]);
			    temppredict = raddlinexsup[0];
			    U = 1.0;
			    for (int j = 1; j <= theorderextrapol ; j++) {
				    U *= u;
				    temppredict += U * raddlinexsup[j];
				}
			    rpredict[i] = temppredict;
		    } 
		    else {
			    if( theinterval == nknots -1) {
				    /* xx[i] is the rightmost boundary knot */
				    theinterval = nknots - theorder;
			    }
			    u = (rxvals[i] - rknots[theinterval-1])/(rknots[theinterval]-rknots[theinterval-1]);
			    /* the usefull matrix is the (theinterval - theorder +1)th matrix of Matrices */
			    theinterval = theinterval - theorder;
			    temppredict = rAddMatrices[theorder*theinterval];
			    U = 1.0;
			    for (int j = 1; j < theorder ; j++) {
				    U *= u;
				    temppredict += U * rAddMatrices[theorder*theinterval + j];
			    }
			    rpredict[i] = temppredict;
		    }
		} else if (ISNAN(rxvals[i])) {
			rpredict[i] = R_NaN;
		} else if (rxvals[i] == R_NegInf) {
			rpredict[i] = rlinexextrainf;
		} else if (rxvals[i] == R_PosInf) {
			rpredict[i] = rlinexextrasup;
		}
		
	}
	
	UNPROTECT(10);
	return(predict);
}
