#' Create the DEGURBA grid cell classification of urban clusters
#'
#' @description
#' The Degree of Urbanisation identifies urban clusters as clusters of continuous grid cells (based on queen contiguity) with a minimum density of 300 inhabitants per km², and a minimum total population of 5000 inhabitants.
#'
#' For more information about the Degree of Urbanisation methodology, see the [methodological manual](https://ec.europa.eu/eurostat/statistics-explained/index.php?title=Applying_the_degree_of_urbanisation_manual), [GHSL Data Package 2022](https://ghsl.jrc.ec.europa.eu/documents/GHSL_Data_Package_2022.pdf) and [GHSL Data Package 2023](https://ghsl.jrc.ec.europa.eu/documents/GHSL_Data_Package_2023.pdf).
#'
#' The arguments of the function allow to adapt the standard specifications in the Degree of Urbanisation in order to construct an alternative version of the grid classification.
#' @param data path to the directory with the data, or named list with the data as returned by function [DoU_preprocess_grid()]
#' @param classification SpatRaster. A grid with the classification of urban centres to which the classification of urban clusters will be added. Note that the grid will be adapted in-place.
#' @param density_threshold numeric. Minimum population density per permanent land of a cell required to belong to an urban cluster
#' @param size_threshold numeric. Minimum total population size required for an urban cluster
#' @param contiguity_rule integer. Which cells are considered adjacent: `4` for rooks case (horizontal and vertical neighbours) or `8` for queens case (horizontal, vertical and diagonal neighbours)
#' @param smooth_pop logical. Whether to smooth the population grid before delineating urban clusters. If `TRUE`, the population grid will be smoothed with a moving average of window size `smooth_pop_window`.
#' @param smooth_pop_window integer. Size of the moving window used to smooth the population grid before delineating urban clusters. Ignored when `smooth_pop` is `FALSE`.
#' @param value integer. Value assigned to urban clusters in the resulting grid
#' @return SpatRaster with the grid cell classification of urban clusters
#' @examples
#' data_belgium <- DoU_load_grid_data_belgium()
#' classification <- DoU_classify_grid_urban_centres(data_belgium)
#' classification <- DoU_classify_grid_urban_clusters(data_belgium, classification = classification)
#' DoU_plot_grid(classification)
#' @export
DoU_classify_grid_urban_clusters <- function(
  data,
  classification,
  density_threshold = 300,
  size_threshold = 5000,
  contiguity_rule = 8,
  smooth_pop = FALSE,
  smooth_pop_window = 5,
  value = 2
) {
  ##### CHECK IF PAREMETERS ARE VALID

  # read data
  if (is.character(data)) {
    data <- DoU_preprocess_grid(data)
  }

  # check if the data is valid
  if (
    !all(
      c("pop", "land", "pop_per_land", "built_per_land", "built") %in%
        names(data)
    )
  ) {
    stop(
      "Invalid argument: The data should contain a named list of pop, land, built, pop_per_land and built_per_land as generated by the function DoU_preprocess_grid."
    )
  }

  # check other arguments
  if (!smooth_pop %in% c(TRUE, FALSE)) {
    stop(paste(
      "Invalid argument:",
      smooth_pop,
      "is not a valid parameter for smooth_pop"
    ))
  }

  if (!is.numeric(smooth_pop_window)) {
    stop(paste(
      "Invalid argument:",
      smooth_pop_window,
      "is not a valid parameter for smooth_pop_window"
    ))
  }

  ##### CONSTRUCT URBAN CLUSTERS

  # smooth the population grid if smooth_pop = TRUE
  if (smooth_pop) {
    pop <- terra::focal(
      data$pop,
      w = smooth_pop_window,
      fun = "mean",
      na.rm = TRUE
    )
    pop_per_land <- terra::focal(
      data$pop_per_land,
      w = smooth_pop_window,
      fun = "mean",
      na.rm = TRUE
    )
  } else {
    pop <- data$pop
    pop_per_land <- data$pop_per_land
  }

  # get dense clusters
  urbanclusters <- flexurba::get_clusters(
    xden = pop_per_land,
    minden = density_threshold,
    xsiz = pop,
    minsiz = size_threshold,
    directions = contiguity_rule
  )

  # add urban clusters to the classification and return the results
  terra::set.values(
    classification,
    which((is.na(classification[])) & (!is.na(urbanclusters[]))),
    value
  )
  names(classification) <- c("layer")

  return(classification)
}

#' Create the DEGURBA grid cell classification of urban clusters
#'
#' @description
#' `r lifecycle::badge("deprecated")`
#'
#' `classify_grid_urban_clusters()` has been renamed to `DoU_classify_grid_urban_clusters()` to create a more consistent API and to better indicate that this function is specifically designed to classify urban clusters in the context of the DEGURBA classification.
#' @param data path to the directory with the data, or named list with the data as returned by function [DoU_preprocess_grid()]
#' @param classification SpatRaster. A grid with the classification of urban centres to which the classification of urban clusters will be added. Note that the grid will be adapted in-place.
#' @param density_threshold numeric. Minimum population density per permanent land of a cell required to belong to an urban cluster
#' @param size_threshold numeric. Minimum total population size required for an urban cluster
#' @param contiguity_rule integer. Which cells are considered adjacent: `4` for rooks case (horizontal and vertical neighbours) or `8` for queens case (horizontal, vertical and diagonal neighbours)
#' @param smooth_pop logical. Whether to smooth the population grid before delineating urban clusters. If `TRUE`, the population grid will be smoothed with a moving average of window size `smooth_pop_window`.
#' @param smooth_pop_window integer. Size of the moving window used to smooth the population grid before delineating urban clusters. Ignored when `smooth_pop` is `FALSE`.
#' @param value integer. Value assigned to urban clusters in the resulting grid
#' @return SpatRaster with the grid cell classification of urban clusters
#' @keywords internal
#' @export
classify_grid_urban_clusters <- function(
  data,
  classification,
  density_threshold = 300,
  size_threshold = 5000,
  contiguity_rule = 8,
  smooth_pop = FALSE,
  smooth_pop_window = 5,
  value = 2
) {
  return(DoU_classify_grid_urban_clusters(
    data,
    classification,
    density_threshold,
    size_threshold,
    contiguity_rule,
    smooth_pop,
    smooth_pop_window,
    value
  ))
}
