% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gdalraster_proc.R
\name{calc}
\alias{calc}
\title{Raster calculation}
\usage{
calc(
  expr,
  rasterfiles,
  bands = NULL,
  var.names = NULL,
  dstfile = tempfile("rastcalc", fileext = ".tif"),
  fmt = NULL,
  dtName = "Int16",
  out_band = NULL,
  options = NULL,
  nodata_value = NULL,
  setRasterNodataValue = FALSE,
  usePixelLonLat = NULL,
  write_mode = "safe",
  quiet = FALSE
)
}
\arguments{
\item{expr}{An R expression as a character string (e.g., \code{"A + B"}).}

\item{rasterfiles}{Character vector of source raster filenames.}

\item{bands}{Integer vector of band numbers to use for each raster layer.}

\item{var.names}{Character vector of variable names to use for each raster
layer.}

\item{dstfile}{Character filename of output raster.}

\item{fmt}{Output raster format name (e.g., "GTiff" or "HFA"). Will attempt
to guess from the output filename if not specified.}

\item{dtName}{Character name of output data type (e.g., Byte, Int16,
UInt16, Int32, UInt32, Float32).}

\item{out_band}{Integer band number(s) in \code{dstfile} for writing output.
Defaults to \code{1}. Multiband output is supported as of gdalraster 1.11.0,
in which case \code{out_band} would be a vector of band numbers.}

\item{options}{Optional list of format-specific creation options in a
vector of "NAME=VALUE" pairs
(e.g., \code{options = c("COMPRESS=LZW")} to set LZW compression
during creation of a GTiff file).}

\item{nodata_value}{Numeric value to assign if \code{expr} returns NA.}

\item{setRasterNodataValue}{Logical. \code{TRUE} will attempt to set the raster
format nodata value to \code{nodata_value}, or \code{FALSE} not to set a raster
nodata value.}

\item{usePixelLonLat}{This argument is deprecated and will be removed in a
future version. Variable names \code{pixelLon} and \code{pixelLat} can be used in
\code{expr}, and the pixel x/y coordinates will be inverse projected to
longitude/latitude (adds computation time).}

\item{write_mode}{Character. Name of the file write mode for output.
One of:
\itemize{
\item \code{safe} - execution stops if \code{dstfile} already exists (no output written)
\item \code{overwrite} - if \code{dstfile} exists if will be overwritten with a new file
\item \code{update} - if \code{dstfile} exists, will attempt to open in update mode
and write output to \code{out_band}
}}

\item{quiet}{Logical scalar. If \code{TRUE}, a progress bar will not be
displayed. Defaults to \code{FALSE}.}
}
\value{
Returns the output filename invisibly.
}
\description{
\code{calc()} evaluates an R expression for each pixel in a raster layer or
stack of layers. Each layer is defined by a raster filename, band number,
and a variable name to use in the R expression. If not specified, band
defaults to 1 for each input raster.
Variable names default to \code{LETTERS} if not specified
(\code{A} (layer 1), \code{B} (layer 2), ...).
All of the input layers must have the same extent and cell size.
The projection will be read from the first raster in the list
of inputs.
Individual pixel coordinates are also available as variables in the
R expression, as either x/y in the raster projected coordinate system or
inverse projected longitude/latitude.
Multiband output is supported as of gdalraster 1.11.0.
}
\details{
The variables in \code{expr} are vectors of length raster xsize
(row vectors of the input raster layer(s)).
The expression should return a vector also of length raster xsize
(an output row).
Four special variable names are available in \code{expr}:
\code{pixelX} and \code{pixelY} provide pixel center coordinates in projection units.
\code{pixelLon} and \code{pixelLat} can also be used, in which case the pixel x/y
coordinates will be inverse projected to longitude/latitude
(in the same geographic coordinate system used by the input projection,
which is read from the first input raster). Note that inverse projection
adds computation time.

To refer to specific bands in a multi-band input file, repeat the filename in
\code{rasterfiles} and specify corresponding band numbers in \code{bands}, along with
optional variable names in \code{var.names}, for example,
\preformatted{
rasterfiles = c("multiband.tif", "multiband.tif")
bands = c(4, 5)
var.names = c("B4", "B5")
}

Output will be written to \code{dstfile}. To update a file that already
exists, set \code{write_mode = "update"} and set \code{out_band} to an existing
band number(s) in \code{dstfile} (new bands cannot be created in \code{dstfile}).

To write multiband output, \code{expr} must return a vector of values
interleaved by band. This is equivalent to, and can also be returned as,
a matrix \code{m} with \code{nrow(m)} equal to \code{length()} of an input vector, and
\code{ncol(m)} equal to the number of output bands. In matrix form, each column
contains a vector of output values for a band.
\code{length(m)} must be equal to the \code{length()} of an input vector multiplied by
\code{length(out_band)}. The dimensions described above are assumed and not
read from the return value of \code{expr}.
}
\examples{
## Using pixel longitude/latitude

# Hopkins bioclimatic index (HI) as described in:
# Bechtold, 2004, West. J. Appl. For. 19(4):245-251.
# Integrates elevation, latitude and longitude into an index of the
# phenological occurrence of springtime. Here it is relativized to
# mean values for an eight-state region in the western US.
# Positive HI means spring is delayed by that number of days relative
# to the reference position, while negative values indicate spring is
# advanced. The original equation had elevation units as feet, so
# converting m to ft in `expr`.

elev_file <- system.file("extdata/storml_elev.tif", package="gdalraster")

# expression to calculate HI
expr <- "round( ((ELEV_M * 3.281 - 5449) / 100) +
                ((pixelLat - 42.16) * 4) +
                ((-116.39 - pixelLon) * 1.25) )"

# calc() writes to a tempfile by default
hi_file <- calc(expr = expr,
                rasterfiles = elev_file,
                var.names = "ELEV_M",
                dtName = "Int16",
                nodata_value = -32767,
                setRasterNodataValue = TRUE)

ds <- new(GDALRaster, hi_file)
# min, max, mean, sd
ds$getStatistics(band = 1, approx_ok = FALSE, force = TRUE)
ds$close()
\dontshow{deleteDataset(hi_file)}


## Calculate normalized difference vegetation index (NDVI)

# Landast band 4 (red) and band 5 (near infrared):
b4_file <- system.file("extdata/sr_b4_20200829.tif", package="gdalraster")
b5_file <- system.file("extdata/sr_b5_20200829.tif", package="gdalraster")

expr <- "((B5 * 0.0000275 - 0.2) - (B4 * 0.0000275 - 0.2)) /
         ((B5 * 0.0000275 - 0.2) + (B4 * 0.0000275 - 0.2))"
ndvi_file <- calc(expr = expr,
                  rasterfiles = c(b4_file, b5_file),
                  var.names = c("B4", "B5"),
                  dtName = "Float32",
                  nodata_value = -32767,
                  setRasterNodataValue = TRUE)

ds <- new(GDALRaster, ndvi_file)
ds$getStatistics(band=1, approx_ok=FALSE, force=TRUE)
ds$close()
\dontshow{deleteDataset(ndvi_file)}


## Reclassify a variable by rule set

# Combine two raster layers and look for specific combinations. Then
# recode to a new value by rule set.
#
# Based on example in:
#   Stratton, R.D. 2009. Guidebook on LANDFIRE fuels data acquisition,
#   critique, modification, maintenance, and model calibration.
#   Gen. Tech. Rep. RMRS-GTR-220. U.S. Department of Agriculture,
#   Forest Service, Rocky Mountain Research Station. 54 p.
# Context: Refine national-scale fuels data to improve fire simulation
#   results in localized applications.
# Issue: Areas with steep slopes (40+ degrees) were mapped as
#   GR1 (101; short, sparse dry climate grass) and
#   GR2 (102; low load, dry climate grass) but were not carrying fire.
# Resolution: After viewing these areas in Google Earth,
#   NB9 (99; bare ground) was selected as the replacement fuel model.

# look for combinations of slope >= 40 and FBFM 101 or 102
lcp_file <- system.file("extdata/storm_lake.lcp", package="gdalraster")
rasterfiles <- c(lcp_file, lcp_file)
var.names <- c("SLP", "FBFM")
bands <- c(2, 4)
tbl <- combine(rasterfiles, var.names, bands)
nrow(tbl)
tbl_subset <- subset(tbl, SLP >= 40 & FBFM \%in\% c(101,102))
print(tbl_subset)       # twelve combinations meet the criteria
sum(tbl_subset$count)   # 85 total pixels

# recode these pixels to 99 (bare ground)
# the LCP driver does not support in-place write so make a copy as GTiff
tif_file <- file.path(tempdir(), "storml_lndscp.tif")
createCopy("GTiff", tif_file, lcp_file)

expr <- "ifelse( SLP >= 40 & FBFM \%in\% c(101,102), 99, FBFM)"
calc(expr = expr,
     rasterfiles = c(lcp_file, lcp_file),
     bands = c(2, 4),
     var.names = c("SLP", "FBFM"),
     dstfile = tif_file,
     out_band = 4,
     write_mode = "update")

# verify the ouput
rasterfiles <- c(tif_file, tif_file)
tbl <- combine(rasterfiles, var.names, bands)
tbl_subset <- subset(tbl, SLP >= 40 & FBFM \%in\% c(101,102))
print(tbl_subset)
sum(tbl_subset$count)

# if LCP file format is needed:
# createCopy("LCP", "storml_edited.lcp", tif_file)
\dontshow{deleteDataset(tif_file)}
}
\seealso{
\code{\link[=GDALRaster]{GDALRaster-class}}, \code{\link[=combine]{combine()}}, \code{\link[=rasterToVRT]{rasterToVRT()}}
}
