% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stat-dens2d-filter.r
\name{stat_dens2d_filter}
\alias{stat_dens2d_filter}
\alias{stat_dens2d_filter_g}
\title{Filter observations by local 2D density}
\usage{
stat_dens2d_filter(
  mapping = NULL,
  data = NULL,
  geom = "point",
  position = "identity",
  keep.fraction = 0.1,
  keep.number = Inf,
  keep.sparse = TRUE,
  invert.selection = FALSE,
  na.rm = TRUE,
  show.legend = FALSE,
  inherit.aes = TRUE,
  h = NULL,
  n = NULL,
  ...
)

stat_dens2d_filter_g(
  mapping = NULL,
  data = NULL,
  geom = "point",
  position = "identity",
  keep.fraction = 0.1,
  keep.number = Inf,
  keep.sparse = TRUE,
  invert.selection = FALSE,
  na.rm = TRUE,
  show.legend = FALSE,
  inherit.aes = TRUE,
  h = NULL,
  n = NULL,
  ...
)
}
\arguments{
\item{mapping}{The aesthetic mapping, usually constructed with
\code{\link[ggplot2]{aes}} or \code{\link[ggplot2]{aes_}}. Only needs
to be set at the layer level if you are overriding the plot defaults.}

\item{data}{A layer specific dataset - only needed if you want to override
the plot defaults.}

\item{geom}{The geometric object to use display the data.}

\item{position}{The position adjustment to use for overlapping points on this
layer}

\item{keep.fraction}{numeric [0..1]. The fraction of the observations (or
rows) in \code{data} to be retained.}

\item{keep.number}{integer Set the maximum number of observations to retain,
effective only if obeying \code{keep.fraction} would result in a larger
number.}

\item{keep.sparse}{logical If \code{TRUE}, the default, observations from the
more sparse regions are retained, if \code{FALSE} those from the densest
regions.}

\item{invert.selection}{logical If \code{TRUE}, the complement of the
selected rows are returned.}

\item{na.rm}{a logical value indicating whether NA values should be stripped
before the computation proceeds.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped. \code{FALSE}
never includes, and \code{TRUE} always includes.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics, rather
than combining with them. This is most useful for helper functions that
define both data and aesthetics and shouldn't inherit behaviour from the
default plot specification, e.g. \code{\link[ggplot2]{borders}}.}

\item{h}{vector of bandwidths for x and y directions. Defaults to normal
reference bandwidth (see bandwidth.nrd). A scalar value will be taken to
apply to both directions.}

\item{n}{Number of grid points in each direction. Can be scalar or a
length-2 integer vector}

\item{...}{other arguments passed on to \code{\link[ggplot2]{layer}}. This
can include aesthetics whose values you want to set, not map. See
\code{\link[ggplot2]{layer}} for more details.}
}
\value{
A copy of \code{data} with a subset of the rows retained based on
  a 2D-density-based filtering criterion.
}
\description{
\code{stat_dens2d_filter} Filters-out/filters-in observations in
  regions of a plot panel with high density of observations, based on the
  values mapped to both \code{x} and \code{y} aesthetics.
  \code{stat_dens2d_filter_g} does the filtering by group instead of by
  panel. This second stat is useful for highlighting observations, while
  the first one tends to be most useful when the aim is to prevent clashes
  among text labels.
}
\examples{

library(ggrepel)

random_string <- function(len = 6) {
paste(sample(letters, len, replace = TRUE), collapse = "")
}

# Make random data.
set.seed(1001)
d <- tibble::tibble(
  x = rnorm(100),
  y = rnorm(100),
  group = rep(c("A", "B"), c(50, 50)),
  lab = replicate(100, { random_string() })
)

# filter (and here highlight) 1/10 observations in sparsest regions
ggplot(data = d, aes(x, y)) +
  geom_point() +
  stat_dens2d_filter(colour = "red")

# filter observations not in the sparsest regions
ggplot(data = d, aes(x, y)) +
  geom_point() +
  stat_dens2d_filter(colour = "blue", invert.selection = TRUE)

# filter observations in dense regions of the plot
ggplot(data = d, aes(x, y)) +
  geom_point() +
  stat_dens2d_filter(colour = "blue", keep.sparse = FALSE)

# filter 1/2 the observations
ggplot(data = d, aes(x, y)) +
  geom_point() +
  stat_dens2d_filter(colour = "red", keep.fraction = 0.5)

# filter 1/2 the observations but cap their number to maximum 12 observations
ggplot(data = d, aes(x, y)) +
  geom_point() +
  stat_dens2d_filter(colour = "red",
                     keep.fraction = 0.5,
                     keep.number = 12)

# density filtering done jointly across groups
ggplot(data = d, aes(x, y, colour = group)) +
  geom_point() +
  stat_dens2d_filter(shape = 1, size = 3, keep.fraction = 1/4)

# density filtering done independently for each group
ggplot(data = d, aes(x, y, colour = group)) +
  geom_point() +
  stat_dens2d_filter_g(shape = 1, size = 3, keep.fraction = 1/4)

# density filtering done jointly across groups by overriding grouping
ggplot(data = d, aes(x, y, colour = group)) +
  geom_point() +
  stat_dens2d_filter_g(colour = "black",
                       shape = 1, size = 3, keep.fraction = 1/4)

# label observations
ggplot(data = d, aes(x, y, label = lab, colour = group)) +
  geom_point() +
  stat_dens2d_filter(geom = "text")

# repulsive labels with ggrepel::geom_text_repel()
ggplot(data = d, aes(x, y, label = lab, colour = group)) +
  geom_point() +
  stat_dens2d_filter(geom = "text_repel")

}
\seealso{
\code{\link[MASS]{kde2d}} used internally.

Other statistics returning a subset of data: 
\code{\link{stat_dens1d_filter}()},
\code{\link{stat_dens1d_labels}()},
\code{\link{stat_dens2d_labels}()}
}
\concept{statistics returning a subset of data}
