% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkw.R
\name{qgkw}
\alias{qgkw}
\title{Generalized Kumaraswamy Distribution Quantile Function}
\usage{
qgkw(
  p,
  alpha = 1,
  beta = 1,
  gamma = 1,
  delta = 0,
  lambda = 1,
  lower.tail = TRUE,
  log.p = FALSE
)
}
\arguments{
\item{p}{Vector of probabilities (values between 0 and 1).}

\item{alpha}{Shape parameter \code{alpha} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{beta}{Shape parameter \code{beta} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{gamma}{Shape parameter \code{gamma} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{delta}{Shape parameter \code{delta} >= 0. Can be a scalar or a vector.
Default: 0.0.}

\item{lambda}{Shape parameter \code{lambda} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{lower.tail}{Logical; if \code{TRUE} (default), probabilities are
\eqn{P(X \le x)}, otherwise, \eqn{P(X > x)}.}

\item{log.p}{Logical; if \code{TRUE}, probabilities \code{p} are given as
\eqn{\log(p)}. Default: \code{FALSE}.}
}
\value{
A vector of quantiles corresponding to the given probabilities \code{p}.
The length of the result is determined by the recycling rule applied to
the arguments (\code{p}, \code{alpha}, \code{beta}, \code{gamma},
\code{delta}, \code{lambda}). Returns:
\itemize{
\item \code{0} for \code{p = 0} (or \code{p = -Inf} if \code{log.p = TRUE}).
\item \code{1} for \code{p = 1} (or \code{p = 0} if \code{log.p = TRUE}).
\item \code{NaN} for \code{p < 0} or \code{p > 1} (or corresponding log scale).
\item \code{NaN} for invalid parameters (e.g., \code{alpha <= 0},
\code{beta <= 0}, \code{gamma <= 0}, \code{delta < 0},
\code{lambda <= 0}).
}
}
\description{
Computes the quantile function (inverse CDF) for the five-parameter
Generalized Kumaraswamy (GKw) distribution. Finds the value \code{x} such
that \eqn{P(X \le x) = p}, where \code{X} follows the GKw distribution.
}
\details{
The quantile function \eqn{Q(p)} is the inverse of the CDF \eqn{F(x)}.
Given \eqn{F(x) = I_{y(x)}(\gamma, \delta+1)} where
\eqn{y(x) = [1-(1-x^{\alpha})^{\beta}]^{\lambda}}, the quantile function is:
\deqn{
Q(p) = x = \left\{ 1 - \left[ 1 - \left( I^{-1}_{p}(\gamma, \delta+1) \right)^{1/\lambda} \right]^{1/\beta} \right\}^{1/\alpha}
}
where \eqn{I^{-1}_{p}(a, b)} is the inverse of the regularized incomplete beta
function, which corresponds to the quantile function of the Beta distribution,
\code{\link[stats]{qbeta}}.

The computation proceeds as follows:
\enumerate{
\item Calculate \code{y = stats::qbeta(p, shape1 = gamma, shape2 = delta + 1, lower.tail = lower.tail, log.p = log.p)}.
\item Calculate \eqn{v = y^{1/\lambda}}.
\item Calculate \eqn{w = (1 - v)^{1/\beta}}. Note: Requires \eqn{v \le 1}.
\item Calculate \eqn{q = (1 - w)^{1/\alpha}}. Note: Requires \eqn{w \le 1}.
}
Numerical stability is maintained by handling boundary cases (\code{p = 0},
\code{p = 1}) directly and checking intermediate results (e.g., ensuring
arguments to powers are non-negative).
}
\examples{
\donttest{
# Basic quantile calculation (median)
median_val <- qgkw(0.5, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1)
print(median_val)

# Computing multiple quantiles
probs <- c(0.01, 0.1, 0.25, 0.5, 0.75, 0.9, 0.99)
quantiles <- qgkw(probs, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1)
print(quantiles)

# Upper tail quantile (e.g., find x such that P(X > x) = 0.1, which is 90th percentile)
q90 <- qgkw(0.1, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1,
            lower.tail = FALSE)
print(q90)
# Check: should match quantile for p = 0.9 with lower.tail = TRUE
print(qgkw(0.9, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1))

# Log probabilities
median_logp <- qgkw(log(0.5), alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1,
                    log.p = TRUE)
print(median_logp) # Should match median_val

# Vectorized parameters
alphas_vec <- c(0.5, 1.0, 2.0)
betas_vec <- c(1.0, 2.0, 3.0)
# Get median for 3 different GKw distributions
medians_vec <- qgkw(0.5, alpha = alphas_vec, beta = betas_vec, gamma = 1, delta = 0, lambda = 1)
print(medians_vec)

# Verify inverse relationship with pgkw
p_val <- 0.75
x_val <- qgkw(p_val, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1)
p_check <- pgkw(x_val, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1)
print(paste("Calculated p:", p_check, " (Expected:", p_val, ")"))
}
}
\references{
Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation}

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.
}
\seealso{
\code{\link{dgkw}}, \code{\link{pgkw}}, \code{\link{rgkw}},
\code{\link[stats]{qbeta}}
}
\author{
Lopes, J. E.
}
\keyword{distribution}
\keyword{quantile}
