% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg-other-methods.R
\name{update.gkwreg}
\alias{update.gkwreg}
\title{Update and Re-fit a GKw Regression Model}
\usage{
\method{update}{gkwreg}(object, formula., ..., data. = NULL, evaluate = TRUE)
}
\arguments{
\item{object}{An object of class \code{"gkwreg"}, typically obtained from
\code{\link{gkwreg}}.}

\item{formula.}{Changes to the formula. This is a formula where \code{.} refers
to the corresponding part of the old formula. For multi-part formulas
(e.g., \code{y ~ x1 | x2 | x3}), you can update each part separately using
the \code{|} separator.}

\item{...}{Additional arguments to the call, or arguments with changed values.
Use \code{name = NULL} to remove an argument.}

\item{data.}{Optional. A new data frame in which to evaluate the updated model.
If omitted, the original data is used.}

\item{evaluate}{Logical. If \code{TRUE} (default), the updated model is fitted.
If \code{FALSE}, the updated call is returned without fitting.}
}
\value{
If \code{evaluate = TRUE}, a new fitted model object of class
\code{"gkwreg"}. If \code{evaluate = FALSE}, an updated call.
}
\description{
Updates and (by default) re-fits a Generalized Kumaraswamy regression model.
This method allows modification of the model formula, data, or other arguments
without having to completely re-specify the model call. Supports formulas with
up to 5 parts (alpha, beta, gamma, delta, lambda) using the \pkg{Formula} package.
}
\details{
The \code{update} method allows you to modify a fitted model and re-fit it
with the changes. The GKw regression model supports formulas with up to 5 parts:
\code{y ~ model_alpha | model_beta | model_gamma | model_delta | model_lambda}

Each part can be updated independently using \code{.} to refer to the current
specification:
\itemize{
\item \code{. ~ . + x | . | . | . | .} - Add \code{x} to alpha only
\item \code{. ~ . | . + x | . | . | .} - Add \code{x} to beta only
\item \code{. ~ . | . | . + x | . | .} - Add \code{x} to gamma only
\item \code{. ~ . + x | . + x | . | . | .} - Add \code{x} to alpha and beta
\item \code{. ~ . - x | . | . | . | .} - Remove \code{x} from alpha
}

Omitting parts at the end is allowed (they default to \code{.}):
\itemize{
\item \code{. ~ . + x | .} is equivalent to \code{. ~ . + x | . | . | . | .}
\item \code{. ~ . | . + x} is equivalent to \code{. ~ . | . + x | . | . | .}
}
}
\examples{
\donttest{
# Load example data
require(gkwreg)

data(GasolineYield)

# EXAMPLE 1: Simple formulas (1 part - alpha only)

m1_0 <- gkwreg(yield ~ 1, data = GasolineYield, family = "kw")
m1_1 <- update(m1_0, . ~ . + temp)
m1_2 <- update(m1_1, . ~ . + batch)
m1_3 <- update(m1_2, . ~ . - temp)

anova(m1_0, m1_1, m1_2)
AIC(m1_0, m1_1, m1_2, m1_3)
BIC(m1_0, m1_1, m1_2, m1_3)

# EXAMPLE 2: Two-part formulas (alpha | beta)

# Start with intercept-only for both
m2_0 <- gkwreg(yield ~ 1 | 1, data = GasolineYield, family = "kw")

# Add temp to alpha
m2_1 <- update(m2_0, . ~ . + temp | .)

# Add batch to beta
m2_2 <- update(m2_1, . ~ . | . + batch)

# Add batch to alpha too
m2_3 <- update(m2_2, . ~ . + batch | .)

anova(m2_0, m2_1, m2_2, m2_3)
AIC(m2_0, m2_1, m2_2, m2_3)

# EXAMPLE 3: Three-part formulas (alpha | beta | gamma)

m3_0 <- gkwreg(yield ~ 1,
  data = GasolineYield,
  family = "gkw",
  control = gkw_control(method = "BFGS", maxit = 2000)
)

m3_1 <- update(m3_0, . ~ . + temp | . | .)
m3_2 <- update(m3_1, . ~ . | . + batch | .)
m3_3 <- update(m3_2, . ~ . | . | . + temp)

anova(m3_0, m3_1, m3_2, m3_3)

# EXAMPLE 4: Practical nested model comparison

# Null model
fit0 <- gkwreg(yield ~ 1,
  data = GasolineYield,
  family = "kw",
  control = gkw_control(method = "BFGS", maxit = 2000)
)

# Add main effects to alpha
fit1 <- update(fit0, . ~ . + temp)
fit2 <- update(fit1, . ~ . + batch)

# Model beta parameter
fit3 <- update(fit2, . ~ . | temp)
fit4 <- update(fit3, . ~ . | . + batch)

# Full comparison
anova(fit0, fit1, fit2, fit3, fit4)
AIC(fit0, fit1, fit2, fit3, fit4)
BIC(fit0, fit1, fit2, fit3, fit4)

# EXAMPLE 5: Changing other parameters

# Change family
fit_gkw <- update(fit2, family = "gkw")

# Change link function
fit_logit <- update(fit2, link = list(alpha = "logit"))

# View call without fitting
update(fit2, . ~ . | . + temp, evaluate = FALSE)
}

}
\seealso{
\code{\link{gkwreg}}, \code{\link[stats]{update}},
\code{\link[Formula]{Formula}}
}
\author{
Lopes, J. E.
}
