% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getResidualCov.gllvm.R
\name{getResidualCov.gllvm}
\alias{getResidualCov.gllvm}
\alias{getResidualCov}
\title{Extract residual covariance matrix from gllvm object}
\usage{
\method{getResidualCov}{gllvm}(object, adjust = 1)
}
\arguments{
\item{object}{an object of class 'gllvm'.}

\item{adjust}{The type of adjustment used for  negative binomial, binomial and normal distribution when computing residual correlation matrix. Options are 0 (no adjustment), 1 (the default adjustment) and 2 (alternative adjustment for NB distribution), see details.}
}
\value{
Function returns following components:
 \item{cov }{residual covariance matrix}
 \item{trace }{trace of the residual covariance matrix}
 \item{trace.q }{trace of the residual covariance matrix per latent variable}
}
\description{
Calculates the residual covariance matrix for gllvm model.
}
\details{
Residual covariance matrix, storing information on species co-occurrence that is not explained by the environmental variables (if included), is calculated using the matrix of latent variables loadings, that is, \eqn{\Theta\Theta'}.

When the responses are modelled using the negative binomial distribution, the residual variances for each species must be adjusted for overdispersion. The two possible adjustement terms are \eqn{log(\phi_j + 1)} (\code{adjust = 1}) and \eqn{\psi^{(1)}(1/\phi_j)} (\code{adjust = 2}), where \eqn{\psi^{(1)}} is the trigamma function.

The negative binomial model can be written using different parametrizations. 
The residual covariance with \code{adjust = 1} can be obtained using the lognormal-Poisson parametrization, that is,
\deqn{Y_{ij} \sim Poisson(\mu_{ij} \lambda_j),}
where \eqn{\lambda_j \sim lognormal(-\sigma^2/2, \sigma^2)} and \eqn{\sigma^2 = log(\phi_j + 1)} and \eqn{log(\mu_{ij}) = \eta_{ij}}. Now \eqn{E[Y_{ij}] = \mu_{ij}} and variance \eqn{V(\mu_{ij}) = \mu_{ij} + \mu_{ij}^2 (exp(\sigma^2) - 1) = \mu_{ij} + \mu_{ij}^2 \phi_j}, which are the same as for the NB distribution.
Therefore, on linear predictor scale, we have the variance 
\deqn{V(log(\mu_{ij} \lambda_j)) = V(log\mu_{ij}) + V(log\lambda_j) = V(u_i'\theta_j) + \sigma^2 = \theta_j'\theta_j + log(\phi_j + 1).}
which leads to the residual covariance matrix \eqn{\Theta \Theta' + \Psi}, where \eqn{\Psi} is the diagonal matrix with \eqn{log(\phi_j + 1)} as diagonal elements (\code{adjust = 1}).

Or, for a GLLVM where species are a quadratic function of the latent variables, we instead have
\deqn{V(log(\mu_{ij} \lambda_j)) = V(log\mu_{ij}) + V(log\lambda_j) = V(u_i'\theta_j-u_i' D_j u_i) + \sigma^2 = \theta_j'\theta_j + 2diag(D_j)'diag(D_j)log(\phi_j + 1).}
which leads to the residual covariance matrix \eqn{\Theta \Theta' + 2 \Gamma_j \Gamma_j' + diag(\Phi)}, where \eqn{\Gamma_j} holds the quadratic coefficients.
Since the quadratic coefficients are constrained to be positive, the residual covariance in the latter case is, given the same coefficients on the linear term, equal or more positive than in the linear case.

The residual covariance matrix with \code{adjust = 2} can be obtained by using Poisson-Gamma parametrization
\deqn{Y_{ij} \sim Poisson(\mu_{ij} \lambda_j),}
where \eqn{\lambda_j \sim Gamma(1/\phi_j, 1/\phi_j)} and \eqn{\mu_{ij}} is as above. The mean and the variance are of similar form as above and we have that
\deqn{V(log(\mu_{ij} \lambda_j)) = V(log\mu_{ij}) + V(log\lambda_j) = \theta_j'\theta_j + \psi^{(1)}(1/\phi_j),}
where \eqn{\psi^{(1)}} is the trigamma function.

In the case of binomial distribution, the adjustment terms (\code{adjust = 1}) are 1 for probit link and \eqn{\pi^2/3} for logit link.
These are obtained by treating binomial model as latent variable model. Assume
\deqn{Y^*_{ij} = \eta_{ij} + e_{ij},}
where \eqn{e_{ij} \sim N(0, 1)} for probit model, and \eqn{e_{ij} ~ logistic(0, 1)} for logit model.
Then binary response is defined as \eqn{Y_{ij} = 1}, if \eqn{Y^*_{ij} > 0} and 0 otherwise.
Now we have that \eqn{\mu_{ij} = P(Y_{ij} = 1) = P(Y^*_{ij} > 0) = P(\eta_{ij} > -e_{ij}) = P(e_{ij} <= \eta_{ij})} which leads to probit and logit models.
On linear predictor scale we then have that
\deqn{V(\eta_{ij} + e_{ij}) = V(\eta_{ij}) + V(e_{ij}).}
For the probit model, the residual covariance matrix is then \eqn{\Theta\Theta' + I_m}, and for the logit model \eqn{\Theta\Theta' + \pi^2/3 I_m}.
Similarly as above, for a GLLVM where species are a quadratic function of the latent variables, the term \eqn{2\Gamma_j\Gamma_j'} is added to the residual covariance matrix.

For normal distribution, we can write
\deqn{Y_{ij} = \eta_{ij} + e_{ij},}
where \eqn{e_{ij} \sim N(0, \phi_j^2)} and thus we have that
\deqn{V(\eta_{ij} + e_{ij}) = V(\eta_{ij}) + V(e_{ij}).}
For the gaussian model, the residual covariance matrix is then \eqn{\Theta\Theta' + diag(\Phi^2)}.
}
\examples{
\dontrun{
# Load a dataset from the mvabund package
data(antTraits)
y <- as.matrix(antTraits$abund)
# Fit gllvm model
fit <- gllvm(y = y, family = poisson())
# residual covariance:
rescov <- getResidualCov(fit)
rescov$cov
# Trace of the covariance matrix
rescov$trace
# Trace per latent variable
rescov$trace.q
}
}
\author{
Francis K.C. Hui, Jenni Niku, David I. Warton, Bert van der Veen
}
