#' Authenticate with Google Analytics OAuth2
#'
#' A wrapper for \link[googleAuthR]{gar_auth} and \link[googleAuthR]{gar_auth_service}
#'
#' @param new_user If TRUE, reauthenticate via Google login screen
#' @param no_auto Skip auto authentication
#' 
#' @details
#' 
#' Run this function first time to authenticate with Google in your browser.  
#' 
#' After initial authentication, a \code{.httr-oauth} will be saved to your working directory, where your authentication details are kept.  Keep this file safe.
#' 
#' If you want to reauthenticate, delete this file from your directory or run \code{ga_auth(new_user = TRUE)}
#' 
#' @section Auto-authentication:
#' 
#' You can choose to auto-authenticate by moving your \code{.httr-oauth} or by 
#'   creating a Google OAuth service account JSON file.
#' 
#' Specify an environment variable in R via a \code{.Renviron} file or using \link{Sys.setenv} which point to the file location of your chosen authentication file.
#' 
#' Once you have set the environment variable \code{GA_AUTH_FILE} to a valid file location,
#'   the function will look there for authentication details upon loading the library meaning 
#'   you will not need to call \code{ga_auth()} yourself as you would normally.
#' 
#' An example \code{.Renviron} file is below:
#' 
#' \code{GA_AUTH_FILE = "/Users/bob/auth/googleAnalyticsR.httr-oauth"}
#'
#' \code{GA_AUTH_FILE} can be either a token generated by \link[googleAuthR]{gar_auth} or
#'   service account JSON ending with file extension \code{.json}
#'   
#' If you use the service account JSON, you will need to add the service account email 
#'   to your Google Analytics users to see data e.g. \code{xxxx@yyyyyy.iam.gserviceaccount.com}
#'
#' @return Invisibly, the token that has been saved to the session
#' @import googleAuthR
#' @importFrom tools file_ext
#' @export
ga_auth <- function(new_user = FALSE, no_auto = FALSE){
  
  needed <- c("https://www.googleapis.com/auth/analytics", 
              "https://www.googleapis.com/auth/analytics.readonly",
              "https://www.googleapis.com/auth/analytics.manage.users.readonly",
              "https://www.googleapis.com/auth/analytics.edit",
              "https://www.googleapis.com/auth/analytics.manage.users",
              "https://www.googleapis.com/auth/analytics.provision"	)
  
  out <- gar_auto_auth(needed,
                       new_user = new_user,
                       no_auto = no_auto,
                       environment_var = "GA_AUTH_FILE",
                       travis_environment_var = "TRAVIS_GA_AUTH_FILE")
  
  myMessage("Authenticated", level = 3)
  invisible(out)
}