#' Create a reactive Google OAuth2 token
#' 
#' Use within a Shiny server.R session to create the access token passed 
#'   to all Google API functions using \code{with_shiny}
#' 
#' @param session A Shiny session object.
#' 
#' @return A reactive Google auth token

#' @examples
#' \dontrun{
#' ## in global.R
#' 
#' ## create the API call function, example with goo.gl URL shortner
#' library(googleAuthR)
#' options("googleAuthR.scopes.selected" = c("https://www.googleapis.com/auth/urlshortener"))
#' 
#' shorten_url <- function(url){
#' 
#'   body = list(
#'     longUrl = url
#'  )
#'  
#'  f <- gar_api_generator("https://www.googleapis.com/urlshortener/v1/url",
#'                         "POST",
#'                         data_parse_function = function(x) x$id)
#'                         
#'  f(the_body = body)
#'  
#'  }
#' 
#' 
#' ## in server.R
#' library(shiny)
#' library(googleAuthR)
#' source('global.R')
#' 
#' shinyServer(function(input, output, session)){
#'   
#'   ## Get auth code from return URL
#'   access_token  <- reactiveAccessToken(session)
#' 
#'   ## Make a loginButton to display using loginOutput
#'   output$loginButton <- renderLogin(session, access_token())
#'
#'   short_url_output <- eventReactive(input$submit, {
#'   ## wrap existing function with_shiny
#'   ## pass the reactive token in shiny_access_token
#'   ## pass other named arguments
#'     short_url <- with_shiny(f = shorten_url, 
#'                            shiny_access_token = access_token(),
#'                            url=input$url)
#'                            
#'    })
#'    
#'    output$short_url <- renderText({
#'    
#'      short_url_output()
#'      
#'    })
#'  }
#' 
#' ## in ui.R
#' library(shiny)
#' library(googleAuthR)
#' 
#' shinyUI(
#'   fluidPage(
#'     loginOutput("loginButton"),
#'     textInput("url", "Enter URL"),
#'     actionButton("submit", "Shorten URL"),
#'     textOutput("short_url")
#'     ))
#' }
#' 
#' @family shiny auth functions
#' @export
reactiveAccessToken <- function(session){
  check_package_loaded("shiny")
  .Deprecated("googleAuth", package = "googleAuthR",
              "reactiveAccessToken is deprecated as of googleAuthR 0.3.0.")
  shiny::reactive({
    ## gets all the parameters in the URL. The auth code should be one of them.
    
    if(!is.null(authReturnCode(session))){
      ## extract the authorization token
      app_url <- gar_shiny_getUrl(session)    
      access_token <- gar_shiny_getToken(authReturnCode(session), app_url)
      
      Authentication$set("public", "app_url", app_url, overwrite=TRUE)
      Authentication$set("public", "shiny", TRUE, overwrite=TRUE)
      
      access_token
      
    } else {
      NULL
    }
  })
}

#' Login/logout Shiny output
#' 
#' USe within a ui.R to render the login button generated by \code{renderLogin}
#' 
#' @param output_name Name of what output object was assigned in \code{renderLogin}
#' 
#' @return A login/logout button in a Shiny app
#' 
#' @export
#' @family shiny auth functions
#' @examples
#' \dontrun{
#' ## in global.R
#' 
#' ## create the API call function, example with goo.gl URL shortner
#' library(googleAuthR)
#' options("googleAuthR.scopes.selected" = c("https://www.googleapis.com/auth/urlshortener"))
#' 
#' shorten_url <- function(url){
#' 
#'   body = list(
#'     longUrl = url
#'  )
#'  
#'  f <- gar_api_generator("https://www.googleapis.com/urlshortener/v1/url",
#'                         "POST",
#'                         data_parse_function = function(x) x$id)
#'                         
#'  f(the_body = body)
#'  
#'  }
#' 
#' 
#' ## in server.R
#' library(shiny)
#' library(googleAuthR)
#' source('global.R')
#' 
#' shinyServer(function(input, output, session)){
#'   
#'   ## Get auth code from return URL
#'   access_token  <- reactiveAccessToken(session)
#' 
#'   ## Make a loginButton to display using loginOutput
#'   output$loginButton <- renderLogin(session, access_token())
#'
#'   short_url_output <- eventReactive(input$submit, {
#'   ## wrap existing function with_shiny
#'   ## pass the reactive token in shiny_access_token
#'   ## pass other named arguments
#'     short_url <- with_shiny(f = shorten_url, 
#'                            shiny_access_token = access_token(),
#'                            url=input$url)
#'                            
#'    })
#'    
#'    output$short_url <- renderText({
#'    
#'      short_url_output()
#'      
#'    })
#'  }
#' 
#' ## in ui.R
#' library(shiny)
#' library(googleAuthR)
#' 
#' shinyUI(
#'   fluidPage(
#'     loginOutput("loginButton"),
#'     textInput("url", "Enter URL"),
#'     actionButton("submit", "Shorten URL"),
#'     textOutput("short_url")
#'     ))
#' }
loginOutput <- function(output_name){
  check_package_loaded("shiny")
  message("loginOutput is deprecated as of googleAuthR 0.3.0. Use googleAuth() and googleAuthUI() instead.")
  shiny::uiOutput(output_name)
}

#' Render a Google API Authentication Login/logout button
#' 
#' Use within a Shiny server.R to assign to an output for ui.R.
#' The login button carries an ActionLink with value "signed_in" 
#'   but as Shiny reloads on pushing it can't be used for detection of login state.  
#'   Use \code{!is.null(access_token())} instead.
#' 
#' @param session A Shiny session object
#' @param access_token A token generated by \code{reactiveAccessToken}
#' @param login_text What the login text will read on the button
#' @param logout_text What the logout text will read on the button
#' @param login_class The Bootstrap class for the login link
#' @param logout_class The Bootstrap class for the logout link
#' @param access_type Online or offline access for the authentication URL. 
#' @param revoke If TRUE a user on logout will need to re-authenticate.
#' @param approval_prompt Whether to show the consent screen on authentication.
#' @return An object to assign to output e.g. output$login
#' 
#' @seealso \link{revokeEventObserver}
#' 
#' @examples
#' \dontrun{
#' ## in global.R
#' 
#' ## create the API call function, example with goo.gl URL shortner
#' library(googleAuthR)
#' options("googleAuthR.scopes.selected" = c("https://www.googleapis.com/auth/urlshortener"))
#' 
#' shorten_url <- function(url){
#' 
#'   body = list(
#'     longUrl = url
#'  )
#'  
#'  f <- gar_api_generator("https://www.googleapis.com/urlshortener/v1/url",
#'                         "POST",
#'                         data_parse_function = function(x) x$id)
#'                         
#'  f(the_body = body)
#'  
#'  }
#' 
#' 
#' ## in server.R
#' library(shiny)
#' library(googleAuthR)
#' source('global.R')
#' 
#' shinyServer(function(input, output, session)){
#'   
#'   ## Get auth code from return URL
#'   access_token  <- reactiveAccessToken(session)
#' 
#'   ## Make a loginButton to display using loginOutput
#'   ## revoke=TRUE means upon logout a user will need to reauthenticate
#'   output$loginButton <- renderLogin(session, access_token(), revoke=TRUE)
#'   
#'   ## Needed if revoke=TRUE above
#'   revokeEventObserver(access_token())
#'
#'   short_url_output <- eventReactive(input$submit, {
#'   ## wrap existing function with_shiny
#'   ## pass the reactive token in shiny_access_token
#'   ## pass other named arguments
#'     short_url <- with_shiny(f = shorten_url, 
#'                            shiny_access_token = access_token(),
#'                            url=input$url)
#'                            
#'    })
#'    
#'    output$short_url <- renderText({
#'    
#'      short_url_output()
#'      
#'    })
#'  }
#' 
#' ## in ui.R
#' library(shiny)
#' library(googleAuthR)
#' 
#' shinyUI(
#'   fluidPage(
#'     loginOutput("loginButton"),
#'     textInput("url", "Enter URL"),
#'     actionButton("submit", "Shorten URL"),
#'     textOutput("short_url")
#'     ))
#' }
#' @export
#' @family shiny auth functions
renderLogin <- function(session, 
                        access_token,
                        login_text="Login via Google",
                        logout_text="Logout",
                        login_class="btn btn-primary",
                        logout_class="btn btn-default",
                        access_type = c("online","offline"),
                        approval_prompt = c("auto","force"),
                        revoke = FALSE){
  check_package_loaded("shiny")
  message("renderLogin is deprecated as of googleAuthR 0.3.0. Use googleAuth() and googleAuthUI() instead.")
  access_type <- match.arg(access_type)
  approval_prompt <- match.arg(approval_prompt)
  
  shiny::renderUI({
    if(is.null(shiny::isolate(access_token))) {
      shiny::actionLink("signed_in",
                        shiny::a(login_text, 
                                 href = gar_shiny_getAuthUrl(gar_shiny_getUrl(session), 
                                                             access_type = access_type,
                                                             approval_prompt = approval_prompt), 
                                 class=login_class, 
                                 role="button"))
    } else {
      if(revoke){
        
        logout_button <- shiny::actionButton("revoke", "Revoke Access", 
                                             href = gar_shiny_getUrl(session), 
                                             class=logout_class,
                                             role="button")
        
      } else {
        logout_button <- shiny::a(logout_text, 
                                  href = gar_shiny_getUrl(session), 
                                  class=logout_class, 
                                  role="button")
      }
      
      logout_button
      
    }
  })
}

#' Listens for a user revoking authentication
#' 
#' @description If the parameter \code{revoke} is set to TRUE for \link{renderLogin}
#'   then this observer is also required in the Shiny server to do the revoking.
#' 
#' @param access_token A token generated by \code{reactiveAccessToken}.
#' @param input the input object from a shinyServer function.
#' 
#' @seealso \link{renderLogin}
#' 
#' @examples
#' \dontrun{
#' ## in global.R
#' 
#' ## create the API call function, example with goo.gl URL shortner
#' library(googleAuthR)
#' options("googleAuthR.scopes.selected" = c("https://www.googleapis.com/auth/urlshortener"))
#' 
#' shorten_url <- function(url){
#' 
#'   body = list(
#'     longUrl = url
#'  )
#'  
#'  f <- gar_api_generator("https://www.googleapis.com/urlshortener/v1/url",
#'                         "POST",
#'                         data_parse_function = function(x) x$id)
#'                         
#'  f(the_body = body)
#'  
#'  }
#' 
#' 
#' ## in server.R
#' library(shiny)
#' library(googleAuthR)
#' source('global.R')
#' 
#' shinyServer(function(input, output, session)){
#'   
#'   ## Get auth code from return URL
#'   access_token  <- reactiveAccessToken(session)
#' 
#'   ## Make a loginButton to display using loginOutput
#'   ## revoke=TRUE means upon logout a user will need to reauthenticate
#'   output$loginButton <- renderLogin(session, access_token(), revoke=TRUE)
#'   
#'   ## Needed if revoke=TRUE above
#'   revokeEventObserver(access_token())
#'
#'   short_url_output <- eventReactive(input$submit, {
#'   ## wrap existing function with_shiny
#'   ## pass the reactive token in shiny_access_token
#'   ## pass other named arguments
#'     short_url <- with_shiny(f = shorten_url, 
#'                            shiny_access_token = access_token(),
#'                            url=input$url)
#'                            
#'    })
#'    
#'    output$short_url <- renderText({
#'    
#'      short_url_output()
#'      
#'    })
#'  }
#' 
#' ## in ui.R
#' library(shiny)
#' library(googleAuthR)
#' 
#' shinyUI(
#'   fluidPage(
#'     loginOutput("loginButton"),
#'     textInput("url", "Enter URL"),
#'     actionButton("submit", "Shorten URL"),
#'     textOutput("short_url")
#'     ))
#' } 
#' @export
#' @family shiny auth functions
revokeEventObserver <- function(access_token, input){
  check_package_loaded("shiny")
  message("revokeEventObserver is deprecated as of googleAuthR 0.3.0. Use googleAuth() and googleAuthUI() instead.")  
  shiny::observeEvent(input$revoke, {
    
    ## GETS the revoke URL for this user's access_token
    httr::GET(httr::modify_url("https://accounts.google.com/o/oauth2/revoke",
                               query = 
                                 list(token = 
                                        shiny::isolate(access_token)$credentials$access_token)))
    myMessage("Revoked access", level=2)
  })
  
}
