% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stepwise.R
\name{stepwise}
\alias{stepwise}
\title{Stepwise selection of regressors}
\usage{
stepwise(data, ic = c("AICc", "AIC", "BIC", "BICc"), silent = TRUE,
  df = NULL, method = c("pearson", "kendall", "spearman"),
  distribution = c("dnorm", "dlaplace", "ds", "dgnorm", "dlogis", "dt",
  "dalaplace", "dlnorm", "dllaplace", "dls", "dlgnorm", "dbcnorm", "dfnorm",
  "dinvgauss", "dgamma", "dpois", "dnbinom", "dlogitnorm", "plogis", "pnorm"),
  occurrence = c("none", "plogis", "pnorm"), ...)
}
\arguments{
\item{data}{Data frame containing dependant variable in the first column and
the others in the rest.}

\item{ic}{Information criterion to use.}

\item{silent}{If \code{silent=FALSE}, then nothing is silent, everything is
printed out. \code{silent=TRUE} means that nothing is produced.}

\item{df}{Number of degrees of freedom to add (should be used if stepwise is
used on residuals).}

\item{method}{Method of correlations calculation. The default is Kendall's
Tau, which should be applicable to a wide range of data in different scales.}

\item{distribution}{Distribution to pass to \code{alm()}. See \link[greybox]{alm}
for details.}

\item{occurrence}{what distribution to use for occurrence part. See
\link[greybox]{alm} for details.}

\item{...}{This is temporary and is needed in order to capture "silent"
parameter if it is provided.}
}
\value{
Function returns \code{model} - the final model of the class "alm".
See \link[greybox]{alm} for details of the output.
}
\description{
Function selects variables that give linear regression with the lowest
information criteria. The selection is done stepwise (forward) based on
partial correlations. This should be a simpler and faster implementation
than step() function from `stats' package.
}
\details{
The algorithm uses alm() to fit different models and cor() to select the next
regressor in the sequence.

Some details and examples of application are also given in the vignette
"Greybox": \code{vignette("greybox","greybox")}
}
\examples{

### Simple example
xreg <- cbind(rnorm(100,10,3),rnorm(100,50,5))
xreg <- cbind(100+0.5*xreg[,1]-0.75*xreg[,2]+rnorm(100,0,3),xreg,rnorm(100,300,10))
colnames(xreg) <- c("y","x1","x2","Noise")
stepwise(xreg)

### Mixture distribution of Log Normal and Cumulative Logit
xreg[,1] <- xreg[,1] * round(exp(xreg[,1]-70) / (1 + exp(xreg[,1]-70)),0)
colnames(xreg) <- c("y","x1","x2","Noise")
ourModel <- stepwise(xreg, distribution="dlnorm",
                     occurrence=stepwise(xreg, distribution="plogis"))
summary(ourModel)

### Fat regression example
xreg <- matrix(rnorm(20000,10,3),100,200)
xreg <- cbind(100+0.5*xreg[,1]-0.75*xreg[,2]+rnorm(100,0,3),xreg,rnorm(100,300,10))
colnames(xreg) <- c("y",paste0("x",c(1:200)),"Noise")
ourModel <- stepwise(xreg,ic="AICc")
plot(ourModel$ICs,type="l",ylim=range(min(ourModel$ICs),max(ourModel$ICs)+5))
points(ourModel$ICs)
text(c(1:length(ourModel$ICs))+0.1,ourModel$ICs+5,names(ourModel$ICs))

}
\references{
\itemize{
\item Burnham Kenneth P. and Anderson David R. (2002). Model Selection
and Multimodel Inference. A Practical Information-Theoretic Approach.
Springer-Verlag New York. DOI: [10.1007/b97636](http://dx.doi.org/10.1007/b97636).
}
}
\seealso{
\code{\link[stats]{step}, \link[greybox]{xregExpander},
\link[greybox]{lmCombine}}
}
\author{
Ivan Svetunkov, \email{ivan@svetunkov.ru}
}
\keyword{models}
\keyword{nonlinear}
\keyword{regression}
\keyword{ts}
