% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_vec.R
\name{vec_fmt_duration}
\alias{vec_fmt_duration}
\title{Format a vector of numeric or duration values as styled time duration strings}
\usage{
vec_fmt_duration(
  x,
  input_units = NULL,
  output_units = NULL,
  duration_style = c("narrow", "wide", "colon-sep", "iso"),
  trim_zero_units = TRUE,
  max_output_units = NULL,
  pattern = "{x}",
  use_seps = TRUE,
  sep_mark = ",",
  force_sign = FALSE,
  locale = NULL,
  output = c("auto", "plain", "html", "latex", "rtf", "word")
)
}
\arguments{
\item{x}{A numeric vector.}

\item{input_units}{If one or more selected columns contains numeric values, a
keyword must be provided for \code{input_units} for \strong{gt} to determine how
those values are to be interpreted in terms of duration. The accepted units
are: \code{"seconds"}, \code{"minutes"}, \code{"hours"}, \code{"days"}, and \code{"weeks"}.}

\item{output_units}{Controls the output time units. The default, \code{NULL},
means that \strong{gt} will automatically choose time units based on the input
duration value. To control which time units are to be considered for output
(before trimming with \code{trim_zero_units}) we can specify a vector of one or
more of the following keywords: \code{"weeks"}, \code{"days"}, \code{"hours"},
\code{"minutes"}, or \code{"seconds"}.}

\item{duration_style}{A choice of four formatting styles for the output
duration values. With \code{"narrow"} (the default style), duration values will
be formatted with single letter time-part units (e.g., 1.35 days will be
styled as \verb{"1d 8h 24m}). With \code{"wide"}, this example value will be expanded
to \code{"1 day 8 hours 24 minutes"} after formatting. The \code{"colon-sep"} style
will put days, hours, minutes, and seconds in the \code{"([D]/)[HH]:[MM]:[SS]"}
format. The \code{"iso"} style will produce a value that conforms to the ISO
8601 rules for duration values (e.g., 1.35 days will become \code{"P1DT8H24M"}).}

\item{trim_zero_units}{Provides methods to remove output time units that have
zero values. By default this is \code{TRUE} and duration values that might
otherwise be formatted as \code{"0w 1d 0h 4m 19s"} with
\code{trim_zero_units = FALSE} are instead displayed as \code{"1d 4m 19s"}. Aside
from using \code{TRUE}/\code{FALSE} we could provide a vector of keywords for more
precise control. These keywords are: (1) \code{"leading"}, to omit all leading
zero-value time units (e.g., \code{"0w 1d"} -> \code{"1d"}), (2) \code{"trailing"}, to
omit all trailing zero-value time units (e.g., \code{"3d 5h 0s"} -> \code{"3d 5h"}),
and \code{"internal"}, which removes all internal zero-value time units (e.g.,
\code{"5d 0h 33m"} -> \code{"5d 33m"}).}

\item{max_output_units}{If \code{output_units} is \code{NULL}, where the output time
units are unspecified and left to \strong{gt} to handle, a numeric value
provided for \code{max_output_units} will be taken as the maximum number of time
units to display in all output time duration values. By default, this is
\code{NULL} and all possible time units will be displayed. This option has no
effect when \code{duration_style = "colon-sep"} (only \code{output_units} can be used
to customize that type of duration output).}

\item{pattern}{A formatting pattern that allows for decoration of the
formatted value. The value itself is represented by \code{{x}} and all other
characters are taken to be string literals.}

\item{use_seps}{An option to use digit group separators. The type of digit
group separator is set by \code{sep_mark} and overridden if a locale ID is
provided to \code{locale}. This setting is \code{TRUE} by default.}

\item{sep_mark}{The mark to use as a separator between groups of digits
(e.g., using \code{sep_mark = ","} with \code{1000} would result in a formatted value
of \verb{1,000}).}

\item{force_sign}{Should the positive sign be shown for positive values
(effectively showing a sign for all values except zero)? If so, use \code{TRUE}
for this option. The default is \code{FALSE}, where only negative value will
display a minus sign.}

\item{locale}{An optional locale ID that can be used for formatting the value
according the locale's rules. Examples include \code{"en"} for English (United
States) and \code{"fr"} for French (France). The use of a valid locale ID will
override any values provided in \code{sep_mark} and \code{dec_mark}. We can use the
\code{\link[=info_locales]{info_locales()}} function as a useful reference for all of the locales that
are supported.}

\item{output}{The output style of the resulting character vector. This can
either be \code{"auto"} (the default), \code{"plain"}, \code{"html"}, \code{"latex"}, \code{"rtf"},
or \code{"word"}. In \strong{knitr} rendering (i.e., Quarto or R Markdown), the
\code{"auto"} option will choose the correct \code{output} value}
}
\value{
A character vector.
}
\description{
Format input values to time duration values whether those input values are
numbers or of the \code{difftime} class. We can specify which time units any
numeric input values have (as weeks, days, hours, minutes, or seconds) and
the output can be customized with a duration style (corresponding to narrow,
wide, colon-separated, and ISO forms) and a choice of output units ranging
from weeks to seconds.
}
\section{Output units for the colon-separated duration style}{


The colon-separated duration style (enabled when
\code{duration_style = "colon-sep"}) is essentially a clock-based output format
which uses the display logic of chronograph watch functionality. It will, by
default, display duration values in the \verb{(D/)HH:MM:SS} format. Any duration
values greater than or equal to 24 hours will have the number of days
prepended with an adjoining slash mark. While this output format is
versatile, it can be changed somewhat with the \code{output_units} option. The
following combinations of output units are permitted:
\itemize{
\item \code{c("minutes", "seconds")} -> \code{MM:SS}
\item \code{c("hours", "minutes")} -> \code{HH:MM}
\item \code{c("hours", "minutes", "seconds")} -> \code{HH:MM:SS}
\item \code{c("days", "hours", "minutes")} -> \verb{(D/)HH:MM}
}

Any other specialized combinations will result in the default set being used,
which is \code{c("days", "hours", "minutes", "seconds")}
}

\section{Examples}{


Let's create a \code{difftime}-based vector for the next few examples:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{difftimes <-
  difftime(
    lubridate::ymd("2017-01-15"),
    lubridate::ymd(c("2015-06-25", "2016-03-07", "2017-01-10"))
  )
}\if{html}{\out{</div>}}

Using \code{vec_fmt_duration()} with its defaults provides us with a succinct
vector of formatted durations.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_duration(difftimes)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "81w 3d" "44w 6d" "5d"
}\if{html}{\out{</div>}}

We can elect to use just only the time units of days to describe the duration
values.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_duration(difftimes, output_units = "days")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "570d" "314d" "5d"
}\if{html}{\out{</div>}}

We can also use numeric values in the input vector \code{vec_fmt_duration()}.
Here's a numeric vector for use with examples:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{num_vals <- c(3.235, 0.23, 0.005, NA)
}\if{html}{\out{</div>}}

The necessary thing with numeric values as an input is defining what time
unit those values have.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_duration(num_vals, input_units = "days")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "3d 5h 38m 23s" "5h 31m 12s" "7m 12s" "NA"
}\if{html}{\out{</div>}}

We can define a set of output time units that we want to see.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_duration(
  num_vals,
  input_units = "days",
  output_units = c("hours", "minutes")
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "77h 38m" "5h 31m" "7m" "NA"
}\if{html}{\out{</div>}}

There are many duration 'styles' to choose from. We could opt for the
\code{"wide"} style.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_duration(
  num_vals,
  input_units = "days",
  duration_style = "wide"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "3 days 5 hours 38 minutes 23 seconds"
#> [2] "5 hours 31 minutes 12 seconds"
#> [3] "7 minutes 12 seconds"
#> [4] "NA"
}\if{html}{\out{</div>}}

We can always perform locale-specific formatting with \code{vec_fmt_duration()}.
Let's attempt the same type of duration formatting as before with the \code{"nl"}
locale.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_duration(
  num_vals,
  input_units = "days",
  duration_style = "wide",
  locale = "nl"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "3 dagen 5 uur 38 minuten 23 seconden"
#> [2] "5 uur 31 minuten 12 seconden"
#> [3] "7 minuten 12 seconden"
#> [4] "NA"
}\if{html}{\out{</div>}}
}

\section{Function ID}{

14-14
}

\seealso{
Other vector formatting functions: 
\code{\link{vec_fmt_bytes}()},
\code{\link{vec_fmt_currency}()},
\code{\link{vec_fmt_datetime}()},
\code{\link{vec_fmt_date}()},
\code{\link{vec_fmt_engineering}()},
\code{\link{vec_fmt_fraction}()},
\code{\link{vec_fmt_integer}()},
\code{\link{vec_fmt_markdown}()},
\code{\link{vec_fmt_number}()},
\code{\link{vec_fmt_partsper}()},
\code{\link{vec_fmt_percent}()},
\code{\link{vec_fmt_roman}()},
\code{\link{vec_fmt_scientific}()},
\code{\link{vec_fmt_time}()}
}
\concept{vector formatting functions}
