% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_stat.R
\name{add_stat}
\alias{add_stat}
\title{Add a custom statistic column}
\usage{
add_stat(
  x,
  fns,
  fmt_fun = NULL,
  header = "**Statistic**",
  footnote = NULL,
  new_col_name = NULL
)
}
\arguments{
\item{x}{tbl_summary object}

\item{fns}{list of formulas indicating the functions that create the statistic}

\item{fmt_fun}{for numeric statistics, \verb{fmt_fun=} is the styling/formatting
function. Default is \code{NULL}}

\item{header}{Column header of new column. Default is \code{"**Statistic**"}}

\item{footnote}{Footnote associated with new column. Default is no
footnote (i.e. NULL)}

\item{new_col_name}{name of new column to be created in \code{.$table_body}.
Default is \code{"add_stat_1"}, unless that column exists then it is \code{"add_stat_2"}, etc.}
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}
The function allows a user to add a new column with a custom, user-defined statistic.
}
\section{Details}{


The custom functions passed in \verb{fns=} are required to follow a specified
format. Each of these function will execute on a single variable from \code{tbl_summary()}.
\enumerate{
\item Each function must return a single scalar or character value of length one.
\item Each function may take the following arguments: \code{foo(data, variable, by, tbl)}
}
\itemize{
\item \verb{data=} is the input data frame passed to \code{tbl_summary()}
\item \verb{variable=} is a string indicating the variable to perform the calculation on
\item \verb{by=} is a string indicating the by variable from \verb{tbl_summary=}, if present
\item \verb{tbl=} the original \code{tbl_summary()} object is also available to utilize
}

The user-defined does not need to utilize each of these inputs. It's
encouraged the user-defined function accept \code{...} as each of the arguments
\emph{will} be passed to the function, even if not all inputs are utilized by
the user's function, e.g. \code{foo(data, variable, by, ...)}
}

\section{Example Output}{

\if{html}{Example 1}

\if{html}{\figure{add_stat_ex1.png}{options: width=60\%}}

\if{html}{Example 2}

\if{html}{\figure{add_stat_ex2.png}{options: width=60\%}}
}

\examples{
# Example 1 ----------------------------------
# this example replicates `add_p()`

# fn returns t-test pvalue
my_ttest <- function(data, variable, by, ...) {
  t.test(data[[variable]] ~ as.factor(data[[by]]))$p.value
}

add_stat_ex1 <-
  trial \%>\%
  select(trt, age, marker) \%>\%
  tbl_summary(by = trt, missing = "no") \%>\%
  add_p(test = everything() ~ t.test) \%>\%
  # replicating result of `add_p()` with `add_stat()`
  add_stat(
    fns = everything() ~ my_ttest, # all variables compared with with t-test
    fmt_fun = style_pvalue,        # format result with style_pvalue()
    header = "**My p-value**"      # new column header
  )

# Example 2 ----------------------------------
# fn returns t-test test statistic and pvalue
my_ttest2 <- function(data, variable, by, ...) {
  tt <- t.test(data[[variable]] ~ as.factor(data[[by]]))

  # returning test statistic and pvalue
  stringr::str_glue(
    "t={style_sigfig(tt$statistic)}, {style_pvalue(tt$p.value, prepend_p = TRUE)}"
  )
}

add_stat_ex2 <-
  trial \%>\%
  select(trt, age, marker) \%>\%
  tbl_summary(by = trt, missing = "no") \%>\%
  add_stat(
    fns = everything() ~ my_ttest2,    # all variables will be compared by t-test
    fmt_fun = NULL, # fn returns and chr, so no formatting function needed
    header = "**Treatment Comparison**",       # column header
    footnote = "T-test statistic and p-value"  # footnote
  )
# Example 1 ----------------------------------
}
