/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5B2module.h" 

#include "H5private.h"   
#include "H5B2pkg.h"     
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 

bool H5_PKG_INIT_VAR = false;

extern const H5B2_class_t H5B2_TEST[1];
extern const H5B2_class_t H5HF_HUGE_BT2_INDIR[1];
extern const H5B2_class_t H5HF_HUGE_BT2_FILT_INDIR[1];
extern const H5B2_class_t H5HF_HUGE_BT2_DIR[1];
extern const H5B2_class_t H5HF_HUGE_BT2_FILT_DIR[1];
extern const H5B2_class_t H5G_BT2_NAME[1];
extern const H5B2_class_t H5G_BT2_CORDER[1];
extern const H5B2_class_t H5SM_INDEX[1];
extern const H5B2_class_t H5A_BT2_NAME[1];
extern const H5B2_class_t H5A_BT2_CORDER[1];
extern const H5B2_class_t H5D_BT2[1];
extern const H5B2_class_t H5D_BT2_FILT[1];
extern const H5B2_class_t H5B2_TEST2[1];

const H5B2_class_t *const H5B2_client_class_g[] = {
    H5B2_TEST,                
    H5HF_HUGE_BT2_INDIR,      
    H5HF_HUGE_BT2_FILT_INDIR, 
    H5HF_HUGE_BT2_DIR,        
    H5HF_HUGE_BT2_FILT_DIR,   
    H5G_BT2_NAME,             
    H5G_BT2_CORDER,           
    H5SM_INDEX,               
    H5A_BT2_NAME,             
    H5A_BT2_CORDER,           
    H5D_BT2,                  
    H5D_BT2_FILT,             
    H5B2_TEST2                
};

H5FL_DEFINE_STATIC(H5B2_t);

H5B2_t *
H5B2_create(H5F_t *f, const H5B2_create_t *cparam, void *ctx_udata)
{
    H5B2_t     *bt2 = NULL;       
    H5B2_hdr_t *hdr = NULL;       
    haddr_t     hdr_addr;         
    H5B2_t     *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(f);
    assert(cparam);

    
    HDcompile_assert(H5B2_NUM_BTREE_ID == NELMTS(H5B2_client_class_g));

    
    if (HADDR_UNDEF == (hdr_addr = H5B2__hdr_create(f, cparam, ctx_udata)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINIT, NULL, "can't create v2 B-tree header");

    
    if (NULL == (bt2 = H5FL_MALLOC(H5B2_t)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "memory allocation failed for v2 B-tree info");

    
    if (NULL == (hdr = H5B2__hdr_protect(f, hdr_addr, ctx_udata, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, NULL, "unable to protect v2 B-tree header");

    
    bt2->hdr = hdr;
    if (H5B2__hdr_incr(bt2->hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINC, NULL,
                    "can't increment reference count on shared v2 B-tree header");

    
    if (H5B2__hdr_fuse_incr(bt2->hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINC, NULL,
                    "can't increment file reference count on shared v2 B-tree header");

    
    bt2->f = f;

    
    ret_value = bt2;

done:
    if (hdr && H5B2__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, NULL, "unable to release v2 B-tree header");
    if (!ret_value && bt2)
        if (H5B2_close(bt2) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTCLOSEOBJ, NULL, "unable to close v2 B-tree");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5B2_t *
H5B2_open(H5F_t *f, haddr_t addr, void *ctx_udata)
{
    H5B2_t     *bt2       = NULL; 
    H5B2_hdr_t *hdr       = NULL; 
    H5B2_t     *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(f);
    assert(H5_addr_defined(addr));

    
    if (NULL == (hdr = H5B2__hdr_protect(f, addr, ctx_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, NULL, "unable to protect v2 B-tree header");

    
    if (hdr->pending_delete)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTOPENOBJ, NULL, "can't open v2 B-tree pending deletion");

    
    if (NULL == (bt2 = H5FL_MALLOC(H5B2_t)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "memory allocation failed for v2 B-tree info");

    
    bt2->hdr = hdr;
    if (H5B2__hdr_incr(bt2->hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINC, NULL,
                    "can't increment reference count on shared v2 B-tree header");

    
    if (H5B2__hdr_fuse_incr(bt2->hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINC, NULL,
                    "can't increment file reference count on shared v2 B-tree header");

    
    bt2->f = f;

    
    ret_value = bt2;

done:
    if (hdr && H5B2__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, NULL, "unable to release v2 B-tree header");
    if (!ret_value && bt2)
        if (H5B2_close(bt2) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTCLOSEOBJ, NULL, "unable to close v2 B-tree");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_insert(H5B2_t *bt2, void *udata)
{
    H5B2_hdr_t *hdr;                 
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(bt2);
    assert(udata);

    
    bt2->hdr->f = bt2->f;

    
    hdr = bt2->hdr;

    
    if (H5B2__insert(hdr, udata) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, FAIL, "unable to insert record into B-tree");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_update(H5B2_t *bt2, void *udata, H5B2_modify_t op, void *op_data)
{
    H5B2_hdr_t          *hdr;                             
    H5B2_update_status_t status    = H5B2_UPDATE_UNKNOWN; 
    herr_t               ret_value = SUCCEED;             

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(bt2);
    assert(udata);

    
    bt2->hdr->f = bt2->f;

    
    hdr = bt2->hdr;

    
    if (!H5_addr_defined(hdr->root.addr)) {
        
        if (H5B2__create_leaf(hdr, hdr, &(hdr->root)) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTINIT, FAIL, "unable to create root node");
    } 

    
    if (hdr->depth > 0) {
        if (H5B2__update_internal(hdr, hdr->depth, NULL, &hdr->root, &status, H5B2_POS_ROOT, hdr, udata, op,
                                  op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTUPDATE, FAIL, "unable to update record in B-tree internal node");
    } 
    else {
        if (H5B2__update_leaf(hdr, &hdr->root, &status, H5B2_POS_ROOT, hdr, udata, op, op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTUPDATE, FAIL, "unable to update record in B-tree leaf node");
    } 

    
    assert(H5B2_UPDATE_UNKNOWN != status);

    
    if (H5B2_UPDATE_INSERT_CHILD_FULL == status) {
        if (H5B2__insert(hdr, udata) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, FAIL, "unable to insert record into B-tree");
    } 
    else if (H5B2_UPDATE_SHADOW_DONE == status || H5B2_UPDATE_INSERT_DONE == status) {
        
        if (H5B2__hdr_dirty(hdr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTMARKDIRTY, FAIL, "unable to mark B-tree header dirty");
    } 
    else {
        
        assert(H5B2_UPDATE_MODIFY_DONE == status);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_get_addr(const H5B2_t *bt2, haddr_t *addr_p)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(bt2);
    assert(addr_p);

    
    *addr_p = bt2->hdr->addr;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5B2_iterate(H5B2_t *bt2, H5B2_operator_t op, void *op_data)
{
    H5B2_hdr_t *hdr;                 
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOERR

    
    assert(bt2);
    assert(op);

    
    bt2->hdr->f = bt2->f;

    
    hdr = bt2->hdr;

    
    if (hdr->root.node_nrec > 0)
        
        if ((ret_value = H5B2__iterate_node(hdr, hdr->depth, &hdr->root, hdr, op, op_data)) < 0)
            HERROR(H5E_BTREE, H5E_CANTLIST, "node iteration failed");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_find(H5B2_t *bt2, void *udata, bool *found, H5B2_found_t op, void *op_data)
{
    H5B2_hdr_t     *hdr;                 
    H5B2_node_ptr_t curr_node_ptr;       
    void           *parent = NULL;       
    uint16_t        depth;               
    int             cmp;                 
    unsigned        idx;                 
    H5B2_nodepos_t  curr_pos;            
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(bt2);
    assert(found);

    
    bt2->hdr->f = bt2->f;

    
    hdr = bt2->hdr;

    
    curr_node_ptr = hdr->root;

    
    if (curr_node_ptr.node_nrec == 0) {
        *found = false;
        HGOTO_DONE(SUCCEED);
    }

    
    if (hdr->min_native_rec != NULL) {
        if ((hdr->cls->compare)(udata, hdr->min_native_rec, &cmp) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
        if (cmp < 0) {
            *found = false; 
            HGOTO_DONE(SUCCEED);
        }
        else if (cmp == 0) { 
            if (op && (op)(hdr->min_native_rec, op_data) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL,
                            "'found' callback failed for B-tree find operation");
            *found = true;
            HGOTO_DONE(SUCCEED);
        } 
    }     
    if (hdr->max_native_rec != NULL) {
        if ((hdr->cls->compare)(udata, hdr->max_native_rec, &cmp) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
        if (cmp > 0) {
            *found = false; 
            HGOTO_DONE(SUCCEED);
        }
        else if (cmp == 0) { 
            if (op && (op)(hdr->max_native_rec, op_data) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL,
                            "'found' callback failed for B-tree find operation");
            *found = true;
            HGOTO_DONE(SUCCEED);
        } 
    }     

    
    depth = hdr->depth;

    
    if (hdr->swmr_write)
        parent = hdr;

    
    cmp      = -1;
    curr_pos = H5B2_POS_ROOT;
    while (depth > 0) {
        H5B2_internal_t *internal;      
        H5B2_node_ptr_t  next_node_ptr; 

        
        if (NULL == (internal = H5B2__protect_internal(hdr, parent, &curr_node_ptr, depth, false,
                                                       H5AC__READ_ONLY_FLAG)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to load B-tree internal node");

        
        if (parent) {
            if (parent != hdr && H5AC_unpin_entry(parent) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
            parent = NULL;
        } 

        
        if (H5B2__locate_record(hdr->cls, internal->nrec, hdr->nat_off, internal->int_native, udata, &idx,
                                &cmp) < 0) {
            
            H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal, H5AC__NO_FLAGS_SET);
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
        } 

        if (cmp > 0)
            idx++;
        if (cmp != 0) {
            
            next_node_ptr = internal->node_ptrs[idx];

            
            if (H5B2_POS_MIDDLE != curr_pos) {
                if (idx == 0) {
                    if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos)
                        curr_pos = H5B2_POS_LEFT;
                    else
                        curr_pos = H5B2_POS_MIDDLE;
                } 
                else if (idx == internal->nrec) {
                    if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos)
                        curr_pos = H5B2_POS_RIGHT;
                    else
                        curr_pos = H5B2_POS_MIDDLE;
                } 
                else
                    curr_pos = H5B2_POS_MIDDLE;
            } 

            
            if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal,
                               (unsigned)(hdr->swmr_write ? H5AC__PIN_ENTRY_FLAG : H5AC__NO_FLAGS_SET)) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

            
            if (hdr->swmr_write)
                parent = internal;

            
            curr_node_ptr = next_node_ptr;
        } 
        else {
            
            if (op && (op)(H5B2_INT_NREC(internal, hdr, idx), op_data) < 0) {
                
                if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal, H5AC__NO_FLAGS_SET) <
                    0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

                HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL,
                            "'found' callback failed for B-tree find operation");
            } 

            
            if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal, H5AC__NO_FLAGS_SET) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

            
            *found = true;
            HGOTO_DONE(SUCCEED);
        } 

        
        depth--;
    } 

    {
        H5B2_leaf_t *leaf; 

        
        if (NULL == (leaf = H5B2__protect_leaf(hdr, parent, &curr_node_ptr, false, H5AC__READ_ONLY_FLAG)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree leaf node");

        
        if (parent) {
            if (parent != hdr && H5AC_unpin_entry(parent) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
            parent = NULL;
        } 

        
        if (H5B2__locate_record(hdr->cls, leaf->nrec, hdr->nat_off, leaf->leaf_native, udata, &idx, &cmp) <
            0) {
            
            H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr.addr, leaf, H5AC__NO_FLAGS_SET);
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
        } 

        if (cmp != 0) {
            
            if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr.addr, leaf, H5AC__NO_FLAGS_SET) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

            
            *found = false;
            HGOTO_DONE(SUCCEED);
        } 
        else {
            
            if (op && (op)(H5B2_LEAF_NREC(leaf, hdr, idx), op_data) < 0) {
                
                if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr.addr, leaf, H5AC__NO_FLAGS_SET) < 0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

                HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL,
                            "'found' callback failed for B-tree find operation");
            } 

            
            
            if (H5B2_POS_MIDDLE != curr_pos) {
                if (idx == 0) {
                    if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                        if (hdr->min_native_rec == NULL)
                            if (NULL == (hdr->min_native_rec = H5MM_malloc(hdr->cls->nrec_size)))
                                HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL,
                                            "memory allocation failed for v2 B-tree min record info");
                        H5MM_memcpy(hdr->min_native_rec, H5B2_LEAF_NREC(leaf, hdr, idx), hdr->cls->nrec_size);
                    } 
                }     
                if (idx == (unsigned)(leaf->nrec - 1)) {
                    if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                        if (hdr->max_native_rec == NULL)
                            if (NULL == (hdr->max_native_rec = H5MM_malloc(hdr->cls->nrec_size)))
                                HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL,
                                            "memory allocation failed for v2 B-tree max record info");
                        H5MM_memcpy(hdr->max_native_rec, H5B2_LEAF_NREC(leaf, hdr, idx), hdr->cls->nrec_size);
                    } 
                }     
            }         
        }             

        
        if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr.addr, leaf, H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

        
        *found = true;
    } 

done:
    if (parent) {
        assert(ret_value < 0);
        if (parent != hdr && H5AC_unpin_entry(parent) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_index(H5B2_t *bt2, H5_iter_order_t order, hsize_t idx, H5B2_found_t op, void *op_data)
{
    H5B2_hdr_t     *hdr;                 
    H5B2_node_ptr_t curr_node_ptr;       
    void           *parent = NULL;       
    uint16_t        depth;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(bt2);
    assert(op);

    
    bt2->hdr->f = bt2->f;

    
    hdr = bt2->hdr;

    
    curr_node_ptr = hdr->root;

    
    if (curr_node_ptr.node_nrec == 0)
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "B-tree has no records");

    
    if (idx >= curr_node_ptr.all_nrec)
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "B-tree doesn't have that many records");

    
    depth = hdr->depth;

    
    if (hdr->swmr_write)
        parent = hdr;

    
    if (order == H5_ITER_DEC)
        idx = curr_node_ptr.all_nrec - (idx + 1);

    
    while (depth > 0) {
        H5B2_internal_t *internal;      
        H5B2_node_ptr_t  next_node_ptr; 
        unsigned         u;             

        
        if (NULL == (internal = H5B2__protect_internal(hdr, parent, &curr_node_ptr, depth, false,
                                                       H5AC__READ_ONLY_FLAG)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to load B-tree internal node");

        
        if (parent) {
            if (parent != hdr && H5AC_unpin_entry(parent) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
            parent = NULL;
        } 

        
        for (u = 0; u < internal->nrec; u++) {
            
            if (internal->node_ptrs[u].all_nrec > idx) {
                
                next_node_ptr = internal->node_ptrs[u];

                
                if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal,
                                   (unsigned)(hdr->swmr_write ? H5AC__PIN_ENTRY_FLAG : H5AC__NO_FLAGS_SET)) <
                    0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

                
                if (hdr->swmr_write)
                    parent = internal;

                
                curr_node_ptr = next_node_ptr;

                
                break;
            } 

            
            if (internal->node_ptrs[u].all_nrec == idx) {
                
                if ((op)(H5B2_INT_NREC(internal, hdr, u), op_data) < 0) {
                    
                    if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal,
                                       H5AC__NO_FLAGS_SET) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

                    HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL,
                                "'found' callback failed for B-tree find operation");
                } 

                
                if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal, H5AC__NO_FLAGS_SET) <
                    0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

                HGOTO_DONE(SUCCEED);
            } 

            
            idx -= (internal->node_ptrs[u].all_nrec + 1);
        } 

        
        if (u == internal->nrec) {
            
            if (internal->node_ptrs[u].all_nrec > idx) {
                
                next_node_ptr = internal->node_ptrs[u];

                
                if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal,
                                   (unsigned)(hdr->swmr_write ? H5AC__PIN_ENTRY_FLAG : H5AC__NO_FLAGS_SET)) <
                    0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

                
                if (hdr->swmr_write)
                    parent = internal;

                
                curr_node_ptr = next_node_ptr;
            } 
            else
                
                assert(0 && "Index off end of tree??");
        } 

        
        depth--;
    } 

    {
        H5B2_leaf_t *leaf; 

        
        if (NULL == (leaf = H5B2__protect_leaf(hdr, parent, &curr_node_ptr, false, H5AC__READ_ONLY_FLAG)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree leaf node");

        
        if (parent) {
            if (parent != hdr && H5AC_unpin_entry(parent) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
            parent = NULL;
        } 

        
        assert(idx < leaf->nrec);

        
        if ((op)(H5B2_LEAF_NREC(leaf, hdr, idx), op_data) < 0) {
            
            if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr.addr, leaf, H5AC__NO_FLAGS_SET) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

            HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "'found' callback failed for B-tree find operation");
        } 

        
        if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr.addr, leaf, H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");
    } 

done:
    if (parent) {
        assert(ret_value < 0);
        if (parent != hdr && H5AC_unpin_entry(parent) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_remove(H5B2_t *bt2, void *udata, H5B2_remove_t op, void *op_data)
{
    H5B2_hdr_t *hdr;                 
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(bt2);

    
    bt2->hdr->f = bt2->f;

    
    hdr = bt2->hdr;

    
    if (0 == hdr->root.all_nrec)
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "record is not in B-tree");

    
    if (hdr->depth > 0) {
        bool depth_decreased = false; 

        if (H5B2__remove_internal(hdr, &depth_decreased, NULL, NULL, hdr->depth, &(hdr->cache_info), NULL,
                                  H5B2_POS_ROOT, &hdr->root, udata, op, op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to remove record from B-tree internal node");

        
        if (depth_decreased) {
            
            if (hdr->node_info[hdr->depth].nat_rec_fac)
                if (H5FL_fac_term(hdr->node_info[hdr->depth].nat_rec_fac) < 0)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL,
                                "can't destroy node's native record block factory");
            if (hdr->node_info[hdr->depth].node_ptr_fac)
                if (H5FL_fac_term(hdr->node_info[hdr->depth].node_ptr_fac) < 0)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL,
                                "can't destroy node's node pointer block factory");

            assert((uint16_t)(hdr->depth - depth_decreased) < hdr->depth);
            hdr->depth = (uint16_t)(hdr->depth - depth_decreased);
        } 
    }     
    else {
        if (H5B2__remove_leaf(hdr, &hdr->root, H5B2_POS_ROOT, hdr, udata, op, op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to remove record from B-tree leaf node");
    } 

    
    hdr->root.all_nrec--;

    
    if (H5B2__hdr_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTMARKDIRTY, FAIL, "unable to mark B-tree header dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_remove_by_idx(H5B2_t *bt2, H5_iter_order_t order, hsize_t idx, H5B2_remove_t op, void *op_data)
{
    H5B2_hdr_t *hdr;                 
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(bt2);

    
    bt2->hdr->f = bt2->f;

    
    hdr = bt2->hdr;

    
    if (0 == hdr->root.all_nrec)
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "record is not in B-tree");

    
    if (idx >= hdr->root.all_nrec)
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "B-tree doesn't have that many records");

    
    if (H5_ITER_DEC == order)
        idx = hdr->root.all_nrec - (idx + 1);

    
    if (hdr->depth > 0) {
        bool depth_decreased = false; 

        if (H5B2__remove_internal_by_idx(hdr, &depth_decreased, NULL, NULL, hdr->depth, &(hdr->cache_info),
                                         NULL, &hdr->root, H5B2_POS_ROOT, idx, op, op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to remove record from B-tree internal node");

        
        if (depth_decreased) {
            
            if (hdr->node_info[hdr->depth].nat_rec_fac)
                if (H5FL_fac_term(hdr->node_info[hdr->depth].nat_rec_fac) < 0)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL,
                                "can't destroy node's native record block factory");
            if (hdr->node_info[hdr->depth].node_ptr_fac)
                if (H5FL_fac_term(hdr->node_info[hdr->depth].node_ptr_fac) < 0)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL,
                                "can't destroy node's node pointer block factory");

            assert((uint16_t)(hdr->depth - depth_decreased) < hdr->depth);
            hdr->depth = (uint16_t)(hdr->depth - depth_decreased);
        } 
    }     
    else {
        if (H5B2__remove_leaf_by_idx(hdr, &hdr->root, H5B2_POS_ROOT, hdr, (unsigned)idx, op, op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to remove record from B-tree leaf node");
    } 

    
    hdr->root.all_nrec--;

    
    if (H5B2__hdr_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTMARKDIRTY, FAIL, "unable to mark B-tree header dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_get_nrec(const H5B2_t *bt2, hsize_t *nrec)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(bt2);
    assert(nrec);

    
    *nrec = bt2->hdr->root.all_nrec;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5B2_neighbor(H5B2_t *bt2, H5B2_compare_t range, void *udata, H5B2_found_t op, void *op_data)
{
    H5B2_hdr_t *hdr;                 
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(bt2);
    assert(op);

    
    bt2->hdr->f = bt2->f;

    
    hdr = bt2->hdr;

    
    if (!H5_addr_defined(hdr->root.addr))
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "B-tree has no records");

    
    if (hdr->depth > 0) {
        if (H5B2__neighbor_internal(hdr, hdr->depth, &hdr->root, NULL, range, hdr, udata, op, op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL,
                        "unable to find neighbor record in B-tree internal node");
    } 
    else {
        if (H5B2__neighbor_leaf(hdr, &hdr->root, NULL, range, hdr, udata, op, op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "unable to find neighbor record in B-tree leaf node");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_modify(H5B2_t *bt2, void *udata, bool try, H5B2_modify_t op, void *op_data)
{
    H5B2_hdr_t     *hdr;                 
    H5B2_node_ptr_t curr_node_ptr;       
    void           *parent = NULL;       
    H5B2_nodepos_t  curr_pos;            
    uint16_t        depth;               
    int             cmp;                 
    unsigned        idx;                 
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(bt2);
    assert(op);

    
    bt2->hdr->f = bt2->f;

    
    hdr = bt2->hdr;

    
    curr_node_ptr = hdr->root;

    
    if (0 == curr_node_ptr.node_nrec) {
        
        if (try)
            HGOTO_DONE(SUCCEED);
        else
            HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "B-tree has no records");
    } 

    
    depth = hdr->depth;

    
    if (hdr->swmr_write)
        parent = hdr;

    
    cmp      = -1;
    curr_pos = H5B2_POS_ROOT;
    while (depth > 0) {
        unsigned         internal_flags = H5AC__NO_FLAGS_SET;
        H5B2_internal_t *internal;      
        H5B2_node_ptr_t  next_node_ptr; 

        
        if (NULL == (internal = H5B2__protect_internal(hdr, parent, &curr_node_ptr, depth, false,
                                                       H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to load B-tree internal node");

        
        if (parent) {
            if (parent != hdr && H5AC_unpin_entry(parent) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
            parent = NULL;
        } 

        
        if (H5B2__locate_record(hdr->cls, internal->nrec, hdr->nat_off, internal->int_native, udata, &idx,
                                &cmp) < 0) {
            
            H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal, H5AC__NO_FLAGS_SET);
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
        } 

        if (cmp > 0)
            idx++;

        if (cmp != 0) {
            
            next_node_ptr = internal->node_ptrs[idx];

            
            if (H5B2_POS_MIDDLE != curr_pos) {
                if (idx == 0) {
                    if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos)
                        curr_pos = H5B2_POS_LEFT;
                    else
                        curr_pos = H5B2_POS_MIDDLE;
                } 
                else if (idx == internal->nrec) {
                    if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos)
                        curr_pos = H5B2_POS_RIGHT;
                    else
                        curr_pos = H5B2_POS_MIDDLE;
                } 
                else
                    curr_pos = H5B2_POS_MIDDLE;
            } 

            
            if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal,
                               (unsigned)(hdr->swmr_write ? H5AC__PIN_ENTRY_FLAG : H5AC__NO_FLAGS_SET)) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

            
            if (hdr->swmr_write)
                parent = internal;

            
            curr_node_ptr = next_node_ptr;
        } 
        else {
            bool changed; 

            
            if ((op)(H5B2_INT_NREC(internal, hdr, idx), op_data, &changed) < 0) {
                
                assert(changed == false);

                
                if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal, H5AC__NO_FLAGS_SET) <
                    0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

                HGOTO_ERROR(H5E_BTREE, H5E_CANTMODIFY, FAIL,
                            "'modify' callback failed for B-tree find operation");
            } 

            
            internal_flags |= changed ? H5AC__DIRTIED_FLAG : 0;

            
            if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr.addr, internal, internal_flags) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

            HGOTO_DONE(SUCCEED);
        } 

        
        depth--;
    } 

    {
        H5B2_leaf_t *leaf;                            
        unsigned     leaf_flags = H5AC__NO_FLAGS_SET; 
        bool         changed    = false;              

        
        if (NULL == (leaf = H5B2__protect_leaf(hdr, parent, &curr_node_ptr, false, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree leaf node");

        
        if (parent) {
            if (parent != hdr && H5AC_unpin_entry(parent) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
            parent = NULL;
        } 

        
        if (H5B2__locate_record(hdr->cls, leaf->nrec, hdr->nat_off, leaf->leaf_native, udata, &idx, &cmp) <
            0) {
            
            H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr.addr, leaf, H5AC__NO_FLAGS_SET);
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
        } 

        if (cmp != 0) {
            
            if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr.addr, leaf, H5AC__NO_FLAGS_SET) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

            
            if (try)
                HGOTO_DONE(SUCCEED);
            else
                HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "record not found");
        }
        else {
            
            if ((op)(H5B2_LEAF_NREC(leaf, hdr, idx), op_data, &changed) < 0) {
                
                assert(changed == false);

                
                if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr.addr, leaf, H5AC__NO_FLAGS_SET) < 0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");

                HGOTO_ERROR(H5E_BTREE, H5E_CANTMODIFY, FAIL,
                            "'modify' callback failed for B-tree find operation");
            } 

            
            
            if (H5B2_POS_MIDDLE != curr_pos) {
                if (idx == 0) {
                    if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                        if (hdr->min_native_rec == NULL)
                            if (NULL == (hdr->min_native_rec = H5MM_malloc(hdr->cls->nrec_size)))
                                HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL,
                                            "memory allocation failed for v2 B-tree min record info");
                        H5MM_memcpy(hdr->min_native_rec, H5B2_LEAF_NREC(leaf, hdr, idx), hdr->cls->nrec_size);
                    } 
                }     
                if (idx == (unsigned)(leaf->nrec - 1)) {
                    if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                        if (hdr->max_native_rec == NULL)
                            if (NULL == (hdr->max_native_rec = H5MM_malloc(hdr->cls->nrec_size)))
                                HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL,
                                            "memory allocation failed for v2 B-tree max record info");
                        H5MM_memcpy(hdr->max_native_rec, H5B2_LEAF_NREC(leaf, hdr, idx), hdr->cls->nrec_size);
                    } 
                }     
            }         
        }             

        
        leaf_flags |= (changed ? H5AC__DIRTIED_FLAG : 0);

        
        if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr.addr, leaf, leaf_flags) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree node");
    } 

done:
    if (parent) {
        assert(ret_value < 0);
        if (parent != hdr && H5AC_unpin_entry(parent) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin parent entry");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_close(H5B2_t *bt2)
{
    haddr_t bt2_addr       = HADDR_UNDEF; 
    bool    pending_delete = false;       
    herr_t  ret_value      = SUCCEED;     

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(bt2);
    assert(bt2->f);

    
    if (0 == H5B2__hdr_fuse_decr(bt2->hdr)) {
        
        bt2->hdr->f = bt2->f;

        
        if (bt2->hdr->pending_delete) {
            
            pending_delete = true;
            bt2_addr       = bt2->hdr->addr;
        } 
    }     

    
    if (pending_delete) {
        H5B2_hdr_t *hdr; 

        
        assert(H5_addr_defined(bt2_addr));

#ifndef NDEBUG
        {
            unsigned hdr_status = 0; 

            
            if (H5AC_get_entry_status(bt2->f, bt2_addr, &hdr_status) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL,
                            "unable to check metadata cache status for v2 B-tree header, address = %llu",
                            (unsigned long long)bt2_addr);

            
            assert(hdr_status & H5AC_ES__IN_CACHE);
            assert(hdr_status & H5AC_ES__IS_PINNED);
            assert(!(hdr_status & H5AC_ES__IS_PROTECTED));
        }
#endif 

        
        
        if (NULL == (hdr = H5B2__hdr_protect(bt2->f, bt2_addr, NULL, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect v2 B-tree header");

        
        hdr->f = bt2->f;

        
        
        if (H5B2__hdr_decr(bt2->hdr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on shared v2 B-tree header");

        
        if (H5B2__hdr_delete(hdr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to delete v2 B-tree");
    } 
    else {
        
        
        if (H5B2__hdr_decr(bt2->hdr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on shared v2 B-tree header");

    } 

    
    bt2 = H5FL_FREE(H5B2_t, bt2);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_delete(H5F_t *f, haddr_t addr, void *ctx_udata, H5B2_remove_t op, void *op_data)
{
    H5B2_hdr_t *hdr       = NULL;    
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));

    
    if (NULL == (hdr = H5B2__hdr_protect(f, addr, ctx_udata, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect v2 B-tree header");

    
    hdr->remove_op      = op;
    hdr->remove_op_data = op_data;

    
    if (hdr->file_rc)
        hdr->pending_delete = true;
    else {
        
        hdr->f = f;

        
        if (H5B2__hdr_delete(hdr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to delete v2 B-tree");
        hdr = NULL;
    } 

done:
    
    if (hdr && H5B2__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release v2 B-tree header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_depend(H5B2_t *bt2, H5AC_proxy_entry_t *parent)
{
    
    H5B2_hdr_t *hdr       = bt2->hdr; 
    herr_t      ret_value = SUCCEED;  

    FUNC_ENTER_NOAPI(SUCCEED)

    
    assert(bt2);
    assert(hdr);
    assert(parent);
    assert(hdr->parent == NULL || hdr->parent == parent);

    
    if (NULL == hdr->parent) {
        
        assert(hdr->top_proxy);

        
        hdr->f = bt2->f;

        
        if (H5AC_proxy_entry_add_child(parent, hdr->f, hdr->top_proxy) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTSET, FAIL, "unable to add v2 B-tree as child of proxy");
        hdr->parent = parent;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2_patch_file(H5B2_t *bt2, H5F_t *f)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(bt2);
    assert(f);

    if (bt2->f != f || bt2->hdr->f != f)
        bt2->f = bt2->hdr->f = f;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
