% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayes.r
\name{ibrm}
\alias{ibrm}
\title{Bayes model}
\usage{
ibrm(
  formula,
  data = NULL,
  M = NULL,
  M.id = NULL,
  method = c("BayesCpi", "BayesA", "BayesL", "BSLMM", "BayesR", "BayesB", "BayesC",
    "BayesBpi", "BayesRR"),
  map = NULL,
  Pi = NULL,
  fold = NULL,
  niter = 20000,
  nburn = 12000,
  thin = 5,
  windsize = NULL,
  windnum = NULL,
  dfvr = NULL,
  s2vr = NULL,
  vg = NULL,
  dfvg = NULL,
  s2vg = NULL,
  ve = NULL,
  dfve = NULL,
  s2ve = NULL,
  lambda = 0,
  printfreq = 100,
  seed = 666666,
  threads = 4,
  verbose = TRUE
)
}
\arguments{
\item{formula}{a two-sided linear formula object describing both the fixed-effects and random-effects part of the model, with the response on the left of a ‘~’ operator and the terms, separated by ‘+’ operators, on the right. Random-effects terms are distinguished by vertical bars (1|’) separating expressions for design matrices from grouping factors.}

\item{data}{the data frame containing the variables named in 'formula', NOTE that the first column in 'data' should be the individual id.}

\item{M}{numeric matrix of genotype with individuals in rows and markers in columns, NAs are not allowed.}

\item{M.id}{vector of id for genotyped individuals, NOTE that no need to adjust the order of id to be the same between 'data' and 'M', the package will do it automatically.}

\item{method}{bayes methods including: "BayesB", "BayesA", "BayesL", "BayesRR", "BayesBpi", "BayesC", "BayesCpi", "BayesR", "BSLMM".
\itemize{
\item "BayesRR": Bayes Ridge Regression, all SNPs have non-zero effects and share the same variance, equals to RRBLUP or GBLUP.
\item "BayesA": all SNPs have non-zero effects, and take different variance which follows an inverse chi-square distribution.
\item "BayesB": only a small proportion of SNPs (1-Pi) have non-zero effects, and take different variance which follows an inverse chi-square distribution.
\item "BayesBpi": the same with "BayesB", but 'Pi' is not fixed.
\item "BayesC": only a small proportion of SNPs (1-Pi) have non-zero effects, and share the same variance.
\item "BayesCpi": the same with "BayesC", but 'Pi' is not fixed.
\item "BayesL": BayesLASSO, all SNPs have non-zero effects, and take different variance which follows an exponential distribution.
\item "BSLMM": all SNPs have non-zero effects, and take the same variance, but a small proportion of SNPs have additional shared variance.
\item "BayesR": only a small proportion of SNPs have non-zero effects, and the SNPs are allocated into different groups, each group has the same variance.
}}

\item{map}{(optional, only for GWAS) the map information of genotype, at least 3 columns are: SNPs, chromosome, physical position.}

\item{Pi}{vector, the proportion of zero effect and non-zero effect SNPs, the first value must be the proportion of non-effect markers.}

\item{fold}{proportion of variance explained for groups of SNPs, the default is c(0, 0.0001, 0.001, 0.01).}

\item{niter}{the number of MCMC iteration.}

\item{nburn}{the number of iterations to be discarded.}

\item{thin}{the number of thinning after burn-in. Note that smaller thinning frequency may have higher accuracy of estimated parameters, but would result in more memory for collecting process, on contrary, bigger frequency may have negative effect on accuracy of estimations.}

\item{windsize}{window size in bp for GWAS, the default is NULL.}

\item{windnum}{fixed number of SNPs in a window for GWAS, if it is specified, 'windsize' will be invalid, the default is NULL.}

\item{dfvr}{the number of degrees of freedom for the distribution of environmental variance.}

\item{s2vr}{scale parameter for the distribution of environmental variance.}

\item{vg}{prior value of genetic variance.}

\item{dfvg}{the number of degrees of freedom for the distribution of genetic variance.}

\item{s2vg}{scale parameter for the distribution of genetic variance.}

\item{ve}{prior value of residual variance.}

\item{dfve}{the number of degrees of freedom for the distribution of residual variance.}

\item{s2ve}{scale parameter for the distribution of residual variance.}

\item{lambda}{value of ridge regression for inverting a matrix.}

\item{printfreq}{frequency of printing iterative details on console.}

\item{seed}{seed for random sample.}

\item{threads}{number of threads used for OpenMP.}

\item{verbose}{whether to print the iteration information on console.}
}
\value{
the function returns a 'blrMod' object containing
\describe{
\item{$mu}{the regression intercept}
\item{$pi}{estimated proportion of zero effect and non-zero effect SNPs}
\item{$beta}{estimated coefficients for all covariates}
\item{$r}{estimated environmental random effects}
\item{$Vr}{estimated variance for all environmental random effect}
\item{$Vg}{estimated genetic variance}
\item{$Ve}{estimated residual variance}
\item{$h2}{estimated heritability (h2 = Vg / (Vr + Vg + Ve))}
\item{$alpha}{estimated effect size of all markers}
\item{$g}{genomic estimated breeding value}
\item{$e}{residuals of the model}
\item{$pip}{the frequency for markers to be included in the model during MCMC iteration, known as posterior inclusive probability (PIP)}
\item{$gwas}{WPPA is defined to be the window posterior probability of association, it is estimated by counting the number of MCMC samples in which \deqn{\alpha} is nonzero for at least one SNP in the window}
\item{$MCMCsamples}{the collected samples of posterior estimation for all the above parameters across MCMC iterations}
}
}
\description{
Bayes linear regression model using individual level data
\deqn{y = X \beta + R r + M \alpha + e}
where \eqn{\beta} is a vector of estimated coefficient for covariates, and \eqn{r} is a vector of environmental random effects. \eqn{M} is a matrix of genotype covariate, \eqn{\alpha} is a vector of estimated marker effect size. \eqn{e} is a vector of residuals.
}
\details{
\itemize{
\item{the fixed effects and covariates in 'formula' must be in factors and numeric, respectively. if not, please remember to use 'as.factor' and 'as.numeric' to transform.}
\item{the package has the automatical function of taking the intersection and adjusting the order of id between 'data' and the genotype 'M', thus the first column in 'data' should be the individual id.}
\item{if any one of the options 'windsize' and 'windnum' is specified, the GWAS results will be returned, and the 'map' information must be provided, in which the physical positions should be all in digital values.}
\item{the 'windsize' or 'windnum' option only works for the methods of which the assumption has a proportion of zero effect markers, e.g., BayesB, BayesBpi, BayesC, BayesCpi, BSLMM, and BayesR.}
}
}
\examples{
# Load the example data attached in the package
pheno_file_path = system.file("extdata", "demo.phe", package = "hibayes")
pheno = read.table(pheno_file_path, header=TRUE)

bfile_path = system.file("extdata", "demo", package = "hibayes")
bin = read_plink(bfile_path, threads=1)
fam = bin$fam
geno = bin$geno
map = bin$map

# For GS/GP
## no environmental effects:
fit = ibrm(T1~1, data=pheno, M=geno, M.id=fam[,2], method="BayesCpi",
	niter=2000, nburn=1200, thin=5, threads=1)

## overview of the returned results
summary(fit)

\donttest{

## add fixed effects or covariates:
fit = ibrm(T1~sex+season+day+bwt, data=pheno, M=geno, M.id=fam[,2],
	method="BayesCpi")
 
## add environmental random effects:
fit = ibrm(T1~sex+(1|loc)+(1|dam), data=pheno, M=geno, M.id=fam[,2],
	method="BayesCpi")

# For GWAS
fit = ibrm(T1~sex+bwt+(1|dam), data=pheno, M=geno, M.id=fam[,2],
	method="BayesCpi", map=map, windsize=1e6)
}

# get the SD of estimated SNP effects for markers
summary(fit)$alpha
# get the SD of estimated breeding values
summary(fit)$g

}
\references{
Meuwissen, Theo HE, Ben J. Hayes, and Michael E. Goddard. "Prediction of total genetic value using genome-wide dense marker maps." Genetics 157.4 (2001): 1819-1829. \cr
de los Campos, G., Hickey, J. M., Pong-Wong, R., Daetwyler, H. D., and Calus, M. P. (2013). Whole-genome regression and prediction methods applied to plant and animal breeding. Genetics, 193(2), 327-345. \cr
Habier, David, et al. "Extension of the Bayesian alphabet for genomic selection." BMC bioinformatics 12.1 (2011): 1-12. \cr
Yi, Nengjun, and Shizhong Xu. "Bayesian LASSO for quantitative trait loci mapping." Genetics 179.2 (2008): 1045-1055. \cr
Zhou, Xiang, Peter Carbonetto, and Matthew Stephens. "Polygenic modeling with Bayesian sparse linear mixed models." PLoS genetics 9.2 (2013): e1003264. \cr
Moser, Gerhard, et al. "Simultaneous discovery, estimation and prediction analysis of complex traits using a Bayesian mixture model." PLoS genetics 11.4 (2015): e1004969. \cr
}
