% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simgenot_new.R
\name{sim.genot.metapop.t}
\alias{sim.genot.metapop.t}
\title{Simulate genetic data from a metapopulation model}
\usage{
sim.genot.metapop.t(size=50,nbal=4,nbloc=5,nbpop=3,N=1000,
mig=diag(3),mut=0.0001,f=0,t=100)
}
\arguments{
\item{size}{the number of sampled individuals per population}

\item{nbal}{the number of alleles per locus (maximum of 99)}

\item{nbloc}{the number of loci to simulate}

\item{nbpop}{the number of populations to simulate}

\item{N}{the effective population sizes of each population. If only one number, all
populations are assumed to be of the same size}

\item{mig}{a matrix with nbpop rows and columns giving the migration rate 
from population i (in row) to population j (in column). Each row must sum to 1.}

\item{mut}{the mutation rate of the loci}

\item{f}{the inbreeding coefficient for each population}

\item{t}{the number of generation since the islands were created}
}
\value{
A data frame with size*nbpop rows and nbloc+1 columns. Each row is an
individual, the first column contains the identifier of the population to which the individual belongs, 
the following nbloc columns contain the genotype for each locus.
}
\description{
This function allows to simulate genetic data from a metapopulation
 model, where each population can have a different size and a different inbreeding 
 coefficient, and migration between each population is given in a migration matrix. 
 
 This function simulates genetic data under a migration matrix model.
Each population \eqn{i} sends a proportion of migrant alleles \eqn{m_{ij}} to population \eqn{j} 
and receives a proportion of migrant alleles \eqn{m_{ji}} from population \eqn{j}.
}
\details{
In this model, \eqn{\theta_t} can be written as a function of population size 
\eqn{N_i}, migration rate \eqn{m_{ij}}, mutation rate \eqn{\mu} and \eqn{\theta_{(t-1)}}.

The rational is as follows:
 
 With probability \eqn{\frac{1}{N_i}}, 2 alleles from 2 different individuals in 
 the current generation are sampled from the same individual of the previous 
 generation:     
 
 -Half the time, the same allele is drawn from the parent;
  
 -The other half, two different alleles are drawn, but they are identical in 
 proportion \eqn{\theta_{(t-1)}}.
  
 -With  probability \eqn{1-\frac{1}{N_i}}, the 2 alleles are drawn from different 
 individuals in the previous generation, in which case they are identical in 
 proportion \eqn{\theta_{(t-1)}}.
 
 This holds providing that neither alleles have mutated or migrated.  This is 
 the case with probability \eqn{m_{ii}^2 \times (1-\mu)^2}.
 If an allele is a mutant, then its coancestry with another allele 
 is 0.

Note also that the mutation scheme assumed is the infinite allele (or site) 
model.  If the number of alleles is finite (as will be the case in what follows),
the corresponding mutation model is the K-allele model and the mutation rate 
has to be adjusted to \eqn{\mu'=\frac{K-1}{K}\mu}.

Continue derivation
}
\examples{

#2 populations
psize<-c(10,1000)
mig.mat<-matrix(c(0.99,0.01,0.1,0.9),nrow=2,byrow=TRUE)
dat<-sim.genot.metapop.t(nbal=10,nbloc=100,nbpop=2,N=psize,mig=mig.mat,mut=0.00001,t=100)
betas(dat)$betaiovl # Population specific estimator of FST

#1D stepping stone
\dontrun{
np<-10
m<-0.2
mig.mat<-diag(np)*(1-m)
diag(mig.mat[-1,-np])<-m/2
diag(mig.mat[-np,-1])<-m/2
mig.mat[1,1:2]<-c(1-m/2,m/2)
mig.mat[np,(np-1):np]<-c(m/2,1-m/2)
dat<-sim.genot.metapop.t(nbal=10,nbloc=50,nbpop=np,mig=mig.mat,t=400)
pcoa(as.matrix(genet.dist(dat))) # principal coordinates plot
}

}
\author{
Jerome Goudet \email{jerome.goudet@unil.ch}
}
