% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{hpaML}
\alias{hpaML}
\title{Semi-nonparametric maximum likelihood estimation}
\usage{
hpaML(
  x,
  pol_degrees = numeric(0),
  tr_left = numeric(0),
  tr_right = numeric(0),
  given_ind = logical(0),
  omit_ind = logical(0),
  x0 = numeric(0),
  cov_type = "sandwich",
  boot_iter = 100L,
  is_parallel = FALSE,
  opt_type = "optim",
  opt_control = NULL
)
}
\arguments{
\item{x}{numeric matrix which rows are realizations of independent 
identically distributed random vectors while columns correspond to
variables.}

\item{pol_degrees}{non-negative integer vector of polynomial degrees.}

\item{tr_left}{numeric matrix of left (lower) truncation limits.
Note that \code{tr_right} rows are observations while variables are columns.
If \code{tr_left} or \code{tr_right} is single row matrix then the same 
truncation limits would be applied to all observations that are determined 
by the first rows of these matrices.}

\item{tr_right}{numeric matrix of right (upper) truncation limits.
Note that \code{tr_right} rows are observations while variables are columns.
If \code{tr_left} or \code{tr_right} is single row matrix then the same 
truncation limits would be applied to all observations that are determined 
by the first rows of these matrices.}

\item{given_ind}{logical vector indicating wheather corresponding
component is conditioned. By default it is a logical vector of FALSE values.}

\item{omit_ind}{logical vector indicating wheather corresponding
component is omitted. By default it is a logical vector of FALSE values.}

\item{x0}{numeric vector of optimization routine initial values.
Note that \code{x0=c(pol_coefficients[-1], mean, sd)}. For 
\code{pol_coefficients}, \code{mean} and \code{sd} documentation 
see \code{\link[hpa]{dhpa}} function.}

\item{cov_type}{string value determining the type of covariance matrix to be
returned and used for summary. If \code{cov_type = "hessian"} then negative
inverse of Hessian matrix will be applied. If \code{cov_type = "gop"} then
inverse of Jacobian outer products will be used.
If \code{cov_type = "sandwich"} (default) then sandwich covariance matrix
estimator will be applied. If \code{cov_type = "bootstrap"} then bootstrap
with \code{boot_iter} iterations will be used.
If \code{cov_type = "hessianFD"} or \code{cov_type = "sandwichFD"} then
accurate but computationally demanding central difference Hessian
approximation will be calculated for the inverse Hessian and sandwich
estimators correspondingly. Central differences are computed via
analytically provided gradient. This Hessian matrix estimation approach
seems to be less accurate then BFGS approximation if polynomial order
is high (usually greater then 5).}

\item{boot_iter}{the number of bootstrap iterations
for \code{cov_type = "bootstrap"} covariance matrix estimator type.}

\item{is_parallel}{if TRUE then multiple cores will be
used for some calculations. It usually provides speed advantage for
large enough samples (about more than 1000 observations).}

\item{opt_type}{string value determining the type of the optimization
routine to be applied. The default is "optim" meaning that BFGS method
from the \code{\link[stats]{optim}} function will be applied.
If \code{opt_type = "GA"} then \code{\link[GA]{ga}} function will be
additionally applied.}

\item{opt_control}{a list containing arguments to be passed to the
optimization routine depending on \code{opt_type} argument value.
Please see details to get additional information.}
}
\value{
This function returns an object of class "hpaML".\cr \cr
An object of class "hpaML" is a list containing the following components:
\itemize{
\item \code{optim} - \code{\link[stats]{optim}} function output. 
If \code{opt_type = "GA"} then it is the list containing 
\code{\link[stats]{optim}} and \code{\link[GA]{ga}} functions outputs.
\item \code{x1} - numeric vector of distribution parameters estimates.
\item \code{mean} - density function mean vector estimate.
\item \code{sd} - density function sd vector estimate.
\item \code{pol_coefficients} - polynomial coefficients estimates.
\item \code{tr_left }- the same as \code{tr_left} input parameter.
\item \code{tr_right} - the same as \code{tr_right} input parameter.
\item \code{omit_ind }- the same as \code{omit_ind} input parameter.
\item \code{given_ind} - the same as \code{given_ind} input parameter.
\item \code{cov_mat} - covariance matrix estimate.
\item \code{results} - numeric matrix representing estimation results.
\item \code{log-likelihood} - value of Log-Likelihood function.
\item \code{AIC} - AIC value.
\item \code{data} - the same as \code{x} input parameter but without \code{NA} observations.
\item \code{n_obs} - number of observations.
\item \code{bootstrap} - list where bootstrap estimation results are stored.}
}
\description{
This function performs semi-nonparametric maximum likelihood estimation
via hermite polynomial densities approximation.
}
\details{
Semi-nonparametric (SNP) approach has been implemented via
densities hermite polynomial approximation

Densities hermite polynomial approximation approach has been
proposed by A. Gallant and D. W. Nychka in 1987. The main idea is to
approximate unknown distribution density with hermite polynomial of  
degree \code{pol_degree}. In this framework hermite polynomial represents
adjusted (to insure integration to 1) product of squared polynomial and 
normal distribution densities. Parameters \code{mean} and \code{sd} 
determine means and standard deviations of normal distribution density
functions which are parts of this polynomial.
For more information please refer to the literature listed below.

Parameters \code{mean}, \code{sd}, \code{given_ind},
\code{omit_ind} should have the same length as \code{pol_degrees} 
parameter.

The first polynomial coefficient (zero powers) 
set to 1 for identification reasons.

The function calculates standard errors via sandwich estimator
and significance levels are reported taking into account quasi maximum
likelihood estimator (QMLE) asymptotic normality. If ones wants to switch
from QMLE to semi-nonparametric estimator (SNPE) during hypothesis testing
then covariance matrix should be estimated again using bootstrap.

This function maximizes (quasi) log-likelihood function 
via \code{\link[stats]{optim}} function setting it's \code{method} 
argument to "BFGS". If \code{opt_type = "GA"} then genetic
algorithm from \code{\link[GA]{ga}} function
will be additionally (after \code{\link[stats]{optim}} putting it's
solution (\code{par}) to \code{suggestions} matrix) applied in order to 
perform global optimization. Note that global optimization takes
much more time (usually minutes but sometimes hours or even days). 
The number of iterations and population size of the genetic algorithm
will grow linearly along with the number of estimated parameters. 
If it is seems that global maximum has not been found than it
is possible to continue the search restarting the function setting 
it's input argument \code{x0} to \code{x1} output value. Note that
if \code{cov_type = "bootstrap"} then \code{\link[GA]{ga}}
function will not be used for bootstrap iterations since it
may be extremely time consuming.

If \code{opt_type = "GA"} then \code{opt_control} should be the
list containing the values to be passed to \code{\link[GA]{ga}}
function. It is possible to pass arguments \code{lower}, \code{upper},
\code{popSize}, \code{pcrossover}, \code{pmutation}, \code{elitism},
\code{maxiter}, \code{suggestions}, \code{optim}, \code{optimArgs},
\code{seed} and \code{monitor}. 
Note that it is possible to set \code{population},
\code{selection}, \code{crossover} and \code{mutation} arguments changing
\code{\link[GA]{ga}} default parameters via \code{\link[GA]{gaControl}} 
function. These arguments information reported in \code{\link[GA]{ga}}.
In order to provide manual values for \code{lower} and \code{upper} bounds
please follow parameters ordering mentioned above for the
\code{x0} argument. If these bonds are not provided manually then
they (except those related to the polynomial coefficients)
will depend on the estimates obtained
by local optimization via \code{\link[stats]{optim}} function
(this estimates will be in the middle
between \code{lower} and \code{upper}).
Specifically for each sd parameter \code{lower} (\code{upper}) bound
is 5 times lower (higher) then this
parameter \code{\link[stats]{optim}} estimate.
For each mean and regression coefficient parameter it's lower and upper
bounds deviate from corresponding \code{\link[stats]{optim}} estimate
by two absolute value of this estimate.
Finally, lower and upper bounds for each polynomial
coefficient are \code{-10} and \code{10} correspondingly (do not depend
on their \code{\link[stats]{optim}} estimates).

The following arguments are differ from their defaults in
\code{\link[GA]{ga}}:
\itemize{
\item \code{pmutation = 0.2},
\item \code{optim = TRUE},
\item \code{optimArgs =
list("method" = "Nelder-Mead", "poptim" = 0.2, "pressel" = 0.5)},
\item \code{seed = 8},
\item \code{elitism = 2 + round(popSize * 0.1)}.}

The arguments \code{popSize} and \code{maxiter} of
\code{\link[GA]{ga}} function have been set proportional to the number of
estimated polynomial coefficients
\itemize{
\item \code{popSize = 10 + (prod(pol_degrees + 1) - 1) * 2}.
\item \code{maxiter = 50 * (prod(pol_degrees + 1))}}
}
\examples{
\donttest{
## Approximate student (t) distribution

# Set seed for reproducibility
set.seed(123)

# Simulate 5000 realizations of student distribution with 5 degrees of freedom
 n <- 5000
df <- 5
x <- matrix(rt(n, df), ncol = 1)
pol_degrees <- c(4)

# Apply pseudo maximum likelihood routine
ml_result <- hpa::hpaML(x = x, pol_degrees = pol_degrees)
summary(ml_result)

# Get predicted probabilites (density values) approximations
predict(ml_result)

# Plot density approximation
plot(ml_result)

## Approximate chi-squared distribution

# Set seed for reproducibility
set.seed(123)

# Simulate 5000 realizations of chi-squared distribution with 5 degrees of freedom

n <- 5000
df <- 5
x <- matrix(rchisq(n, df), ncol = 1)
pol_degrees <- c(5)

# Apply pseudo maximum likelihood routine
ml_result <- hpaML(x = x, pol_degrees = as.vector(pol_degrees), 
				tr_left = 0)
summary(ml_result)

# Get predicted probabilites (density values) approximations
predict(ml_result)

# Plot density approximation
plot(ml_result)

## Approximate multivariate student (t) distribution
## Note that calculations may take up to a minute

# Set seed for reproducibility
set.seed(123)

# Simulate 5000 realizations of three dimensional student distribution with 5 degrees of freedom
library("mvtnorm")
cov_mat <- matrix(c(1, 0.5, -0.5, 0.5, 1, 0.5, -0.5, 0.5, 1), ncol = 3)
x <- rmvt(n = 5000, sigma = cov_mat, df = 5)

# Estimate approximating joint distribution parameters
ml_result <- hpaML(x = x, pol_degrees = c(1, 1, 1))

# Get summary
summary(ml_result)

# Get predicted values for joint density function
predict(ml_result)

# Plot density approximation for the
# second random variable
plot(ml_result, ind = 2)

# Plot density approximation for the
# second random variable conditioning
# on x1 = 1
plot(ml_result, ind = 2, given = c(1, NA, NA))

## Approximate student (t) distribution and plot densities approximated
## under different hermite polynomial degrees against 
## true density (of student distribution)

# Simulate 5000 realizations of t-distribution with 5 degrees of freedom
n <- 5000
df <- 5
x <- matrix(rt(n, df), ncol=1)

# Apply pseudo maximum likelihood routine
# Create matrix of lists where i-th element contains hpaML results for K=i
ml_result <- matrix(list(), 4, 1)
for(i in 1:4)
{
 ml_result[[i]] <- hpa::hpaML(x = x, pol_degrees = i)
}

# Generate test values
test_values <- seq(qt(0.001, df), qt(0.999, df), 0.001)
n0 <- length(test_values)

# t-distribution density function at test values points
true_pred <- dt(test_values, df)

# Create matrix of lists where i-th element contains densities predictions for K=i
PGN_pred <- matrix(list(), 4, 1)
for(i in 1:4)
{
  PGN_pred[[i]] <- predict(object = ml_result[[i]], 
                           newdata = matrix(test_values, ncol=1))
}
# Plot the result
library("ggplot2")

# prepare the data
h <- data.frame("values" = rep(test_values,5),
                "predictions" = c(PGN_pred[[1]],PGN_pred[[2]],
                                  PGN_pred[[3]],PGN_pred[[4]],
                                  true_pred), 
                "Density" = c(
                  rep("K=1",n0), rep("K=2",n0),
                  rep("K=3",n0), rep("K=4",n0),
                  rep("t-distribution",n0))
                  )
                  
# build the plot
ggplot(h, aes(values, predictions)) + geom_point(aes(color = Density)) +
  theme_minimal() + theme(legend.position = "top", text = element_text(size=26),
                         legend.title=element_text(size=20), legend.text=element_text(size=28)) +
  guides(colour = guide_legend(override.aes = list(size=10))
  )

# Get informative estimates summary for K=4
summary(ml_result[[4]])
}
\dontshow{
## Approximate student (t) distribution

# Simulate 250 realizations of student distribution with 5 degrees of freedom
 set.seed(2)
 n <- 250
df <- 5
x <- matrix(rt(n, df), ncol = 1)
pol_degrees <- c(2)

# Apply pseudo maximum likelihood routine
ml_result <- hpa::hpaML(x = x, pol_degrees = pol_degrees)
 summary(ml_result)

# Get predicted probabilites (density values) approximations
predict(ml_result)

# Plot the results
plot(ml_result)
}
}
\references{
A. Gallant and D. W. Nychka (1987) <doi:10.2307/1913241>
}
\seealso{
\link[hpa]{summary.hpaML}, \link[hpa]{predict.hpaML}, \link[hpa]{AIC.hpaML}, \link[hpa]{logLik.hpaML}
}
