#' Create a data table of draws of parameter values for Monte Carlo
#' 
#' This function creates a data table of draws of parameter values for use with 
#' Monte Carlo methods
#' 
#' @param chem.cas Chemical Abstract Services Registry Number (CAS-RN) -- if
#'  parameters is not specified then the chemical must be identified by either
#'  CAS, name, or DTXISD
#' @param chem.name Chemical name (spaces and capitalization ignored) --  if
#'  parameters is not specified then the chemical must be identified by either
#'  CAS, name, or DTXISD
#' @param dtxsid EPA's DSSTox Structure ID (\url{https://comptox.epa.gov/dashboard})  
#'  -- if parameters is not specified then the chemical must be identified by 
#' either CAS, name, or DTXSIDs
#' @param parameters Parameters from the appropriate parameterization function
#' for the model indicated by argument model
#' @param samples Number of samples generated in calculating quantiles.
#' @param species Species desired (either "Rat", "Rabbit", "Dog", "Mouse", or
#' default "Human"). Species must be set to "Human" to run httkpop model. 
#' @param suppress.messages Whether or not to suppress output message.
#' @param model Model used in calculation: 'pbtk' for the multiple compartment
#' model,'3compartment' for the three compartment model, '3compartmentss' for
#' the three compartment steady state model, and '1compartment' for one
#' compartment model.  This only applies when httkpop=TRUE and species="Human",
#' otherwise '3compartmentss' is used.
#' @param httkpop Whether or not to use the Ring et al. (2017) "httkpop"
#' population generator. Species must be 'Human'.
#' @param invitrouv Logical to indicate whether to include in vitro parameters
#' such as intrinsic hepatic clearance rate and fraction unbound in plasma
#' in uncertainty and variability analysis
#' @param calcrb2p Logical determining whether or not to recalculate the 
#' chemical ratio of blood to plasma 
#' @param censored.params The parameters listed in censored.params are sampled
#' from a normal distribution that is censored for values less than the limit
#' of detection (specified separately for each paramter). This argument should
#' be a list of sub-lists. Each sublist is named for a parameter in
#' "parameters" and contains two elements: "CV" (coefficient of variation) and
#' "LOD" (limit of detection, below which parameter values are censored. New
#' values are sampled with mean equal to the value in "parameters" and standard
#' deviation equal to the mean times the CV.  Censored values are sampled on a
#' uniform distribution between 0 and the limit of detection. Not used with
#' httkpop model.
#' @param vary.params The parameters listed in vary.params are sampled from a
#' normal distribution that is truncated at zero. This argument should be a
#' list of coefficients of variation (CV) for the normal distribution. Each
#' entry in the list is named for a parameter in "parameters". New values are
#' sampled with mean equal to the value in "parameters" and standard deviation
#' equal to the mean times the CV. Not used with httkpop model.
#' @param return.samples Whether or not to return the vector containing the
#' samples from the simulation instead of the selected quantile.
#' @param tissue Desired steady state tissue conentration.
#' @param httkpop.dt A data table generated by \code{\link{httkpop_generate}}.
#' This defaults to NULL, in which case \code{\link{httkpop_generate}} is 
#' called to generate this table.
#' @param httkpop.generate.arg.list Additional parameters passed to 
#' \code{\link{httkpop_generate}}.
#' @param convert.httkpop.arg.list Additional parameters passed to the 
#' convert_httkpop_* function for the model.
#' @param invitro.mc.arg.list Additional parameters passed to 
#' \code{\link{invitro_mc}}.
#' @param propagate.invitrouv.arg.list Additional parameters passed to model's
#' associated in vitro uncertainty and variability propagation function
#' @param parameterize.arg.list Additional parameters passed to the 
#' parameterize_* function for the model.
#' @author Caroline Ring, Robert Pearce, and John Wambaugh
#'
#' @references 
#' Wambaugh, John F., et al. "Toxicokinetic triage for 
#' environmental chemicals." Toxicological Sciences 147.1 (2015): 55-67.
#'
#' Ring, Caroline L., et al. "Identifying populations sensitive to
#' environmental chemicals by simulating toxicokinetic variability."
#' Environment international 106 (2017): 105-118. 
#'
#' @keywords Monte-Carlo
#' 
#' @examples 
#' 
#' \dontrun{
#' sample_set = create_mc_samples(chem.name = 'bisphenol a')
#' }
#'
#' @import stats
#' @import methods
#' @export create_mc_samples
create_mc_samples <- function(chem.cas=NULL,
                        chem.name=NULL,
                        dtxsid = NULL,
                        parameters=NULL,
                        samples=1000,
                        species="Human",
                        suppress.messages=F,
                        model='3compartmentss',
                        httkpop=T,
                        invitrouv=T,
                        calcrb2p=T,
                        censored.params=list(),
                        vary.params=list(),
                        return.samples=F,
                        tissue=NULL,
                        httkpop.dt=NULL,
                        invitro.mc.arg.list=list(
                          adjusted.Funbound.plasma=T,
                          poormetab=T,
                          fup.censored.dist=FALSE,
                          fup.lod=0.01,
                          fup.meas.cv=0.4,
                          clint.meas.cv=0.3,
                          fup.pop.cv=0.3,
                          clint.pop.cv=0.3),
                        httkpop.generate.arg.list=list(
                          method='direct resampling',
                          gendernum=NULL,
                          agelim_years=NULL,
                          agelim_months=NULL,
                          weight_category =  c(
                            "Underweight", 
                            "Normal", 
                            "Overweight", 
                            "Obese"),
                          gfr_category = c(
                            "Normal", 
                            "Kidney Disease", 
                            "Kidney Failure"),
                          reths = c(
                            "Mexican American", 
                            "Other Hispanic", 
                            "Non-Hispanic White",
                            "Non-Hispanic Black", 
                            "Other")),
                        convert.httkpop.arg.list=list(),
                        propagate.invitrouv.arg.list=list(),
                        parameterize.arg.list=list(
                          restrictive.clearance = T,
                          default.to.human=F,
                          clint.pvalue.threshold=0.05,
                          regression=T))
{

#
#
# ERROR CHECKING AND INITIALIZATION:
#
#

# We need to describe the chemical to be simulated one way or another:
  if (is.null(chem.cas) & 
      is.null(chem.name) & 
      is.null(dtxsid) &
      is.null(parameters)) 
    stop('Parameters, chem.name, chem.cas, or dtxsid must be specified.')

  if (is.null(model)) stop("Model must be specified.")
# We need to know model-specific information (from modelinfo_[MODEL].R]) 
# to set up the solver:
  model <- tolower(model)
  if (!(model %in% names(model.list)))            
  {
    stop(paste("Model",model,"not available. Please select from:",
      paste(names(model.list),collapse=", ")))
  }

  # Column names for data.tables
  # Appease R CMD check --as-cran variable binding:
  variable <- Name  <- Parameter <- hematocrit <- this.chem <- Krbc2pu <- NULL
  Rblood2plasma <- Qgutf <- Funbound.plasma <- Qtotal.liverc <- NULL
  Qcardiacc <- Qliverf <- hepatic.bioavailability <- ..parameter.names <- NULL
  Species <- NULL
  
# Check to see if we need to call the parameterize_MODEL function:
  if (is.null(parameters))
  {
#Depending on model, choose the function in HTTK that will return the default
#HTTK parameters for this chemical
    paramfun <- model.list[[model]]$parameterize.func
    parameterize.args <- c(list(chem.cas=chem.cas,
                             chem.name=chem.name,
                             dtxsid=dtxsid,
                             species=species),
                             parameterize.arg.list)
    # Make sure all the arguments are used by the function:
    parameterize.args <- parameterize.args[names(parameterize.args) %in% 
      methods::formalArgs(paramfun)]
    parameters.mean <- do.call(getFromNamespace(paramfun, "httk"),
                         args=parameterize.args)
    pschmitt <- parameterize_schmitt(
                  chem.cas=chem.cas,
                  chem.name,
                  dtxsid=dtxsid,
                  species=species,
                  suppress.messages=T)
# The Schmitt parameters are useful if we need to redo partitioning later:
    pschmitt <- pschmitt[!(names(pschmitt)%in%names(parameters.mean))]
    parameters.mean <- c(parameters.mean, pschmitt)
  } else {
    if (!is.list(parameters)) stop(
"Argument \"parameters\" to create_mc_samples should be a list of model parameters.")
    parameters.mean <- parameters 
  }
  parameter.names <- names(parameters.mean)
    
# Make sure that parameters that monte_carlo samples won't be overwritten later:
  if (httkpop & any(c(names(censored.params),names(vary.params)) %in%
    model.list[[model]]$httkpop.params)) stop(paste("Parameters",
      paste(c(names(censored.params),names(vary.params))[
        c(names(censored.params),names(vary.params)) %in%
        model.list[[model]]$httkpop.params],collapse=", "), 
        "are specified to be sampled by monte_carlo and then overwritten by httkpop_mc."))
        
# Make sure that parameters that monte_carlo samples won't be overwritten later:
  if (invitrouv & (any(c(names(censored.params),names(vary.params)) %in%
    model.list[[model]]$invitro.params))) stop(paste("Parameters",
      paste(c(names(censored.params),names(vary.params))[
        c(names(censored.params),names(vary.params)) %in%
        model.list[[model]]$invitro.params],collapse=", "), 
        "are specified to be sampled by monte_carlo and then overwritten by invitro_mc."))
  
#
#
# MONTE CARLO STEP ONE
#
#

# Sample any parameters requested with the conventional sampler
# (Wambaugh et al., 2015):
  parameters.dt <- monte_carlo(parameters.mean,
                     censored.params=censored.params,
                     cv.params=vary.params,
                     samples=samples)
                      
#
#
# MONTE CARLO STEP TWO
# httk-pop (Ring et al., 2017)
#
#

  if (httkpop==T & tolower(species)=="human")
  {
    physiology.dt <- httkpop_mc(
                       model=model,
                       samples=samples,
                       httkpop.dt=httkpop.dt,
                                             httkpop.generate.arg.list)
# Overwrite parameters specified by httk-pop:
    parameters.dt[,names(physiology.dt):=physiology.dt]
    
  # Convert the httk-pop parameters to appropriate model variables
    converthttkpopfun <- model.list[[model]]$convert.httkpop.func
    if (!is.null(converthttkpopfun))
      parameters.dt <- do.call(converthttkpopfun, args=c(list(
                       parameters.dt=parameters.dt,
                       httkpop.dt=httkpop.dt),
                       convert.httkpop.arg.list))
   } else {
    if(httkpop==T) 
      warning('httkpop model only available for human and thus not used.\n\
Set species=\"Human\" to run httkpop model.')   
     this.tissuedata <- subset(tissue.data, tolower(Species)==tolower(species))
     these.vols <- subset(this.tissuedata,variable=="Vol (L/kg)")
     these.vols$Name <- paste("V",these.vols$Tissue,"c",sep="")
     for (this.name in these.vols$Name)
       if (!(this.name %in% names(parameters.dt)))
         parameters.dt[,eval(this.name):=subset(these.vols,Name==this.name)$value]
     these.flows <- subset(this.tissuedata,variable=="Flow (mL/min/kg^(3/4))")
     these.flows$Name <- paste("Q",these.vols$Tissue,"f",sep="")
     these.flows$value <- these.flows$value/
       as.numeric(subset(httk::physiology.data,Parameter=="Cardiac Output")[
       tolower(colnames(httk::physiology.data))==tolower(species)])
     for (this.name in these.flows$Name)
       if (!(this.name %in% names(parameters.dt)))
         parameters.dt[,eval(this.name):=subset(these.flows,Name==this.name)$value]
     parameters.dt[, hematocrit:=
       as.numeric(subset(httk::physiology.data,Parameter=="Hematocrit")[
       tolower(colnames(httk::physiology.data))==tolower(species)])]
  }
#
#
# MONTE CARLO STEP THREE
# PERFORM MONTE CARLO ON THE IN VITRO PARAMETERS (Wambaugh et al., 2019)
#
#

# Next add chemical-specific Funbound.plasma and CLint values
# Just cbind them together for now
  if (invitrouv) 
  {
    parameters.dt <- do.call(invitro_mc,
                       args=c(list(
                         parameters.dt=parameters.dt,
                         samples=samples),
                         invitro.mc.arg.list))
  }

# CLEAN UP PARAMETER MATRIX (bug fix v1.10.1)
#
# Force pKa to NA_real_ so data.table doesn't replace everything with text
  if (any(c("pKa_Donor","pKa_Accept") %in% names(parameters.dt)))
  {
    suppressWarnings(parameters.dt[, c("pKa_Donor","pKa_Accept") := NULL]) %>% .[, c("pKa_Donor","pKa_Accept") := NA_real_]
  }

#
#
# MONTE CARLO STEP FOUR
# PROPAGATE ANY CHANGES IN PARAMETER VALUES:
#
#

# Do we need to calculate first pass metabolism for this model (i.e., is flow
# from the gut into the liver not included in the model)
  firstpass <- model.list[[model]]$firstpass

  if (calcrb2p & any(c(!is.null(chem.cas),
                       !is.null(chem.name),
                       !is.null(dtxsid))))
  {
    Rb2p.invivo <- get_rblood2plasma(chem.cas=chem.cas,
                                     chem.name=chem.name,
                                     dtxsid=dtxsid)
  } else Rb2p.invivo <- NA

  if (model.list[[model]]$calcpc | (calcrb2p & is.na(Rb2p.invivo)) | firstpass)
  { 
#Now, predict the partitioning coefficients using Schmitt's method. The
#result will be a list of numerical vectors, one vector for each
#tissue-to-plasma partitioning coefficient, and one element of each vector
#for each individual. The list element names specify which partition
#coefficient it is, e.g. Kliver2plasma, Kgut2plasma, etc.
    PCs <- predict_partitioning_schmitt(
             parameters=parameters.dt,
             chem.name=chem.name,
             chem.cas=this.chem,
             dtxsid=dtxsid,
             species=species,
             adjusted.Funbound.plasma=invitro.mc.arg.list$adjusted.Funbound.plasma,
             regression=parameterize.arg.list$regression,
             suppress.messages=T)
  }

# If the model uses partion coefficients we need to lump each individual
# separately in case rest of body organ volumes or PCs vary:
  if (model.list[[model]]$calcpc)
  {
     lumptissues <- lump_tissues(
                      PCs,
                      parameters=parameters.dt,
                      tissuelist=model.list[[model]]$tissue.list,
                      species=species
                      ) 
                      
     parameters.dt[, names(lumptissues):= lumptissues]
  }
  
  if (calcrb2p | firstpass)
  {
# If we have an in vivo value, then back-calculate the partition coefficient:
    if (!is.na(Rb2p.invivo))
    {
# From Pearce et al. (2017):
      parameters.dt[, Krbc2pu:=calc_krbc2pu(Rb2p.invivo,
                                            parameters.mean$Funbound.plasma,
                                            parameters.mean$hematocrit)]
    } else { 
      parameters.dt[, Krbc2pu:=PCs[['Krbc2pu']]]
# Calculate Rblood2plasma based on hematocrit, Krbc2plasma, and Funboun.plasma. 
# This is the ratio of chemical in blood vs. in plasma.
    }  
    parameters.dt[,Rblood2plasma := calc_rblood2plasma(
                                      hematocrit=parameters.dt$hematocrit,
                                      Krbc2pu=parameters.dt$Krbc2pu,
                                      Funbound.plasma=Funbound.plasma)]
  }
  
  if (firstpass)
  {
  
# For models that don't described first pass blood flow from the gut, need the
# total liver blood flow to cacluate a hepatic bioavailability (Rowland, 1973):
    if (!("Qtotal.liverc" %in% names(parameters.dt)))
      parameters.dt[, Qtotal.liverc:=Qcardiacc*(Qgutf+Qliverf)] # L/h/(kgBW)^3/4
  
# For models that don't described first pass blood flow from the gut, need the
# unscaled hepatic clearance to cacluate a hepatic bioavailability 
# (Rowland, 1973):      
  cl <- calc_hep_clearance(parameters=parameters.dt,
          hepatic.model='unscaled',
          suppress.messages=T)#L/h/kg body weight

  parameters.dt[,hepatic.bioavailability := calc_hep_bioavailability(
    parameters=list(
      Qtotal.liverc=parameters.dt$Qtotal.liverc, # L/h/kg^3/4
      Funbound.plasma=parameters.dt$Funbound.plasma,
      Clmetabolismc=cl, # L/h/kg
      Rblood2plasma=parameters.dt$Rblood2plasma,
      BW=parameters.dt$BW),
    restrictive.clearance=parameterize.arg.list$restrictive.clearance)] 
  }
  
#
# Do any model-specific uncertainty propagation
#
  propagateuvfun <- model.list[[model]]$propagateuv.func
  if (!is.null(propagateuvfun))
    parameters.dt <- do.call(propagateuvfun, args=c(list(
                       parameters.dt=parameters.dt),
                       propagate.invitrouv.arg.list))
  
#Return only the HTTK parameters for the specified model. That is, only the
#columns whose names are in the names of the default parameter set.
  return(parameters.dt[,model.list[[model]]$param.names,with=F])
}
