#' QC on clusters (nGenes, UMIs and percent mito)
#'
#' This function takes an object of class iCellR and creates QC plot.
#' @param x An object of class iCellR.
#' @param plot.type Choose from "box.umi", "box.mito", "box.gene", default = "box.mito".
#' @param cell.color Choose a color for points in the plot.
#' @param cell.size A number for the size of the points in the plot, default = 1.
#' @param box.color A color for the boxes in the "boxplot", default = "red".
#' @param box.line.col A color for the lines around the "boxplot", default = "green".
#' @param notch Notch the box plots, default = FALSE.
#' @param back.col Background color, default = "white"
#' @param cell.transparency Color transparency for points in "scatterplot" and "boxplot", default = 0.5.
#' @param interactive If set to TRUE an interactive HTML file will be created, default = TRUE.
#' @param out.name If "interactive" is set to TRUE, the out put name for HTML, default = "plot".
#' @return An object of class iCellR.
#' @examples
#' clust.stats.plot(demo.obj,
#'                    plot.type = "box.mito",
#'                    interactive = FALSE,
#'                    out.name = "box.mito.clusters")
#' @export
clust.stats.plot <- function (x = NULL,
                        plot.type = "box.mito",
                        cell.color = "slategray3",
                        cell.size = 1,
                        cell.transparency = 0.5,
                        box.color = "red",
                        box.line.col = "green",
                        back.col = "white",
                        notch = FALSE,
                        interactive = TRUE,
                        out.name = "plot")
{
  if ("iCellR" != class(x)[1]) {
    stop("x should be an object of class iCellR")
  }
  # get stats data for all cells
  DATA <- x@stats
  row.names(DATA) <- DATA$CellIds
  # get cluster data for all cells
  MyClusts <- x@best.clust
  # merge
  DATA <- merge(DATA, MyClusts, by = "row.names", all.x=FALSE, all.y=TRUE)
  # plot
  # mito
  mito.percent.plot <- ggplot(DATA,aes(y=mito.percent, x=as.factor(clusters))) +
    geom_jitter(color = cell.color, size = cell.size, alpha = cell.transparency) +
    geom_violin(trim=FALSE, col = "black", alpha = cell.transparency) +
    geom_boxplot(fill = box.color, col = "green", notch = notch, outlier.shape = NA, alpha = cell.transparency) +
    xlab("clusters") + ylab("percent of mito genes per cell") +
    stat_summary(fun.y=mean, geom="point", size=2, color="black") +
    theme_bw() + theme(axis.text.x=element_text(angle=90))
  # nGenes
  nGenes.plot <- ggplot(DATA,aes(y=nGenes,x=as.factor(clusters))) +
    geom_jitter(color = cell.color, size = cell.size, alpha = cell.transparency) +
    geom_violin(trim=FALSE, col = "black", alpha = cell.transparency) +
    geom_boxplot(fill = box.color, col = box.line.col, notch = notch, outlier.shape = NA, alpha = cell.transparency) +
    xlab("clusters") + ylab("number of genes per cell") +
    stat_summary(fun.y=mean, geom="point", size=2, color="black") +
    theme_bw() + theme(axis.text.x=element_text(angle=90))
  # UMIs
  UMIsplot <- ggplot(DATA,aes(y=UMIs,x=as.factor(clusters))) +
    geom_jitter(color = cell.color, size = cell.size, alpha = cell.transparency) +
    geom_violin(trim=FALSE, col = "black", alpha = cell.transparency) +
    geom_boxplot(fill = box.color, col = box.line.col, notch = notch, outlier.shape = NA, alpha = cell.transparency) +
    xlab("clusters") + ylab("number of UMIs per cell") +
    stat_summary(fun.y=mean, geom="point", size=2, color="black") +
    theme_bw() + theme(axis.text.x=element_text(angle=90))
# return
  if (plot.type == "box.umi") {
    if (interactive == TRUE) {
      OUT.PUT <- paste(out.name, ".html", sep="")
      htmlwidgets::saveWidget(ggplotly(Mito.UMIs),OUT.PUT)
    }
    else
      return(UMIsplot)
  }
  #
  if (plot.type == "box.mito") {
    if (interactive == TRUE) {
      OUT.PUT <- paste(out.name, ".html", sep="")
      htmlwidgets::saveWidget(ggplotly(Mito.UMIs),OUT.PUT)
    }
    else
      return(mito.percent.plot)
  }
  #
  if (plot.type == "box.gene") {
    if (interactive == TRUE) {
      OUT.PUT <- paste(out.name, ".html", sep="")
      htmlwidgets::saveWidget(ggplotly(Mito.UMIs),OUT.PUT)
    }
    else
      return(nGenes.plot)
  }
}
