### Prefix for code comments used with grep and gsub.
prefix <- "^### "

decomment <- function
### Remove comment prefix and join lines of code to form a
### documentation string.
(comments
### Character vector of prefixed comment lines.
 ){
  paste(gsub(prefix,"",comments),collapse=" ")
### String without prefixes or newlines.
}

### Necessary fields in DESCRIPTION, otherwise error.
fields <- c("Package","Maintainer","Author","Version",
            "License","Title","Description")
### Default DESCRIPTION, written if it doesn't exist.
empty.description <- matrix("",ncol=length(fields),dimnames=list(NULL,fields))

package.skeleton.dx <- function # Package skeleton deluxe
### Automates more of the setup process for a new source
### package. After inspecting the specified R code files to find
### inline documentation, it calls the standard package.skeleton
### function, which creates bare Rd files. The inline documentation is
### added to these Rd files and then these files are copied to
### ../man. It will overwrite files in the pkgdir/man directory.
(pkgdir="..",
### package directory where the DESCRIPTION file lives. Your code
### should be in pkgdir/R. We will setwd to pkgdir/R for the duration
### of the function, then switch back to where you were previously.
 code_files=NULL,
### Character vector with the names of the R code files, to be passed
### to package.skeleton, and also inspected for inline
### documentation. NULL means all files ending in .R.
 check=""
### Value indicating whether or not to check the package after
### documentation is built. Anything other than "" (the default) will
### check the package. "noex" means check the package without running
### examples, only tests (useful for debugging if your tests/examples
### take a long time).
 ){
  chdir <- file.path(pkgdir,"R")
  old.wd <- setwd(chdir)
  on.exit(setwd(old.wd))

  descfile <- file.path("..","DESCRIPTION")
  if(!file.exists(descfile)){
    write.dcf(empty.description,descfile)
    stop("Need ",descfile,"; please fill that in and try again")
  }

  ## Read description and check for errors
  desc <- read.dcf(descfile)
  for(f in fields){
    if(! f %in% colnames(desc))stop("Need ",f," in ",descfile)
    if(desc[,f]=="")stop("Need a value for ",f," in ",descfile)
  }

  ## extract docs from each file
  if(is.null(code_files))code_files <- Sys.glob("*.R")
  docs <- list()
  for(cf in code_files){
    L <- extract.docs.file(cf,check!="noex")
    for(N in names(L))docs[[N]] <- L[[N]]
  }
  for(i in names(docs)){
    docs[[i]]$author <- desc[,"Maintainer"]
    if(! 'title' %in% names(docs[[i]]))
      docs[[i]]$title <- gsub("[._]"," ",i)
  }
  name <- desc[,"Package"]
  details <- paste(paste(colnames(desc),": \\tab ",desc,"\\cr",sep=""),
                   collapse="\n")
  docs[[paste(name,"-package",sep="")]] <-
    list(title=desc[,"Title"],
         description=desc[,"Description"],
         `tabular{ll}`=details,
         author=desc[,"Maintainer"])
  ## Make package skeleton and edit Rd files
  unlink(name,rec=TRUE)
  package.skeleton(name,code_files=code_files)
  cat("Modifying files automatically generated by package.skeleton:\n")
  for(N in names(docs))modify.Rd.file(N,name,docs)
  file.copy(file.path(name,'man'),"..",rec=TRUE)
  unlink(name,rec=TRUE)

  if(check!=""){
    Rdir <- setwd(file.path("..",".."))
    cdir <- basename(dirname(Rdir))
    system(paste("R CMD check",cdir))
  }
  ## rebuild examples if we didn't test them:
  if(check=="noex")package.skeleton.dx(pkgdir,code_files,"")
}

modify.Rd.file <- function
### Add inline documentation from comments to an Rd file
### automatically-generated by package.skeleton.
(N,
### Name of function/file to which we will add documentation.
 pkg,
### Package name.
 docs
### Named list of documentation in extracted comments.
 ){
  fb <- paste(N,".Rd",sep="")
  f <- file.path(pkg,'man',fb)
  ## If there are no significant docs in the comments then the object
  ## should still be documented, by writing the file by hand in the
  ## man directory. This will write a blank Rd file if none exists, so
  ## it's easy to get started.
  if((length(docs[[N]])<3) &&
     file.exists(file.path("..","man",fb))){
    unlink(f)
    return()
  }
  cat(N,":",sep="")
  d <- docs[[N]]
  dlines <- readLines(f)

  ## cut out alias line if we are in the package file and there is a
  ## matching function
  if(length(grep("-package$",N)))
    dlines <- dlines[-grep(paste("alias[{]",N,sep=""),dlines)-1]

  ## cut out all comments {} interferes with regex matching
  dlines <- dlines[-grep("^[%~]",dlines)]

  ## cut out a couple of sections that cause warnings
  o <- grep("Optionally",dlines)
  if(length(o))dlines <- dlines[-(o:(o+1))]
  ## delete examples til the end of the file (also includes keywords)
  dlines <- dlines[1:(tail(grep("examples[{]$",dlines),1)-1)]
  ## add back a minimal examples section to find and replace
  dlines <- c(dlines,"\\examples{}")

  ## Find and replace based on data in d
  txt <- paste(dlines,collapse="\n")
  for(torep in names(d)){
    cat(" ",torep,sep="")
    FIND1 <- gsub("\\\\","\\\\\\\\",torep)
    FIND <- paste(gsub("([{}])","\\\\\\1",FIND1),"[{][^}]*[}]",sep="")
    ## need to escape backslashes for faithful copying of the comments
    ## to the Rd file:
    REP <- paste(FIND1,"{",gsub("\\\\","\\\\\\\\",d[[torep]]),"}",sep="")
    ## escape percent signs in R code:
    REP <- gsub("%","\\\\\\\\%",REP)
    txt <- gsub(FIND,REP,txt)
  }

  ## Fix usage
  m <- regexpr("usage[{][^}]*[}]",txt)
  Mend <- m+attr(m,"match.length")
  utxt <- substr(txt,m,Mend)
  if(length(grep("usage[{]data",utxt)))
     utxt <- gsub("data[(]([^)]*)[)]","\\1",utxt)
  ## add another backslash due to bug in package.skeleton
  txt <- paste(substr(txt,1,m-1),
               gsub("\\\\","\\\\\\\\",utxt),
               substr(txt,Mend+1,nchar(txt)),
               sep="")
  ## This doesn't work if there are quotes in the default values:
  ## gsub(",",paste("\n",paste(rep(" ",l=nchar(N)-1),collapse="")),utxt)
  cat(txt,file=f)
  cat("\n")
}

extract.docs.file <- function # Extract documentation from a file
### Parse an R code file and extract inline documentation from
### comments around each function.
(code.file,
### The R code file to parse.
 write.examples=TRUE
### Gather examples from test files?
 ){
  code <- readLines(code.file)
  comment.lines <- grep(prefix,code)
  e <- new.env()
  old <- options(keep.source.pkgs=TRUE)
  r <- try(sys.source(code.file,e),TRUE)
  if(class(r)=="try-error")
    stop("source ",code.file," failed with error:\n",r)
  options(old)
  objs <- sapply(ls(e),get,e,simplify=FALSE)
  extract.docs <- function(on){
    res <- try({
      o <- objs[[on]]
      doc <- if("function"%in%class(o)){
        tdoc <- extract.docs.fun(o)
        if(write.examples){ ## do not get examples from test files.
          tfile <- file.path("..","tests",paste(on,".R",sep=""))
          if(file.exists(tfile))
            tdoc[["examples"]] <- paste(readLines(tfile),collapse="\n")
        }
        tdoc
      }else list()
      ## Take the line before the first occurence of the variable
      defined.on <- grep(paste("^",on,sep=""),code)[1]-1
      if(!"description"%in%names(doc) && defined.on%in%comment.lines){
        comments.before <- comment.lines[comment.lines<=defined.on]
        these.comment.lines <- if(length(comments.before)==1)comments.before else{
          diffs <- diff(comments.before)
          comments.before[max(which(diffs>1))+1]:defined.on
        }
        doc$description <- decomment(code[these.comment.lines])
      }
      doc
    },FALSE)
    if(class(res)=="try-error"){
      cat("Failed to extract docs for: ",on,"\n\n")
      list()
    }
    else res
  }
  res <- sapply(names(objs),extract.docs,simplify=FALSE)
  res
### named list of lists. Each element is the result of a call to
### extract.docs.fun, with names corresponding to functions found in
### the R code file.
}

extract.docs.fun <- function # Extract documentation from a function
### Given a function, return a list describing inline documentation in
### the source of that function (relies on source attr).
(fun
### The function to examine.
 ){
  res <- list()
  code <- attr(fun,"source")
  clines <- grep("^#",code)
  if(length(grep("#",code[1])))res$title <- gsub("[^#]*#(.*)","\\1",code[1])
  if(length(clines)==0)return(res) ## no comments found
  bounds <- which(diff(clines)!=1)
  starts <- c(1,bounds+1)
  ends <- c(bounds,length(clines))
  for(i in seq_along(starts)){
    start <- clines[starts[i]]
    end <- clines[ends[i]]
    lab <- if(end+1==length(code))"value"
    else if(start==2)"description"
    else {
      arg <- gsub("^[ (]*","",code[start-1])
      arg <- gsub("^([^=,]*)[=,].*","\\1",arg)
      arg <- gsub("...","\\dots",arg,fix=TRUE) ##special case for dots
      paste("item{",arg,"}",sep="")
    }
    res[[lab]] <- decomment(code[start:end])
  }
  ##details<<
  ## For simple functions/arguments, the argument may also be documented by
  ## appending ##<< comments on the same line as the argument. For those who
  ## wish to become confused, any following ### comment lines will be appended.
  arg.pat <- "^\\s*\\(?\\s*([\\w\\.]+).*##<<\\s*(\\S.*?)\\s*$"
  for ( k in 2:length(code) ){
    if ( 0 < length(grep(arg.pat,code[k],perl=TRUE)) ){
      arg <- gsub(arg.pat,"\\\\item\\{\\1\\}",code[k],perl=TRUE)
      comment <- gsub(arg.pat,"\\2",code[k],perl=TRUE);
      res[[arg]] <-
        if ( is.null(res[[arg]]) )comment
        else paste(comment,res[[arg]],sep=" ")
    }
  }

  skeleton.fields <- c("alias","details","keyword","references","author",
                       "note","seealso","value")
  ##details<<
  ## Additionally, contiguous sections of ## comment lines beginning with
  ## xxx<< (where xxx is one of the "other" fields: alias, details, keyword,
  ## references, author, note, seealso or value) are accumulated and inserted in
  ## the relevant part of the .Rd file.
  ##
  ## In the case of value, the extra information is appended to that from
  ## any final ### comment lines.

  ## but this should not appear, because separated by a blank line
  extra.regexp <- paste("^\\s*##<<(",paste(skeleton.fields,collapse="|"),
                        ")\\s*(.*)$",sep="")
  starts.extra <- grep(extra.regexp,code,perl=TRUE)
  cont.re <- "^\\s*##\\s*";
  for ( start in starts.extra ){
    line <- code[start]
    field <- gsub(extra.regexp,"\\1",line,perl=TRUE)
    payload <- gsub(extra.regexp,"\\2",line,perl=TRUE)
    if ( start < length(code) ){
      for ( k in (start+1):length(code) ){
        if ( k %in% starts.extra        # go on to next extra section
            || 0 == length(grep(cont.re,code[k])) ){ # blank line or code ends
          if ( "alias" == field ){
            ##note<< Alias extras are automatically split at new lines.
            payload <- gsub("\\n+",paste("\\}\n\\",field,"\\{",sep=""),
                            payload,perl=TRUE)
          } else if ("keyword" == field ){
            ##keyword<< documentation utilities
            ##note<< Keyword extras are auto-split at white space.
            payload <- gsub("\\s+",paste("\\}\n\\",field,"\\{",sep=""),
                            payload,perl=TRUE)
          }
          ##details<< Each separate extra section appears as a new paragraph
          ## except that empty sections (no matter how many lines) are ignored.
          if ( 0 == length(grep("^\\s*$",payload,perl=TRUE)) )
            res[[field]] <-
              if ( is.null(res[[field]]) )payload
              else res[[field]] <- paste(res[[field]], payload, sep="\n\n")
          break;
        } else {
          payload <- paste(payload,sub(cont.re,"",code[k],perl=TRUE),sep="\n")
        }
      }
    }
  }
  res
### Named list of character strings extracted from comments. For each
### name N we will look for N\{...\} in the Rd file and replace it
### with the string in this list (implemented in modify.Rd.file).
}

.result <- 
 list(decomment = list(format="",title="decomment",definition = "decomment <- function\n### Remove comment prefix and join lines of code to form a\n### documentation string.\n(comments\n### Character vector of prefixed comment lines.\n ){\n  paste(gsub(prefix,\"\",comments),collapse=\" \")\n### String without prefixes or newlines.\n}",  
     description = "Remove comment prefix and join lines of code to form a\ndocumentation string.",  
     `item{comments}` = "Character vector of prefixed comment lines.",  
     value = "String without prefixes or newlines."),
      empty.description = list(format="",title="empty description",
     definition = "empty.description <- matrix(\"\",ncol=length(fields),dimnames=list(NULL,fields))",  
     description = "Default DESCRIPTION, written if it doesn't exist."),  
     extract.docs.file = list(definition = "extract.docs.file <- function # Extract documentation from a file\n### Parse an R code file and extract inline documentation from\n### comments around each function.\n(code.file,\n### The R code file to parse.\n write.examples=TRUE\n### Gather examples from test files?\n ){\n  code <- readLines(code.file)\n  comment.lines <- grep(prefix,code)\n  e <- new.env()\n  old <- options(keep.source.pkgs=TRUE)\n  r <- try(sys.source(code.file,e),TRUE)\n  if(class(r)==\"try-error\")\n    stop(\"source \",code.file,\" failed with error:\\n\",r)\n  options(old)\n  objs <- sapply(ls(e),get,e,simplify=FALSE)\n  extract.docs <- function(on){\n    res <- try({\n      o <- objs[[on]]\n      doc <- if(\"function\"%in%class(o)){\n        tdoc <- extract.docs.fun(o)\n        if(write.examples){ ## do not get examples from test files.\n          tfile <- file.path(\"..\",\"tests\",paste(on,\".R\",sep=\"\"))\n          if(file.exists(tfile))\n            tdoc[[\"examples\"]] <- paste(readLines(tfile),collapse=\"\\n\")\n        }\n        tdoc\n      }else list()\n      ## Take the line before the first occurence of the variable\n      defined.on <- grep(paste(\"^\",on,sep=\"\"),code)[1]-1\n      if(!\"description\"%in%names(doc) && defined.on%in%comment.lines){\n        comments.before <- comment.lines[comment.lines<=defined.on]\n        these.comment.lines <- if(length(comments.before)==1)comments.before else{\n          diffs <- diff(comments.before)\n          comments.before[max(which(diffs>1))+1]:defined.on\n        }\n        doc$description <- decomment(code[these.comment.lines])\n      }\n      doc\n    },FALSE)\n    if(class(res)==\"try-error\"){\n      cat(\"Failed to extract docs for: \",on,\"\\n\\n\")\n      list()\n    }\n    else res\n  }\n  res <- sapply(names(objs),extract.docs,simplify=FALSE)\n  res\n### named list of lists. Each element is the result of a call to\n### extract.docs.fun, with names corresponding to functions found in\n### the R code file.\n}",  
         title = "Extract documentation from a file",
       format="",
       description = "Parse an R code file and extract inline documentation from\ncomments around each function.",  
         `item{code.file}` = "The R code file to parse.", `item{write.examples}` = "Gather examples from test files?",  
         value = "named list of lists. Each element is the result of a call to\nextract.docs.fun, with names corresponding to functions found in\nthe R code file."),  
     extract.docs.fun = list(format="",
       definition = "extract.docs.fun <- function # Extract documentation from a function\n### Given a function, return a list describing inline documentation in\n### the source of that function (relies on source attr).\n(fun\n### The function to examine.\n ){\n  res <- list()\n  code <- attr(fun,\"source\")\n  clines <- grep(\"^#\",code)\n  if(length(grep(\"#\",code[1])))res$title <- gsub(\"[^#]*#(.*)\",\"\\\\1\",code[1])\n  if(length(clines)==0)return(res) ## no comments found\n  bounds <- which(diff(clines)!=1)\n  starts <- c(1,bounds+1)\n  ends <- c(bounds,length(clines))\n  for(i in seq_along(starts)){\n    start <- clines[starts[i]]\n    end <- clines[ends[i]]\n    lab <- if(end+1==length(code))\"value\"\n    else if(start==2)\"description\"\n    else {\n      arg <- gsub(\"^[ (]*\",\"\",code[start-1])\n      arg <- gsub(\"^([^=,]*)[=,].*\",\"\\\\1\",arg)\n      arg <- gsub(\"...\",\"\\\\dots\",arg,fix=TRUE) ##special case for dots\n      paste(\"item{\",arg,\"}\",sep=\"\")\n    }\n    res[[lab]] <- decomment(code[start:end])\n  }\n  ##details<<\n  ## For simple functions/arguments, the argument may also be documented by\n  ## appending ##<< comments on the same line as the argument. For those who\n  ## wish to become confused, any following ### comment lines will be appended.\n  arg.pat <- \"^\\\\s*\\\\(?\\\\s*([\\\\w\\\\.]+).*##<<\\\\s*(\\\\S.*?)\\\\s*$\"\n  for ( k in 2:length(code) ){\n    if ( 0 < length(grep(arg.pat,code[k],perl=TRUE)) ){\n      arg <- gsub(arg.pat,\"\\\\\\\\item\\\\{\\\\1\\\\}\",code[k],perl=TRUE)\n      comment <- gsub(arg.pat,\"\\\\2\",code[k],perl=TRUE);\n      res[[arg]] <-\n        if ( is.null(res[[arg]]) )comment\n        else paste(comment,res[[arg]],sep=\" \")\n    }\n  }\n\n  skeleton.fields <- c(\"alias\",\"details\",\"keyword\",\"references\",\"author\",\n                       \"note\",\"seealso\",\"value\")\n  ##details<<\n  ## Additionally, contiguous sections of ## comment lines beginning with\n  ## xxx<< (where xxx is one of the \"other\" fields: alias, details, keyword,\n  ## references, author, note, seealso or value) are accumulated and inserted in\n  ## the relevant part of the .Rd file.\n  ##\n  ## In the case of value, the extra information is appended to that from\n  ## any final ### comment lines.\n\n  ## but this should not appear, because separated by a blank line\n  extra.regexp <- paste(\"^\\\\s*##<<(\",paste(skeleton.fields,collapse=\"|\"),\n                        \")\\\\s*(.*)$\",sep=\"\")\n  starts.extra <- grep(extra.regexp,code,perl=TRUE)\n  cont.re <- \"^\\\\s*##\\\\s*\";\n  for ( start in starts.extra ){\n    line <- code[start]\n    field <- gsub(extra.regexp,\"\\\\1\",line,perl=TRUE)\n    payload <- gsub(extra.regexp,\"\\\\2\",line,perl=TRUE)\n    if ( start < length(code) ){\n      for ( k in (start+1):length(code) ){\n        if ( k %in% starts.extra        # go on to next extra section\n            || 0 == length(grep(cont.re,code[k])) ){ # blank line or code ends\n          if ( \"alias\" == field ){\n            ##note<< Alias extras are automatically split at new lines.\n            payload <- gsub(\"\\\\n+\",paste(\"\\\\}\\n\\\\\",field,\"\\\\{\",sep=\"\"),\n                            payload,perl=TRUE)\n          } else if (\"keyword\" == field ){\n            ##keyword<< documentation utilities\n            ##note<< Keyword extras are auto-split at white space.\n            payload <- gsub(\"\\\\s+\",paste(\"\\\\}\\n\\\\\",field,\"\\\\{\",sep=\"\"),\n                            payload,perl=TRUE)\n          }\n          ##details<< Each separate extra section appears as a new paragraph\n          ## except that empty sections (no matter how many lines) are ignored.\n          if ( 0 == length(grep(\"^\\\\s*$\",payload,perl=TRUE)) )\n            res[[field]] <-\n              if ( is.null(res[[field]]) )payload\n              else res[[field]] <- paste(res[[field]], payload, sep=\"\\n\\n\")\n          break;\n        } else {\n          payload <- paste(payload,sub(cont.re,\"\",code[k],perl=TRUE),sep=\"\\n\")\n        }\n      }\n    }\n  }\n  res\n### Named list of character strings extracted from comments. For each\n### name N we will look for N\\{...\\} in the Rd file and replace it\n### with the string in this list (implemented in modify.Rd.file).\n}",  
         title = "Extract documentation from a function", description = "Given a function, return a list describing inline documentation in\nthe source of that function (relies on source attr).",  
         `item{fun}` = "The function to examine.", value = "Named list of character strings extracted from comments. For each\nname N we will look for N\\{...\\} in the Rd file and replace it\nwith the string in this list (implemented in modify.Rd.file).",  
         details = "For simple functions/arguments, the argument may also be documented by\nappending ##<< comments on the same line as the argument. For those who\nwish to become confused, any following ### comment lines will be appended.\n\nAdditionally, contiguous sections of ## comment lines beginning with\nxxx<< (where xxx is one of the \"other\" fields: alias, details, keyword,\nreferences, author, note, seealso or value) are accumulated and inserted in\nthe relevant part of the .Rd file.\n\nIn the case of value, the extra information is appended to that from\nany final ### comment lines.\n\nEach separate extra section appears as a new paragraph\nexcept that empty sections (no matter how many lines) are ignored.",  
         note = "Alias extras are automatically split at new lines.\n\nKeyword extras are auto-split at white space.",  
         keyword = "documentation}\n\\keyword{utilities"),
      fields = list(format="",title="fields",
         definition = "fields <- c(\"Package\",\"Maintainer\",\"Author\",\"Version\",\n            \"License\",\"Title\",\"Description\")",  
         description = "Necessary fields in DESCRIPTION, otherwise error."),  
     modify.Rd.file = list(definition = "modify.Rd.file <- function\n### Add inline documentation from comments to an Rd file\n### automatically-generated by package.skeleton.\n(N,\n### Name of function/file to which we will add documentation.\n pkg,\n### Package name.\n docs\n### Named list of documentation in extracted comments.\n ){\n  fb <- paste(N,\".Rd\",sep=\"\")\n  f <- file.path(pkg,'man',fb)\n  ## If there are no significant docs in the comments then the object\n  ## should still be documented, by writing the file by hand in the\n  ## man directory. This will write a blank Rd file if none exists, so\n  ## it's easy to get started.\n  if((length(docs[[N]])<3) &&\n     file.exists(file.path(\"..\",\"man\",fb))){\n    unlink(f)\n    return()\n  }\n  cat(N,\":\",sep=\"\")\n  d <- docs[[N]]\n  dlines <- readLines(f)\n\n  ## cut out alias line if we are in the package file and there is a\n  ## matching function\n  if(length(grep(\"-package$\",N)))\n    dlines <- dlines[-grep(paste(\"alias[{]\",N,sep=\"\"),dlines)-1]\n\n  ## cut out all comments {} interferes with regex matching\n  dlines <- dlines[-grep(\"^[%~]\",dlines)]\n\n  ## cut out a couple of sections that cause warnings\n  o <- grep(\"Optionally\",dlines)\n  if(length(o))dlines <- dlines[-(o:(o+1))]\n  ## delete examples til the end of the file (also includes keywords)\n  dlines <- dlines[1:(tail(grep(\"examples[{]$\",dlines),1)-1)]\n  ## add back a minimal examples section to find and replace\n  dlines <- c(dlines,\"\\\\examples{}\")\n\n  ## Find and replace based on data in d\n  txt <- paste(dlines,collapse=\"\\n\")\n  for(torep in names(d)){\n    cat(\" \",torep,sep=\"\")\n    FIND1 <- gsub(\"\\\\\\\\\",\"\\\\\\\\\\\\\\\\\",torep)\n    FIND <- paste(gsub(\"([{}])\",\"\\\\\\\\\\\\1\",FIND1),\"[{][^}]*[}]\",sep=\"\")\n    ## need to escape backslashes for faithful copying of the comments\n    ## to the Rd file:\n    REP <- paste(FIND1,\"{\",gsub(\"\\\\\\\\\",\"\\\\\\\\\\\\\\\\\",d[[torep]]),\"}\",sep=\"\")\n    ## escape percent signs in R code:\n    REP <- gsub(\"%\",\"\\\\\\\\\\\\\\\\%\",REP)\n    txt <- gsub(FIND,REP,txt)\n  }\n\n  ## Fix usage\n  m <- regexpr(\"usage[{][^}]*[}]\",txt)\n  Mend <- m+attr(m,\"match.length\")\n  utxt <- substr(txt,m,Mend)\n  if(length(grep(\"usage[{]data\",utxt)))\n     utxt <- gsub(\"data[(]([^)]*)[)]\",\"\\\\1\",utxt)\n  ## add another backslash due to bug in package.skeleton\n  txt <- paste(substr(txt,1,m-1),\n               gsub(\"\\\\\\\\\",\"\\\\\\\\\\\\\\\\\",utxt),\n               substr(txt,Mend+1,nchar(txt)),\n               sep=\"\")\n  ## This doesn't work if there are quotes in the default values:\n  ## gsub(\",\",paste(\"\\n\",paste(rep(\" \",l=nchar(N)-1),collapse=\"\")),utxt)\n  cat(txt,file=f)\n  cat(\"\\n\")\n}",  
         description = "Add inline documentation from comments to an Rd file\nautomatically-generated by package.skeleton.",
       format="",title="modify Rd file",
         `item{N}` = "Name of function/file to which we will add documentation.",  
         `item{pkg}` = "Package name.", `item{docs}` = "Named list of documentation in extracted comments."),  
     package.skeleton.dx = list(format="",
       definition = "package.skeleton.dx <- function # Package skeleton deluxe\n### Automates more of the setup process for a new source\n### package. After inspecting the specified R code files to find\n### inline documentation, it calls the standard package.skeleton\n### function, which creates bare Rd files. The inline documentation is\n### added to these Rd files and then these files are copied to\n### ../man. It will overwrite files in the pkgdir/man directory.\n(pkgdir=\"..\",\n### package directory where the DESCRIPTION file lives. Your code\n### should be in pkgdir/R. We will setwd to pkgdir/R for the duration\n### of the function, then switch back to where you were previously.\n code_files=NULL,\n### Character vector with the names of the R code files, to be passed\n### to package.skeleton, and also inspected for inline\n### documentation. NULL means all files ending in .R.\n check=\"\"\n### Value indicating whether or not to check the package after\n### documentation is built. Anything other than \"\" (the default) will\n### check the package. \"noex\" means check the package without running\n### examples, only tests (useful for debugging if your tests/examples\n### take a long time).\n ){\n  chdir <- file.path(pkgdir,\"R\")\n  old.wd <- setwd(chdir)\n  on.exit(setwd(old.wd))\n\n  descfile <- file.path(\"..\",\"DESCRIPTION\")\n  if(!file.exists(descfile)){\n    write.dcf(empty.description,descfile)\n    stop(\"Need \",descfile,\"; please fill that in and try again\")\n  }\n\n  ## Read description and check for errors\n  desc <- read.dcf(descfile)\n  for(f in fields){\n    if(! f %in% colnames(desc))stop(\"Need \",f,\" in \",descfile)\n    if(desc[,f]==\"\")stop(\"Need a value for \",f,\" in \",descfile)\n  }\n\n  ## extract docs from each file\n  if(is.null(code_files))code_files <- Sys.glob(\"*.R\")\n  docs <- list()\n  for(cf in code_files){\n    L <- extract.docs.file(cf,check!=\"noex\")\n    for(N in names(L))docs[[N]] <- L[[N]]\n  }\n  for(i in names(docs)){\n    docs[[i]]$author <- desc[,\"Maintainer\"]\n    if(! 'title' %in% names(docs[[i]]))\n      docs[[i]]$title <- gsub(\"[._]\",\" \",i)\n  }\n  name <- desc[,\"Package\"]\n  details <- paste(paste(colnames(desc),\": \\\\tab \",desc,\"\\\\cr\",sep=\"\"),\n                   collapse=\"\\n\")\n  docs[[paste(name,\"-package\",sep=\"\")]] <-\n    list(title=desc[,\"Title\"],\n         description=desc[,\"Description\"],\n         `tabular{ll}`=details,\n         author=desc[,\"Maintainer\"])\n  ## Make package skeleton and edit Rd files\n  unlink(name,rec=TRUE)\n  package.skeleton(name,code_files=code_files)\n  cat(\"Modifying files automatically generated by package.skeleton:\\n\")\n  for(N in names(docs))modify.Rd.file(N,name,docs)\n  file.copy(file.path(name,'man'),\"..\",rec=TRUE)\n  unlink(name,rec=TRUE)\n\n  if(check!=\"\"){\n    Rdir <- setwd(file.path(\"..\",\"..\"))\n    cdir <- basename(dirname(Rdir))\n    system(paste(\"R CMD check\",cdir))\n  }\n  ## rebuild examples if we didn't test them:\n  if(check==\"noex\")package.skeleton.dx(pkgdir,code_files,\"\")\n}",  
         title = "Package skeleton deluxe", description = "Automates more of the setup process for a new source\npackage. After inspecting the specified R code files to find\ninline documentation, it calls the standard package.skeleton\nfunction, which creates bare Rd files. The inline documentation is\nadded to these Rd files and then these files are copied to\n../man. It will overwrite files in the pkgdir/man directory.",  
         `item{pkgdir}` = "package directory where the DESCRIPTION file lives. Your code\nshould be in pkgdir/R. We will setwd to pkgdir/R for the duration\nof the function, then switch back to where you were previously.",  
         `item{code_files}` = "Character vector with the names of the R code files, to be passed\nto package.skeleton, and also inspected for inline\ndocumentation. NULL means all files ending in .R.",  
         `item{check}` = "Value indicating whether or not to check the package after\ndocumentation is built. Anything other than \"\" (the default) will\ncheck the package. \"noex\" means check the package without running\nexamples, only tests (useful for debugging if your tests/examples\ntake a long time)."),  
     prefix = list(definition = "prefix <- \"^### \"", description = "Prefix for code comments used with grep and gsub.",
       format="",title="prefix"))
 
