% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summarizeCosts.R
\name{computeAvgTotCost}
\alias{computeAvgTotCost}
\title{Calculate the cumulative and average annual cost over a single period of time}
\usage{
computeAvgTotCost(
  costdb,
  cost.column = "Cost_estimate_per_year_2017_USD_exchange_rate",
  year.column = "Impact_year",
  min.year = NULL,
  max.year = NULL
)
}
\arguments{
\item{costdb}{The \bold{expanded 'InvaCost' database} output from 
\code{\link{expandYearlyCosts}},
where annual costs occurring over several years are repeated for each year.}

\item{cost.column}{Name of the cost column to use in \code{costdb} (usually, 
choose between the exchange rate (default) or Purchase Power Parity cost per 
year)}

\item{year.column}{Name of the year column to use in \code{costdb}.}

\item{min.year}{The minimum year of the period (specify only if different from
the range of data)}

\item{max.year}{The minimum year of the period (specify only if different from
the range of data)}
}
\value{
a named \code{list} with 5 elements
\itemize{
 \item{\code{initial_year}: first year in the data}
 \item{\code{final_year}: last year in the data}
 \item{\code{time_span}: the difference between initial and final years.}
 \item{\code{total_cost}: total cost.}
 \item{\code{annual.cost}: cost per year}
 \item{\code{number_estimates}: the number of cost estimates before expansion 
via \code{\link{expandYearlyCosts}}
 \item{\code{number_year_values}: the number of costs per year included}
}}
}
\description{
This simple function calculates the cumulative cost average annual cost of
invasive alien species over a single period of time. It is used internally
by \code{\link{summarizeCosts}}.
}
\note{
Arguments \code{min.year} and \code{max.year} do not filter the data. Only 
specify them if you wish to change the interval over which averages are 
calculated. For example, if your data have values from 1960 to 1964 but you
want to calculated the average value from 1960 to 1969, set 
\code{min.year = 1960} and \code{max.year = 1969}.

However, if you want to calculate values for an interval narrower than your
data, filter the data BEFORE running this function.
}
\examples{
data(invacost)

### Cleaning steps
# Eliminating data with no information on starting and ending years
invacost <- invacost[-which(is.na(invacost$Probable_starting_year_adjusted)), ]
invacost <- invacost[-which(is.na(invacost$Probable_ending_year_adjusted)), ]
# Keeping only observed and reliable costs
invacost <- invacost[invacost$Implementation == "Observed", ]
invacost <- invacost[which(invacost$Method_reliability == "High"), ]
# Eliminating data with no usable cost value
invacost <- invacost[-which(is.na(invacost$Cost_estimate_per_year_2017_USD_exchange_rate)), ]

### Expansion
\donttest{
db.over.time <- expandYearlyCosts(invacost,
                                  startcolumn = "Probable_starting_year_adjusted",
                                  endcolumn = "Probable_ending_year_adjusted")
                                  
### Analysis
res <- computeAvgTotCost(db.over.time,
                         min.year = 1960,
                         max.year = 2020) # Excluding data after 2020 (e.g. planned budgets)
res}
}
\references{
\url{https://github.com/Farewe/invacost}

Leroy Boris, Kramer Andrew M, Vaissière Anne-Charlotte, Courchamp Franck & Diagne Christophe (2020). Analysing global economic costs of invasive alien species with the
invacost R package. biorXiv. \url{https://doi.org/10.1101/2020.12.10.419432}
}
\seealso{
\code{\link{expandYearlyCosts}} to get the database in appropriate format.
}
\author{
Boris Leroy \email{leroy.boris@gmail.com}, Andrew Kramer, Anne-Charlotte
Vaissière, Christophe Diagne
}
