# Copyright 2010-2018 Meik Michalke <meik.michalke@hhu.de>
#
# This file is part of the R package koRpus.
#
# koRpus is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# koRpus is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with koRpus.  If not, see <http://www.gnu.org/licenses/>.


#' Getter/setter methods for koRpus objects
#' 
#' These methods should be used to get or set values of tagged text objects
#' generated by koRpus functions like \code{treetag()} or \code{tokenize()}.
#' 
#' \itemize{
#'   \item{\code{taggedText()} }{returns the \code{TT.res} slot.}
#'   \item{\code{describe()} }{returns the \code{desc} slot.}
#'   \item{\code{language()} }{returns the \code{lang} slot.}
#'   \item{\code{[}/\code{[[} }{Can be used as a shortcut to index the results of \code{taggedText()}.}
#'   \item{\code{fixObject} }{returns the same object upgraded to the object structure of this package version (e.g., new columns, changed names, etc.).}
#'   \item{\code{tif_as_tokens_df} }{returns the \code{TT.res} slot in a TIF[1] compliant format, i.e., \code{doc_id} is not a factor but a character vector.}
#' }
#' @param add.desc Logical, determines whether the \code{desc} column should be re-written with descriptions
#'    for all POS tags.
#' @param doc_id Logical (except for \code{fixObject}), if \code{TRUE} the \code{doc_id} column will be a factor with the respective value
#'    of the \code{desc} slot, i.\,e., the document ID will be preserved in the data.frame. If used with \code{fixObject}, can be a character string
#'    to set the document ID manually (the default \code{NA} will preserve existing values and not overwrite them).
#' @rdname kRp.taggedText-methods
#' @docType methods
#' @export
#' @references
#'    [1] Text Interchange Formats (\url{https://github.com/ropensci/tif})
#' @examples
#' \dontrun{
#' taggedText(tagged.txt)
#' }
setGeneric("taggedText", function(obj, add.desc=FALSE, doc_id=FALSE) standardGeneric("taggedText"))
#' @rdname kRp.taggedText-methods
#' @export
#' @docType methods
#' @aliases
#'    taggedText,-methods
#'    taggedText,kRp.taggedText-method
#' @include koRpus-internal.R
setMethod("taggedText",
  signature=signature(obj="kRp.taggedText"),
  function (obj, add.desc=FALSE, doc_id=FALSE){
    result <- slot(obj, name="TT.res")
    if(isTRUE(add.desc)){
      result[["desc"]] <- explain_tags(
        tags=result[["tag"]],
        lang=language(obj),
        cols="desc"
      )
    } else {}
    if(isTRUE(doc_id)){
      result[["doc_id"]] <- as.factor(describe(obj)[["doc_id"]])
    } else {}
    return(result)
  }
)

#' @rdname kRp.taggedText-methods
#' @export
#' @docType methods
#' @param value The new value to replace the current with.
setGeneric("taggedText<-", function(obj, value) standardGeneric("taggedText<-"))
#' @rdname kRp.taggedText-methods
#' @export
#' @docType methods
#' @aliases
#'    taggedText<-,-methods
#'    taggedText<-,kRp.taggedText-method
#' @include koRpus-internal.R
setMethod("taggedText<-",
  signature=signature(obj="kRp.taggedText"),
  function (obj, value){
    slot(obj, name="TT.res") <- value
    return(obj)
  }
)

#' @rdname kRp.taggedText-methods
#' @param x An object of class \code{kRp.taggedText} or \code{kRp.hyphen}.
#' @param i Defines the row selector (\code{[}) or the name to match (\code{[[}).
#' @param j Defines the column selector.
#' @export
#' @docType methods
#' @aliases
#'    [,-methods
#'    [,kRp.taggedText,ANY,ANY-method
#' @include koRpus-internal.R
setMethod("[",
  signature=signature(x="kRp.taggedText"),
  function (x, i, j){
    return(taggedText(x)[i, j])
  }
)

#' @rdname kRp.taggedText-methods
#' @export
#' @docType methods
#' @aliases
#'    [<-,-methods
#'    [<-,kRp.taggedText,ANY,ANY,ANY-method
#' @include koRpus-internal.R
setMethod("[<-",
  signature=signature(x="kRp.taggedText"),
  function (x, i, j, value){
    taggedText(x)[i, j] <- value
    return(x)
  }
)

#' @rdname kRp.taggedText-methods
#' @export
#' @docType methods
#' @aliases
#'    [[,-methods
#'    [[,kRp.taggedText,ANY-method
#' @include koRpus-internal.R
setMethod("[[",
  signature=signature(x="kRp.taggedText"),
  function (x, i){
    return(taggedText(x)[[i]])
  }
)

#' @rdname kRp.taggedText-methods
#' @export
#' @docType methods
#' @aliases
#'    [[<-,-methods
#'    [[<-,kRp.taggedText,ANY,ANY-method
#' @include koRpus-internal.R
setMethod("[[<-",
  signature=signature(x="kRp.taggedText"),
  function (x, i, value){
    taggedText(x)[[i]] <- value
    return(x)
  }
)

## the standard generic for describe() is defined in the sylly package
#' @importFrom sylly describe
#' @rdname kRp.taggedText-methods
#' @export
#' @docType methods
#' @aliases
#'    describe,-methods
#'    describe,kRp.taggedText-method
#' @include koRpus-internal.R
setMethod("describe",
  signature=signature(obj="kRp.taggedText"),
  function (obj){
    result <- slot(obj, name="desc")
    return(result)
  }
)

## the standard generic for describe()<- is defined in the sylly package
#' @importFrom sylly describe<-
#' @rdname kRp.taggedText-methods
#' @export
#' @docType methods
#' @aliases
#'    describe<-,-methods
#'    describe<-,kRp.taggedText-method
#' @include koRpus-internal.R
setMethod("describe<-",
  signature=signature(obj="kRp.taggedText"),
  function (obj, value){
    slot(obj, name="desc") <- value
    return(obj)
  }
)

## the standard generic for language() is defined in the sylly package
#' @importFrom sylly language
#' @rdname kRp.taggedText-methods
#' @export
#' @docType methods
#' @aliases
#'    language,-methods
#'    language,kRp.taggedText-method
#' @include koRpus-internal.R
setMethod("language",
  signature=signature(obj="kRp.taggedText"),
  function (obj){
    result <- slot(obj, name="lang")
    return(result)
  }
)

## the standard generic for language()<- is defined in the sylly package
#' @importFrom sylly language<-
#' @rdname kRp.taggedText-methods
#' @export
#' @docType methods
#' @aliases
#'    language<-,-methods
#'    language<-,kRp.taggedText-method
#' @include koRpus-internal.R
setMethod("language<-",
  signature=signature(obj="kRp.taggedText"),
  function (obj, value){
    slot(obj, name="lang") <- value
    return(obj)
  }
)

#' @param obj An arbitrary \code{R} object.
#' @rdname kRp.taggedText-methods
#' @export
is.taggedText <- function(obj){
  inherits(obj, "kRp.taggedText")
}


#' @rdname kRp.taggedText-methods
#' @docType methods
#' @export
setGeneric("fixObject", function(obj, doc_id=NA) standardGeneric("fixObject"))
#' @rdname kRp.taggedText-methods
#' @export
#' @docType methods
#' @aliases
#'    fixObject,-methods
#'    fixObject,kRp.taggedText-method
setMethod("fixObject",
  signature=signature(obj="kRp.taggedText"),
  function (obj, doc_id=NA){
    currentDf <- slot(obj, "TT.res")
    currentDesc <- slot(obj, "desc")
    currentCols <- colnames(currentDf)
    newDf <- init.kRp.tagged.df(rows=nrow(currentDf))
    # move all present columns to the new data.frame
    newDf[,currentCols] <- currentDf[,currentCols]
    # adjust column classes where needed

    lang <- slot(obj, "lang")
    tag.class.def <- kRp.POS.tags(lang)
    for (thisCol in c("tag","wclass","desc")){
      if(
        all(
          !is.factor(newDf[[thisCol]]),
          any(
            !thisCol %in% "desc",
            !all(is.na(newDf[[thisCol]]))
          )
        )
      ){
        # make tag a factor with all possible tags for this language as levels
        newDf[[thisCol]] <- factor(
          newDf[[thisCol]],
          levels=unique(tag.class.def[,thisCol])
        )
      } else {}
    }
    newDf <- indexSentenceDoc(newDf, lang=lang, doc_id=doc_id)

    # fix desc slot
    if(any(!"doc_id" %in% names(currentDesc), !is.na(doc_id))){
      currentDesc[["doc_id"]] <- doc_id
    } else {}

    slot(obj, "TT.res") <- newDf
    slot(obj, "desc") <- currentDesc

    return(obj)
  }
)

#' @rdname kRp.taggedText-methods
#' @docType methods
#' @export
setGeneric("tif_as_tokens_df", function(tokens) standardGeneric("tif_as_tokens_df"))
#' @rdname kRp.taggedText-methods
#' @param tokens An object of class \code{\link[koRpus:kRp.tagged-class]{kRp.tagged}}.
#' @export
#' @docType methods
#' @aliases
#'    tif_as_tokens_df,-methods
#'    tif_as_tokens_df,kRp.taggedText-method
setMethod("tif_as_tokens_df",
  signature=signature(tokens="kRp.taggedText"),
  function(tokens){
    result <- taggedText(tokens)
    # TIF needs doc_id to be a character vector, not a factor
    result[["doc_id"]] <- as.character(result[["doc_id"]])
    return(result)
  }
)
