% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Methods-LandsepiParams.R
\name{runSimul}
\alias{runSimul}
\title{Run a simulation}
\usage{
runSimul(
  params,
  graphic = TRUE,
  writeTXT = TRUE,
  videoMP4 = FALSE,
  keepRawResults = FALSE
)
}
\arguments{
\item{params}{a LandsepiParams Object containing all simulation parameters. Must be initialised
with \code{\link{createSimulParams}} and updated using \code{set*()} methods
(see vignettes for details).}

\item{graphic}{a logical indicating if graphics must be generated (TRUE, default)
or not (FALSE).}

\item{writeTXT}{a logical indicating if outputs must be written in text files (TRUE, default)
or not (FALSE).}

\item{videoMP4}{a logical indicating if a video must be generated (TRUE) or not (FALSE, default).
Works only if graphic=TRUE and audpc is computed.}

\item{keepRawResults}{a logical indicating if binary files must be kept after the end of
the simulation (default=FALSE). Careful, many files may be generated if keepRawResults=TRUE.}
}
\value{
A list containing all required outputs.
A set of text files, graphics and a video showing epidemic dynamics can be generated.
If keepRawResults=TRUE, a set of binary files is generated for every year of simulation and
every compartment: \itemize{
\item H: healthy hosts,
\item Hjuv: juvenile healthy hosts,
\item L: latently infected hosts,
\item I: infectious hosts,
\item R: removed hosts,
\item P: propagules.}
Each file indicates for every time-step the number of individuals in each field, and when
appropriate for each host and pathogen genotypes.
}
\description{
Runs a simulation with landsepi,
a stochastic, spatially-explicit, demo-genetic model simulating the spread and evolution
of a pathogen in a heterogeneous landscape and generating a wide range of epidemiological,
evolutionary and economic outputs.
}
\details{
See ?landsepi for details on the model, assumptions and outputs, and our vignettes
for tutorials (\code{browseVignettes("landsepi")}). The function runs the model simulation using
a LandsepiParams object.
Briefly, the model is stochastic, spatially explicit (the basic spatial unit is an
individual field), based on a SEIR (‘susceptible-exposed-infectious-removed’, renamed HLIR
for 'healthy-latent-infectious-removed' to avoid confusions
with 'susceptible host') structure with a discrete time step. It simulates the spread and
evolution of a pathogen in a heterogeneous cropping landscape, across cropping seasons split
by host harvests which impose potential bottlenecks to the pathogen. A wide array of
resistance deployment strategies can be simulated and evaluated using several possible
outputs to assess the epidemiological, evolutionary and economic performance
of deployment strategies.
}
\examples{
\dontrun{
## Here is an example of simulation of a mosaic of three cultivars (S + R1 + R2). See our 
## tutorials for more examples.
## Initialisation
simul_params <- createSimulParams(outputDir = getwd())
## Seed & Time parameters
simul_params <- setSeed(simul_params, seed = 1)
simul_params <- setTime(simul_params, Nyears = 10, nTSpY = 120)
## Pathogen & inoculum parameters
simul_params <- setPathogen(simul_params, loadPathogen("rust"))
simul_params <- setInoculum(simul_params, 5e-4)
## Landscape & dispersal
simul_params <- setLandscape(simul_params, loadLandscape(1))
simul_params <- setDispersalPathogen(simul_params, loadDispersalPathogen(1))
## Genes
gene1 <- loadGene(name = "MG 1", type = "majorGene")
gene2 <- loadGene(name = "MG 2", type = "majorGene")
genes <- data.frame(rbind(gene1, gene2), stringsAsFactors = FALSE)
simul_params <- setGenes(simul_params, genes)
## Cultivars
cultivar1 <- loadCultivar(name = "Susceptible", type = "growingHost")
cultivar2 <- loadCultivar(name = "Resistant1", type = "growingHost")
cultivar3 <- loadCultivar(name = "Resistant2", type = "growingHost")
cultivars <- data.frame(rbind(cultivar1, cultivar2, cultivar3), stringsAsFactors = FALSE)
simul_params <- setCultivars(simul_params, cultivars)
## Allocate genes to cultivars
simul_params <- allocateCultivarGenes(simul_params, "Resistant1", c("MG 1"))
simul_params <- allocateCultivarGenes(simul_params, "Resistant2", c("MG 2"))
## Allocate cultivars to croptypes
croptypes <- loadCroptypes(simul_params, names = c("Susceptible crop"
, "Resistant crop 1"
, "Resistant crop 2"))
croptypes <- allocateCroptypeCultivars(croptypes, "Susceptible crop", "Susceptible")
croptypes <- allocateCroptypeCultivars(croptypes, "Resistant crop 1", "Resistant1")
croptypes <- allocateCroptypeCultivars(croptypes, "Resistant crop 2", "Resistant2")
simul_params <- setCroptypes(simul_params, croptypes)
## Allocate croptypes to landscape        
rotation_sequence <- croptypes$croptypeID ## No rotation -> 1 rotation_sequence element
rotation_period <- 0 ## same croptypes every years
prop <- c(1 / 3, 1 / 3, 1 / 3) ## croptypes proportions
aggreg <- 10 ## aggregated landscape
simul_params <- allocateLandscapeCroptypes(simul_params, rotation_period = rotation_period,
rotation_sequence = rotation_sequence,
rotation_realloc = FALSE, prop = prop, aggreg = aggreg)
## list of outputs to be generated
simul_params <- setOutputs(simul_params, loadOutputs())
## Check simulation parameters
checkSimulParams(simul_params)
## Save deployment strategy into GPKG file
simul_params <- saveDeploymentStrategy(simul_params)
## Run simulation
runSimul(simul_params)
}
}
\references{
Rimbaud L., Papaïx J., Rey J.-F., Barrett L. G. and Thrall P. H. (2018).
Assessing the durability andefficiency of landscape-based strategies to deploy plant
resistance to pathogens. \emph{PLoS Computational Biology} 14(4):e1006067.
}
