


### Copyright 2001-2002 Deepayan Sarkar <deepayan@stat.wisc.edu>
###
### This file is part of the lattice library for R.
### It is made available under the terms of the GNU General Public
### License, version 2, or at your option, any later version,
### incorporated herein by reference.
###
### This program is distributed in the hope that it will be
### useful, but WITHOUT ANY WARRANTY; without even the implied
### warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
### PURPOSE.  See the GNU General Public License for more
### details.
###
### You should have received a copy of the GNU General Public
### License along with this program; if not, write to the Free
### Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
### MA 02111-1307, USA


lpretty <- function(x, ...) { 
    eps <- 1e-10
    at <- pretty(x[is.finite(x)], ...)
    at <- ifelse(abs(at-round(at, 3))<eps, round(at, 3), at)
}


oneway <-
    function(formula, data, location = mean,
             spread = function(x) sqrt(var(x)))
{
    if(missing(data)) data <- sys.frame(sys.parent())
    form <- latticeParseFormula(formula, data)
    y <- form$left
    x <- form$right
    if (!is.shingle(x)) x <- as.factor(x)
    is.f.x <- is.factor(x)
    num.l.x <- nlevels(x) 
    foo <- list()
    if (is.f.x) {
        foo$location <-
            if (is.function(location)) as.vector(tapply(X=y, INDEX=list(x), FUN = location))
            else rep(location, num.l.x)
        foo$spread <- 
            if (is.function(spread)) as.vector(tapply(X=y, INDEX=list(x), FUN = spread))
            else rep(spread, num.l.x)
        foo$fitted.values <- numeric(length(y))
        sc <- numeric(length(y))
        for (i in seq(along = y)){
            foo$fitted.values[i] <- foo$location[as.numeric(x)[i]]
            sc[i] <- foo$spread[as.numeric(x)[i]]
        }
        foo$residuals <- y - foo$fitted.values
        foo$scaled.residuals <- foo$residuals/sc
    }
    else stop("x must be (coercible to be) a factor")
    foo
}


do.breaks  <- function(endpoints, nint)
{
    if (length(endpoints)!=2) stop("error")
    endpoints[1] + diff(endpoints) * 0:nint / nint
}


is.characterOrExpression <- function(x)
    is.character(x) || is.expression(x)




## This converts character to factor, numeric to shingle, and
## in addition, takes subsets
as.factorOrShingle <- function(x, subset = TRUE, drop = FALSE)
{
    x <-
        if (is.numeric(x))
            as.shingle(x)
        else ##if (is.character(x)) or logical or ??
            as.factor(x)
    x[subset, drop = drop]
}



"[.shingle" <-
    function(x, subset, drop = FALSE)
{
    if (!is.shingle(x)) stop("x must be a shingle")
    ans <- as.numeric(x)[subset]
    attr(ans, "levels") <- levels(x)
    class(attr(ans, "levels")) <- "shingleLevel"
    if (drop) {
        xlvs <- levels(ans)
        dl <- logical(nlevels(ans))
        for (i in seq(along=dl))
            dl[i] <- any( ans >= xlvs[[i]][1] & ans <= xlvs[[i]][2] )
        attr(ans, "levels") <- xlvs[dl]
        class(attr(ans, "levels")) <- "shingleLevel"
    }
    class(ans) <- "shingle"
    ans
}



Rows <- function(x, which)
{
    for (i in seq(along = x)) x[[i]] <- x[[i]][which]
    x
}
## S-Plus trellis function needed for nlme.



make.list.from.intervals <- function(x)
{
    if (ncol(x)!=2) stop("x must be matrix with 2 columns")
    if (nrow(x)<1) stop("x must be matrix with at least 1 row")
    ans <- as.list(1:nrow(x))
    for (i in 1:nrow(x))
        ans[[i]] <- x[i,]
    ans
}



equal.count <-
  function(x, ...)
{
    attr(x, "levels") <- make.list.from.intervals(co.intervals(x,...))
    class(attr(x, "levels")) <- "shingleLevel"
    class(x) <- "shingle"
    x
}



shingle <-
    function(x, intervals=sort(unique(x)))
{
    if (ncol(as.matrix(intervals))==1)
        intervals <- cbind(intervals, intervals)
    else if (ncol(as.matrix(intervals)) > 2)
        stop("bad value of 'intervals'")
    attr(x, "levels") <- make.list.from.intervals(intervals)
    class(attr(x, "levels")) <- "shingleLevel"
    class(x) <- "shingle"
    x
}


as.data.frame.shingle <- as.data.frame.factor

is.shingle <-
    function(x) inherits(x, "shingle")


as.shingle <-
    function(x) if (is.shingle(x)) x else shingle(x)



summary.shingle <- function(object, ...) print.shingle(object, ...)


print.shingleLevel <-
    function(x, ...) {
        print(do.call("rbind", x))
        invisible(x)
    }

print.shingle <- function(x, showValues = TRUE, ...) {
    cat("\nData:\n")
    if (showValues) print(as.numeric(x))
    l <- levels(x)
    n <- nlevels(x)
    if (n<1) cat("\nno intervals\n")
    else {
        int <- data.frame(min = numeric(n), max = numeric(n), count = numeric(n))
        for (i in 1:n) {
            int$min[i] <- l[[i]][1]
            int$max[i] <- l[[i]][2]
            int$count[i] <- length(x[x>=l[[i]][1] & x<=l[[i]][2]])
        }
        cat("\nIntervals:\n")
        print(int)
        olap <- numeric(n-1)
        if (n>2)
            for (i in 1:(n-1))
                olap[i] <- length(x[ x>=l[[i]][1] & x<=l[[i]][2] &
                                    x>=l[[i+1]][1] & x<=l[[i+1]][2]])
        cat("\nOvrlap between adjacent intervals:\n")
        print(olap)
    }
    invisible(x)
}





strip.default <-
    function(which.given,
             which.panel,
             var.name,
             factor.levels,
             shingle.intervals,
             strip.names = c(FALSE, TRUE),
             style = 1,
             bg = trellis.par.get("strip.background")$col[which.given],
             fg = trellis.par.get("strip.shingle")$col[which.given],
             par.strip.text = trellis.par.get("add.text"))
{
    default.fontsize <- trellis.par.get("fontsize")$default
    name <- var.name[which.given]
    level <- which.panel[which.given]
    strip.names <- rep(strip.names, length = 2)
    
    if (is.null(factor.levels)) { # means this is a  shingle, as opposed to a factor
        if (is.null(shingle.intervals)) stop("both factor.levels and shingle.intervals cannot be NULL")
        strip.names <- strip.names[2]
        grid.rect(gp = gpar(fill=bg))
        t <- range(shingle.intervals)
        r <- (range(shingle.intervals[level,])-t[1])/diff(t)
        grid.rect(x = unit(r%*%c(.5,.5),"npc"), width = max(unit( c(diff(r), 1), c("npc", "mm"))),
                  gp = gpar(col=fg, fill=fg))
        if (strip.names)
            grid.text(label = name,
                      gp = gpar(col = par.strip.text$col,
                      font = par.strip.text$font,
                      fontsize = par.strip.text$cex *
                      default.fontsize))
        grid.rect()
    }
    else if (is.null(shingle.intervals)) { # factor
        strip.names <- strip.names[1]
        x <- factor.levels
        num <- length(x)
        if (style == 1) {
            grid.rect(gp = gpar(fill=bg))
            if (strip.names) {
                grid.text(name,
                          x=unit(0.5, "npc") - unit(1, "mm"),
                          gp = gpar(col = par.strip.text$col,
                          font = par.strip.text$font,
                          fontsize = par.strip.text$cex *
                          default.fontsize),
                          just="right")
                grid.text(":",
                          x=unit(0.5, "npc"),
                          gp = gpar(col = par.strip.text$col,
                          font = par.strip.text$font,
                          fontsize = par.strip.text$cex *
                          default.fontsize))
                grid.text(x[level],
                          x=unit(0.5, "npc") + unit(1, "mm"),
                          gp = gpar(col = par.strip.text$col,
                          font = par.strip.text$font,
                          fontsize = par.strip.text$cex *
                          default.fontsize),
                          just="left")
            }
            else grid.text(label = x[level],
                           gp = gpar(col = par.strip.text$col,
                           font = par.strip.text$font,
                           fontsize = par.strip.text$cex *
                           default.fontsize))
            grid.rect()
        }
        else if (style == 2) {
            grid.rect(x = unit((2*level-1)/(2*num), "npc"),
                      width = unit(1/num, "npc"),
                      gp = gpar(fill=fg, col = NULL))
            grid.text(label=x,
                      x = (2*1:num-1)/(2*num),
                      gp = gpar(col = par.strip.text$col,
                      font = par.strip.text$font,
                      fontsize = par.strip.text$cex *
                      default.fontsize))
            grid.rect()
        }
        else if (style == 3){
            grid.rect(gp = gpar(fill=bg))
            grid.rect(x = unit((2*level-1)/(2*num), "npc"),
                      width = unit(1/num, "npc"),
                      gp = gpar(fill=fg, col = NULL))
            grid.text(label =
                      if (strip.names) paste(name, x[level], sep = ": ")
                      else x[level],
                      gp = gpar(col = par.strip.text$col, 
                      font = par.strip.text$font,
                      fontsize = par.strip.text$cex *
                      default.fontsize))
            grid.rect()
        }
        else if(style == 4){
            grid.rect(gp = gpar(fill=bg))
            grid.rect(x = unit((2*level-1)/(2*num), "npc"),
                      width = unit(1/num, "npc"),
                      gp = gpar(col=NULL, fill=fg))
            grid.text(label=x,
                      x = (2* 1:num - 1)/(2*num),   #using default.units
                      gp = gpar(col = par.strip.text$col, 
                      font = par.strip.text$font,
                      fontsize = par.strip.text$cex *
                      default.fontsize))
            grid.rect()
        }
        else if(style >= 5){
            grid.rect(gp = gpar(fill=bg))
            grid.text(label=x[level],
                      x = (2* level - 1)/(2*num),   #using default.units
                      gp = gpar(col = par.strip.text$col, 
                      font = par.strip.text$font,
                      fontsize = par.strip.text$cex *
                      default.fontsize))
            grid.rect()
        }
    }
}







lsegments <-
    function(x0, y0, x1, y1, 
             col = add.line$col,
             lty = add.line$lty,
             lwd = add.line$lwd, ...)
{
    add.line <- trellis.par.get("add.line")
    ml <- max(length(x0), length(x1), length(y0), length(y1))
    x0 <- rep(x0, length = ml)
    x1 <- rep(x1, length = ml)
    y0 <- rep(y0, length = ml)
    y1 <- rep(y1, length = ml)

    grid.segments(x0 = x0, x1 = x1,
                  y0 = y0, y1 = y1,
                  gp = gpar(lty=lty,
                  col=col, lwd=lwd),
                  default.units="native")
}


larrows <-
    function(x0, y0, x1, y1, angle = 30, code = 2, length = NULL, proportion = .05, ...) 
{
    if (!is.null(length)) warning("length not implemented in larrows, use proportion instead")
    ##warning("larrows not corectly implemented yet")
    angle <- angle / 180 * pi
    start <- rbind(x0, y0)
    end <- rbind(x1, y1)
    v.forward <- end - start
    v.backward <- start - end
    lsegments(x0, y0, x1, y1, ...)
    
    if (code %in% c(1,3)) { # arrow at starting point
        edge.1 <- proportion * 
            matrix( c(cos(angle), -sin(angle), sin(angle), cos(angle)), 2, 2) %*% v.forward
        edge.2 <- proportion *
            matrix( c(cos(-angle), -sin(-angle), sin(-angle), cos(-angle)), 2, 2) %*% v.forward
        lsegments(x0, y0, x0 + edge.1[1,], y0 + edge.1[2,], ...)
        lsegments(x0, y0, x0 + edge.2[1,], y0 + edge.2[2,], ...)
    }
    if (code %in% c(2,3)) { # arrow at ending point
        edge.1 <- proportion * 
            matrix( c(cos(angle), -sin(angle), sin(angle), cos(angle)), 2, 2) %*% v.backward
        edge.2 <- proportion *
            matrix( c(cos(-angle), -sin(-angle), sin(-angle), cos(-angle)), 2, 2) %*% v.backward
        lsegments(x1, y1, x1 + edge.1[1,], y1 + edge.1[2,], ...)
        lsegments(x1, y1, x1 + edge.2[1,], y1 + edge.2[2,], ...)
    }
}



ltext <-
    function(x, y = NULL, labels = seq(along = x),
             col = add.text$col,
             cex = add.text$cex,
             srt = 0,
             font = 1,
             adj = c(.5, .5),
             pos,
             ...)
{
    add.text <- trellis.par.get("add.text")
    xy <- xy.coords(x, y)
    if (!missing(pos))
        adj <-
            if (pos == 1) c(.5, 1)
            else if (pos == 2) c(1, .5)
            else if (pos == 3) c(.5, 0)
            else if (pos == 4) c(0, .5)
    if (length(adj) == 1) adj <- c(adj, .5)
    grid.text(label = labels, x = xy$x, y = xy$y,
              gp = gpar(col = col, font = font,
              fontsize = cex * trellis.par.get("fontsize")$default),
              just = c(if (adj[1] == 0) "left"
              else if (adj[1] == 1) c("right")
              else "centre",
              if (adj[2] == 0) "bottom"
              else if (adj[2] == 1) c("top")
              else "centre"),
              rot = srt,
              default.units = "native")
}





llines <-
    function(x, y = NULL, type = "l",
             col = plot.line$col,
             lty = plot.line$lty,
             lwd = plot.line$lwd, ...)
{
    plot.line <- trellis.par.get("plot.line")
    lplot.xy(xy.coords(x, y), type = type,
             col = col, lty = lty, lwd = lwd, ...)
}




lpoints <-
    function(x, y = NULL, type = "p",
             col = plot.symbol$col,
             pch = plot.symbol$pch,
             cex = plot.symbol$cex, ...)
{
    plot.symbol <- trellis.par.get("plot.symbol")
    lplot.xy(xy.coords(x, y), type = type,
             col = col, pch = pch, cex = cex, ...)
}






lplot.xy <-
    function(xy, type, pch = 1, lty = 1, col = 1, cex = 1, lwd = 1, font = 1, ...)
{
    x <- xy$x
    y <- xy$y

    if (type %in% c("l", "o", "b", "c"))
        grid.lines(x=x, y=y, gp = gpar(lty=lty, col=col, lwd=lwd),
                   default.units="native")
    
    if (type %in% c("p", "o", "b", "c"))
        grid.points(x = x, y = y, size = unit(cex * 2.5, "mm"),
                    gp = gpar(col = col),
                    pch = pch, 
                    default.units="native")

    if (type %in% c("s", "S")) {
        ord <- order(x)
        n <- length(x)
        xx <- numeric(2*n-1)
        yy <- numeric(2*n-1)

        xx[2*1:n-1] <- x[ord]
        yy[2*1:n-1] <- y[ord]
        xx[2*1:(n-1)] <- x[ord][if (type=="s") -1 else -n]
        yy[2*1:(n-1)] <- y[ord][if (type=="s") -n else -1]
        grid.lines(x=xx, y=yy,
                   gp = gpar(lty=lty, col=col, lwd=lwd),
                   default.units="native")
    }

    if (type == "h") {

        ylim <- current.viewport()$yscale

        zero <-
            if (ylim[1] > 0) ylim[1]
            else if (ylim[2] < 0) ylim[2]
            else 0

        ##print(zero) ?
        ##print(x) 
        for (i in seq(along=x))
            grid.lines(x=rep(x[i],2), y=c(y[i], zero),
                       gp = gpar(lty=lty, col=col, lwd=lwd),
                       default.units="native")
    }
}







