% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compare2.R
\name{compare2}
\alias{compare2}
\alias{compare2.lm}
\alias{compare2.gls}
\alias{compare2.lme}
\alias{compare2.lvmfit}
\alias{compare2.lm2}
\alias{compare2.gls2}
\alias{compare2.lme2}
\alias{compare2.lvmfit2}
\alias{.compare2}
\title{Test Linear Hypotheses with small sample correction}
\usage{
compare2(object, df, bias.correct, ...)

\method{compare2}{lm}(object, df = TRUE, bias.correct = TRUE, ...)

\method{compare2}{gls}(object, df = TRUE, bias.correct = TRUE,
  cluster = NULL, ...)

\method{compare2}{lme}(object, df = TRUE, bias.correct = TRUE, ...)

\method{compare2}{lvmfit}(object, df = TRUE, bias.correct = TRUE,
  cluster = NULL, ...)

\method{compare2}{lm2}(object, ...)

\method{compare2}{gls2}(object, ...)

\method{compare2}{lme2}(object, ...)

\method{compare2}{lvmfit2}(object, ...)

.compare2(object, par = NULL, contrast = NULL, null = NULL,
  rhs = NULL, robust = FALSE, cluster = NULL, df = TRUE,
  as.lava = TRUE, F.test = TRUE, level = 0.95)
}
\arguments{
\item{object}{an object that inherits from lm/gls/lme/lvmfit.}

\item{df}{[logical] should the degree of freedoms of the Wald statistic be computed using the Satterthwaite correction?
Otherwise the degree of freedoms are set to \code{Inf}, i.e. a normal distribution is used instead of a Student's t distribution when computing the p-values.}

\item{bias.correct}{[logical] should the standard errors of the coefficients be corrected for small sample bias? Argument passed to \code{sCorrect}.}

\item{...}{[internal] only used by the generic method.}

\item{cluster}{[integer vector] the grouping variable relative to which the observations are iid.}

\item{par}{[vector of characters] expression defining the linear hypotheses to be tested.
See the examples section.}

\item{contrast}{[matrix] a contrast matrix defining the left hand side of the linear hypotheses to be tested.}

\item{null, rhs}{[vector] the right hand side of the linear hypotheses to be tested.}

\item{robust}{[logical] should the robust standard errors be used instead of the model based standard errors?}

\item{as.lava}{[logical] should the output be similar to the one return by \code{lava::compare}?}

\item{F.test}{[logical] should a joint test be performed?}

\item{level}{[numeric 0-1] the confidence level of the confidence interval.}
}
\value{
If \code{as.lava=TRUE} an object of class \code{htest}.
Otherwise a \code{data.frame} object.
}
\description{
Test Linear Hypotheses using a multivariate Wald statistic.
Similar to \code{lava::compare} but with small sample correction.
}
\details{
The \code{par} argument or the arguments \code{contrast} and \code{null} (or equivalenty \code{rhs})
specify the set of linear hypotheses to be tested. They can be written:
\deqn{
  contrast * \theta = null
}
where \eqn{\theta} is the vector of the model coefficients. \cr
The \code{par} argument must contain expression(s) involving the model coefficients.
For example \code{"beta = 0"} or \code{c("-5*beta + alpha = 3","-alpha")} are valid expressions if alpha and beta belong to the set of model coefficients.
A contrast matrix and the right hand side will be generated inside the function. \cr

When directly specified, the contrast matrix must contain as many columns as there are coefficients in the model (mean and variance coefficients).
Each hypothesis correspond to a row in the contrast matrix. \cr

The null vector should contain as many elements as there are row in the contrast matrix. \cr

Argument rhs and null are equivalent.
This redondance enable compatibility between \code{lava::compare}, \code{compare2}, \code{multcomp::glht}, and \code{glht2}.
}
\examples{
#### simulate data ####
set.seed(10)
mSim <- lvm(Y~0.1*X1+0.2*X2)
categorical(mSim, labels = c("a","b","c")) <- ~X1
transform(mSim, Id~Y) <- function(x){1:NROW(x)}
df.data <- lava::sim(mSim, 1e2)

#### with lm ####
## direct use of compare2
e.lm <- lm(Y~X1+X2, data = df.data)
anova(e.lm)
compare2(e.lm, par = c("X1b=0","X1c=0"))

## or first compute the derivative of the information matrix
sCorrect(e.lm) <- TRUE

## and define the contrast matrix
C <- createContrast(e.lm, par = c("X1b=0","X1c=0"), add.variance = TRUE)

## run compare2
compare2(e.lm, contrast = C$contrast, null = C$null)
compare2(e.lm, contrast = C$contrast, null = C$null, robust = TRUE)

#### with gls ####
library(nlme)
e.gls <- gls(Y~X1+X2, data = df.data, method = "ML")

## first compute the derivative of the information matrix
sCorrect(e.gls, cluster = 1:NROW(df.data)) <- TRUE

compare2(e.gls, par = c("5*X1b+2*X2 = 0","(Intercept) = 0"))

#### with lvm ####
m <- lvm(Y~X1+X2)
e.lvm <- estimate(m, df.data)

compare2(e.lvm, par = c("-Y","Y~X1b+Y~X1c"))
}
\seealso{
\code{\link{createContrast}} to create contrast matrices. \cr
\code{\link{sCorrect}} to pre-compute quantities for the small sample correction.
}
\concept{small sample inference}
