% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rasterize.R, R/rasterize_canopy.R,
%   R/rasterize_density.R, R/rasterize_terrain.R
\name{rasterize}
\alias{rasterize}
\alias{rasterize_canopy}
\alias{rasterize_density}
\alias{rasterize_terrain}
\title{Rasterize a point cloud}
\usage{
rasterize_canopy(las, res = 1, algorithm = p2r(), ...)

rasterize_density(las, res = 4, ...)

rasterize_terrain(
  las,
  res = 1,
  algorithm = tin(),
  use_class = c(2L, 9L),
  shape = "convex",
  ...
)
}
\arguments{
\item{las}{An object of class \link[lidR:LAS-class]{LAS} or \link[lidR:LAScatalog-class]{LAScatalog}.}

\item{res}{numeric. The size of a grid cell in point cloud coordinates units. Can also be
\code{RasterLayer} or a \code{stars} or a \code{SpatRaster} used as layout.}

\item{algorithm}{function. A function that implements an algorithm to compute a digital surface model
or a digital terrain model. \code{lidR} implements \link{p2r}, \link{dsmtin}, \link{pitfree}
for digital surface models, and \link{knnidw}, \link{tin}, and \link{kriging} for digital terrain
models (see respective documentation and examples).}

\item{...}{Use \code{pkg = "terra|raster|stars"} to get an output in \code{SpatRaster}, \code{RasterLayer}
or \code{stars} format. Default is \code{getOption("lidR.raster.default")}.}

\item{use_class}{integer vector. By default the terrain is computed by using ground points
(class 2) and water points (class 9).}

\item{shape}{By default the interpolation is made only within the \code{"convex"} hull of
the point cloud to get a DTM with the shape of the point cloud. This prevents meaningless
interpolations where there is no data. It can also be \code{"concave"} or \code{"bbox"}. It can also be an \code{sfc}
to define a polygon in which to perform the interpolation.}
}
\value{
\code{RasterLayer} or a \code{stars} or a \code{SpatRaster} depending on the settings.
}
\description{
Rasterize a point cloud in different ways to compute a DTM, a CHM or a density map. Most
raster products can be computed with \link{pixel_metrics} but some are more complex and require
dedicated and optimized functions. See Details and Examples.
}
\details{
\describe{
\item{\code{rasterize_terrain}}{Interpolates the ground points and creates a rasterized
digital terrain model. The algorithm uses the points classified as "ground" and "water"
(Classification = 2 and 9, respectively, according to
\href{https://www.asprs.org/wp-content/uploads/2019/07/LAS_1_4_r15.pdf}{LAS file format specifications})
to compute the interpolation. How well the edges of the dataset are interpolated depends on the
interpolation method used. A buffer around the region of interest is always recommended to avoid
edge effects.}
\item{\code{rasterize_canopy}}{Creates a digital surface model (DSM) using several
possible algorithms. If the user provides a normalized point cloud, the output is indeed a canopy
height model (CHM).}
\item{\code{rasterize_density}}{Creates a map of the point density. If a "pulseID"
attribute is found, also returns a map of the pulse density.}
}
}
\section{Non-supported LAScatalog options}{

The option \code{select} is not supported and not respected in \verb{rasterize_*} because it is internally
known what is best to select.\cr
The option \code{chunk_buffer} is not supported and not respected in \code{rasterize_canopy} and
\code{rasterize_density} because it is not necessary.
}

\examples{

# =====================
# Digital Terrain Model
# =====================

LASfile <- system.file("extdata", "Topography.laz", package="lidR")
las = readLAS(LASfile, filter = "-inside 273450 5274350 273550 5274450")
#plot(las)

dtm1 = rasterize_terrain(las, algorithm = knnidw(k = 6L, p = 2))
dtm2 = rasterize_terrain(las, algorithm = tin())

\dontrun{
dtm3 = rasterize_terrain(las, algorithm = kriging(k = 10L))

plot(dtm1, col = gray(0:25/25))
plot(dtm2, col = gray(0:25/25))
plot(dtm3, col = gray(0:25/25))
plot_dtm3d(dtm1)
plot_dtm3d(dtm2)
plot_dtm3d(dtm3)
}

# =====================
# Digital Surface Model
# =====================

LASfile <- system.file("extdata", "MixedConifer.laz", package="lidR")
las <- readLAS(LASfile, filter = "-inside 481280 3812940 481330 3812990")
col <- height.colors(15)

# Points-to-raster algorithm with a resolution of 1 meter
chm <- rasterize_canopy(las, res = 1, p2r())
plot(chm, col = col)

# Points-to-raster algorithm with a resolution of 0.5 meters replacing each
# point by a 20-cm radius circle of 8 points
chm <- rasterize_canopy(las, res = 0.5, p2r(0.2))
plot(chm, col = col)

# Basic triangulation and rasterization of first returns
chm <- rasterize_canopy(las, res = 0.5, dsmtin())
plot(chm, col = col)

# Khosravipour et al. pitfree algorithm
chm <- rasterize_canopy(las, res = 0.5, pitfree(c(0,2,5,10,15), c(0, 1.5)))
plot(chm, col = col)

# ====================
# Digital Density Map
# ====================

LASfile <- system.file("extdata", "Megaplot.laz", package="lidR")
las <- readLAS(LASfile,  filter = "-inside 684800 5017800 684900 5017900")

d <- rasterize_density(las, 5)
plot(d)

las <- retrieve_pulses(las)
d <- rasterize_density(las)
plot(d)
}
