% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lingmatch.R
\name{lma_simets}
\alias{lma_simets}
\title{Similarity Calculations}
\usage{
lma_simets(a, b = NULL, metric = NULL, group = NULL, lag = 0,
  agg = TRUE, agg.mean = TRUE, pairwise = TRUE, symmetrical = FALSE,
  mean = FALSE, return.list = FALSE)
}
\arguments{
\item{a}{A vector or matrix. If a vector, \code{b} must also be provided. If a matrix and \code{b}
is missing, each row will be compared. If a matrix and \code{b} is not missing, each row will
be compared with \code{b} or each row of \code{b}.}

\item{b}{A vector or matrix to be compared with \code{a} or rows of \code{a}.}

\item{metric}{A character or vector of characters at least partially matching one of the
available metric names (or 'all' to explicitly include all metrics),
or a number or vector of numbers indicating the metric by index:
\itemize{
  \item \strong{\code{jaccard}}: \code{sum(a & b) / sum(a | b)}
  \item \strong{\code{euclidean}}: \code{1 / (1 + sqrt(sum((a - b) ^ 2)))}
  \item \strong{\code{canberra}}: \code{mean(1 - abs(a - b) / (a + b))}
  \item \strong{\code{cosine}}: \code{sum(a * b) / sqrt(sum(a ^ 2 * sum(b ^ 2)))}
  \item \strong{\code{pearson}}: \code{(mean(a * b) - (mean(a) * mean(b))) /} \cr
    \code{sqrt(mean(a ^ 2) - mean(a) ^ 2) / sqrt(mean(b ^ 2) - mean(b) ^ 2)}
}}

\item{group}{If \code{b} is missing and \code{a} has multiple rows, this will be used to make
comparisons between rows of \code{a}, as modified by \code{agg} and \code{agg.mean}.}

\item{lag}{Amount to adjust the \code{b} index; either rows if \code{b} has multiple rows (e.g.,
for \code{lag = 1}, \code{a[1, ]} is compared with \code{b[2, ]}), or values otherwise (e.g.,
for \code{lag = 1}, \code{a[1]} is compared with \code{b[2]}). If \code{b} is not supplied,
\code{b} is a copy of \code{a}, resulting in lagged self-comparisons or autocorrelations.}

\item{agg}{Logical: if \code{FALSE}, only the boundary rows between groups will be compared, see
example.}

\item{agg.mean}{Logical: if \code{FALSE} aggregated rows are summed instead of averaged.}

\item{pairwise}{Logical: if \code{FALSE} and \code{a} and \code{b} are matrices with the same number of
rows, only paired rows are compared. Otherwise (and if only \code{a} is supplied), all pairwise
comparisons are made.}

\item{symmetrical}{Logical: if \code{TRUE} and pairwise comparisons between \code{a} rows were made,
the results in the lower triangle are copied to the upper triangle.}

\item{mean}{Logical: if \code{TRUE}, a single mean for each metric is returned per row of \code{a}.}

\item{return.list}{Logical: if \code{TRUE}, a list-like object will always be returned, with an entry
for each metric, even when only one metric is requested.}
}
\value{
Output varies based on the dimensions of \code{a} and \code{b}:
  \itemize{
    \item \strong{Out:} A vector with a value per metric. \cr
      \strong{In:} Only when \code{a} and \code{b} are both vectors.
    \item \strong{Out:} A vector with a value per row. \cr
      \strong{In:} Any time a single value is expected per row: \code{a} or \code{b} is a vector,
      \code{a} and \code{b} are matrices with the same number of rows and \code{pairwise = FALSE}, a group is
      specified, or \code{mean = TRUE}, and only one metric is requested.
    \item \strong{Out:} A data.frame with a column per metric. \cr
      \strong{In:} When multiple metrics are requested in the previous case.
    \item \strong{Out:} A sparse matrix with a \code{metric} attribute with the metric name. \cr
      \strong{In:} Pairwise comparisons within an \code{a} matrix or between
      an \code{a} and \code{b} matrix, when only 1 metric is requested.
    \item \strong{Out:} A list with a sparse matrix per metric. \cr
      \strong{In:} When multiple metrics are requested in the previous case.
  }
}
\description{
Enter a numerical matrix, set of vectors, or set of matrices to calculate similarity per vector.
}
\details{
Use \code{\link[RcppParallel]{setThreadOptions}} to change parallelization options; e.g., run
RcppParallel::setThreadOptions(4) before a call to lma_simets to set the number of CPU
threads to 4.
}
\examples{
text = c(
  'words of speaker A', 'more words from speaker A',
  'words from speaker B', 'more words from speaker B'
)
(dtm = lma_dtm(text))

# compare each entry
lma_simets(dtm)

# compare each entry with the mean of all entries
lma_simets(dtm, colMeans(dtm))

# compare by group (corresponding to speakers and turns in this case)
speaker = c('A', 'A', 'B', 'B')

## by default, consecutive rows from the same group are averaged:
lma_simets(dtm, group = speaker)

## with agg = FALSE, only the rows at the boundary between
## groups (rows 2 and 3 in this case) are used:
lma_simets(dtm, group = speaker, agg = FALSE)
}
