% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{rpsft}
\alias{rpsft}
\title{Rank preserving structured failure time model (RPSFTM) for
treatment switching}
\usage{
rpsft(
  data,
  stratum = "stratum",
  time = "time",
  event = "event",
  treat = "treat",
  rx = "rx",
  censor_time = "censor_time",
  base_cov = "none",
  low_psi = -1,
  hi_psi = 1,
  n_eval_z = 100L,
  alpha = 0.05,
  treat_modifier = 1,
  recensor = 1L,
  autoswitch = 1L,
  gridsearch = 0L,
  boot = 0L,
  n_boot = 1000L
)
}
\arguments{
\item{data}{The input data frame that contains the following variables:
\itemize{
\item \code{stratum}: The stratum.
\item \code{time}: The survival time for right censored data.
\item \code{event}: The event indicator, 1=event, 0=no event.
\item \code{treat}: The randomized treatment indicator, 1=treatment,
0=control.
\item \code{rx}: The proportion of time on active treatment.
\item \code{censor_time}: The administrative censoring time. It should
be provided for all subjects including those who had events.
\item \code{base_cov}: The values of baseline covariates.
This is the full-rank design matrix (excluding treat)
for the Cox model, assuming that factor variables
have already been expanded into dummy variables.
}}

\item{stratum}{The name of the stratum variable in the input data.}

\item{time}{The name of the time variable in the input data.}

\item{event}{The name of the event variable in the input data.}

\item{treat}{The name of the treatment variable in the input data.}

\item{rx}{The name of the rx variable in the input data.}

\item{censor_time}{The name of the censor_time variable in the input data.}

\item{base_cov}{The vector of names of baseline covariates (excluding
treat) in the input data.}

\item{low_psi}{The lower limit of the causal parameter of RPSFTM.}

\item{hi_psi}{The upper limit of the causal parameter of RPSFTM.}

\item{n_eval_z}{The number of points between low_psi and hi_psi at which
to evaluate the log-rank Z-statistics.}

\item{alpha}{The significance level to calculate confidence intervals.}

\item{treat_modifier}{The optional sensitivity parameter for the
constant treatment effect assumption.}

\item{recensor}{Whether to apply recensoring to counter-factual
survival times. Defaults to \code{TRUE}.}

\item{autoswitch}{Whether to exclude recensoring for treatment arms
with no switching. Defaults to \code{TRUE}.}

\item{gridsearch}{Whether to use grid search to estimate the causal
parameter psi. Defaults to \code{FALSE}, in which case, a root
finding algorithm will be used.}

\item{boot}{Whether to use bootstrap to obtain the confidence
interval for hazard ratio. Defaults to \code{FALSE}, in which case,
the confidence interval will be constructed to match the log-rank
test p-value.}

\item{n_boot}{The number of bootstrap samples.}
}
\value{
A list with the following components:
\itemize{
\item \code{psi}: The estimated causal parameter for RPSFTM.
\item \code{psi_CI}: The confidence interval for psi.
\item \code{psi_type}: The type of psi estimate, either "grid search" or
"root finding".
\item \code{Sstar}: A data frame containing the counter-factual untreated
survival times and the event indicators.
\item \code{kmstar}: A data frame containing the Kaplan-Meier estimates
based on the counter-factual untreated survival times by treatment arm.
\item \code{eval_z}: A data frame containing the log-rank test Z-statistics
evaluated at a sequence of psi values. Used to plot and to check
if the range of psi values to search for the solution and
limits of confidence interval of psi need be modified.
\item \code{pvalue}: The p-value of the log-rank test based on
the treatment policy strategy.
\item \code{hr}: The estimated hazard ratio from the Cox model.
\item \code{hr_CI}: The confidence interval for hazard ratio.
\item \code{hr_CI_type}: The type of confidence interval for hazard ratio,
either "log-rank p-value" or "bootstrap quantile".
}
}
\description{
Obtains the causal parameter estimate of the RPSFTM from
the log-rank test and the hazard ratio estimate from the Cox model.
}
\details{
We use the following steps to obtain the hazard ratio estimate
and confidence interval had there been no treatment switching:
\itemize{
\item use RPSFTM to estimate the causal parameter psi based on the log-rank
test for counter-factual untreated survival times for both arms:
\eqn{U = T_{off} + T_{on} e^{\psi}}.
\item Fit the Cox proportional hazards model to the observed survival times
on the treatment arm and the counter-factual untreated survival times
on the control arm to obtain the hazard ratio estimate.
\item Use either the log-rank test p-value for the treatment policy strategy
or bootstrap to construct the confidence interval for hazard ratio.
}
}
\examples{

library(dplyr)

data <- immdef \%>\% mutate(rx = 1-xoyrs/progyrs)

fit <- rpsft(data, time = "progyrs", event = "prog", treat = "imm",
             rx = "rx", censor_time = "censyrs", boot = 0)

c(fit$hr, fit$hr_CI)

}
\author{
Kaifeng Lu, \email{kaifenglu@gmail.com}
}
