% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mable-ph-model.r
\name{maple.ph}
\alias{maple.ph}
\title{Mable fit of the PH model with given regression coefficients}
\usage{
maple.ph(
  formula,
  data,
  M,
  g,
  pi0 = NULL,
  tau = Inf,
  x0 = NULL,
  controls = mable.ctrl(),
  progress = TRUE
)
}
\arguments{
\item{formula}{regression formula. Response must be \code{cbind}.  See 'Details'.}

\item{data}{a dataset}

\item{M}{a positive integer or a vector \code{(m0, m1)}. If \code{M = m} or \code{m0 = m1 = m},
then \code{m} is a preselected degree. If \code{m0 < m1} it specifies the set of 
consective candidate model degrees \code{m0:m1} for searching an optimal degree,
where \code{m1-m0>3}.}

\item{g}{the given \eqn{d}-vector of regression coefficients}

\item{pi0}{Initial guess of \eqn{\pi(x_0) = F(\tau_n|x_0)}. Without right censored data, \code{pi0 = 1}. See 'Details'.}

\item{tau}{right endpoint of support \eqn{[0, \tau)} must be greater than or equal to the maximum observed time}

\item{x0}{a working baseline covariate. See 'Details'.}

\item{controls}{Object of class \code{mable.ctrl()} specifying iteration limit 
and other control options. Default is \code{\link{mable.ctrl}}.}

\item{progress}{if \code{TRUE} a text progressbar is displayed}
}
\value{
a class '\code{mable_reg}' object, a list with components
\itemize{ 
  \item \code{M} the vector \code{(m0, m1)}, where \code{m1} is the last candidate degree when the search stoped
  \item \code{m} the selected optimal degree \code{m}
  \item \code{p} the estimate of \eqn{p = (p_0, \dots, p_m,p_{m+1})}, the coefficients of Bernstein polynomial of degree \code{m}
  \item \code{coefficients} the given regression coefficients of the PH model
  \item \code{mloglik} the maximum log-likelihood at an optimal degree \code{m}
  \item \code{lk} log-likelihoods evaluated at \eqn{m \in \{m_0, \ldots, m_1\}}
  \item \code{lr} likelihood ratios for change-points evaluated at \eqn{m \in \{m_0+1, \ldots, m_1\}}
  \item \code{tau.n} maximum observed time \eqn{\tau_n}
  \item \code{tau} right endpoint of support \eqn{[0, \tau)}
  \item \code{x0} the working baseline covariates 
  \item \code{egx0} the value of \eqn{e^{\gamma'x_0}} 
  \item \code{convergence} an integer code. 0 indicates successful completion(the iteration is 
   convergent). 1 indicates that the maximum candidate degree had been reached in the calculation;
  \item \code{delta} the final convergence criterion for EM iteration;
  \item \code{chpts} the change-points among the candidate degrees;
  \item \code{pom} the p-value of the selected optimal degree \code{m} as a change-point;
 }
}
\description{
Maximum approximate profile likelihood estimation of Bernstein
 polynomial model in Cox's proportional hazards regression  based on interal 
 censored event time data with a given regression coefficients which are efficient
 estimates provided by other semiparametric methods. Select optimal degree with a 
 given regression coefficients.
}
\details{
Consider Cox's PH model with covariate for interval-censored failure time data: 
\eqn{S(t|x) = S(t|x_0)^{\exp(\gamma'(x-x_0))}}, where \eqn{x_0} satisfies \eqn{\gamma'(x-x_0)\ge 0}.   
  Let \eqn{f(t|x)} and \eqn{F(t|x) = 1-S(t|x)} be the density and cumulative distribution
functions of the event time given \eqn{X = x}, respectively.
Then \eqn{f(t|x_0)} on \eqn{[0,\tau_n]} can be approximated by  
\eqn{f_m(t|x_0; p) = \tau_n^{-1}\sum_{i=0}^m p_i\beta_{mi}(t/\tau_n)},
where \eqn{p_i \ge 0}, \eqn{i = 0, \ldots, m}, \eqn{\sum_{i=0}^mp_i = 1-p_{m+1}},
\eqn{\beta_{mi}(u)} is the beta denity with shapes \eqn{i+1} and \eqn{m-i+1}, and
\eqn{\tau_n} is the largest observed time, either uncensored time, or right endpoint of interval/left censored,
or left endpoint of right censored time. So we can approximate  \eqn{S(t|x_0)} on \eqn{[0, \tau_n]} by
\eqn{S_m(t|x_0; p) = \sum_{i=0}^{m+1} p_i \bar B_{mi}(t/\tau_n)}, where 
\eqn{\bar B_{mi}(u)}, \eqn{i = 0, \ldots, m}, is the beta survival function with shapes 
 \eqn{i+1} and \eqn{m-i+1}, \eqn{\bar B_{m,m+1}(t) =  1}, \eqn{p_{m+1} = 1-\pi(x_0)}, and
\eqn{\pi(x_0) = F(\tau_n|x_0)}. For data without right-censored time, \eqn{p_{m+1} = 1-\pi(x_0) = 0.}  

Response variable should be of the form \code{cbind(l, u)}, where \code{(l, u)} is the interval 
containing the event time. Data is uncensored if \code{l = u}, right censored 
if \code{u = Inf} or \code{u = NA}, and  left censored data if \code{l = 0}.
The associated covariate contains \eqn{d} columns. The baseline \code{x0} should chosen so that 
\eqn{\gamma'(x-x_0)} is nonnegative for all the observed \eqn{x}.

 The search for optimal degree \code{m} is stoped if either \code{m1} is reached or the test 
 for change-point results in a p-value \code{pval} smaller than \code{sig.level}.
}
\examples{
\donttest{
 ## Simulated Weibull data
   require(icenReg) 
   set.seed(123)
   simdata<-simIC_weib(70, inspections = 5, inspectLength = 1)
   sp<-ic_sp(cbind(l, u) ~ x1 + x2, data = simdata)
   res0<-maple.ph(cbind(l, u) ~ x1 + x2, data = simdata, M=c(2,20), 
        g=sp$coefficients, tau=7)
   op<-par(mfrow=c(1,2))
   plot(res0,  which=c("likelihood","change-point"))
   par(op)
   res1<-mable.ph(cbind(l, u) ~ x1 + x2, data = simdata, M=res0$m, 
      g=c(.5,-.5), tau=7)
   op<-par(mfrow=c(1,2))
   plot(res1, y=data.frame(c(0,0)), which="density", add=FALSE, type="l", 
       xlab="Time", main="Desnity Function")
   lines(xx<-seq(0, 7, len=512), dweibull(xx, 2,2), lty=2, col=2)
   legend("topright", bty="n", lty=1:2, col=1:2, c("Estimated","True"))
   plot(res1, y=data.frame(c(0,0)), which="survival", add=FALSE, type="l", 
       xlab="Time", main="Survival Function")
   lines(xx, 1-pweibull(xx, 2, 2), lty=2, col=2)
   legend("topright", bty="n", lty=1:2, col=1:2, c("Estimated","True"))
   par(op)
}
}
\references{
Guan, Z. (2019) Maximum Approximate Bernstein Likelihood Estimation in Proportional Hazard Model for Interval-Censored Data, 
arXiv:1906.08882 .
}
\seealso{
\code{\link{mable.ph}}
}
\author{
Zhong Guan <zguan@iusb.edu>
}
\concept{Cox proportional hazards model}
\concept{interval censoring}
\keyword{distribution}
\keyword{models}
\keyword{nonparametric}
\keyword{regression}
\keyword{smooth}
\keyword{survival}
