% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dlyEngWtrFlux.R
\name{dlyEngWtrFluxPoints}
\alias{dlyEngWtrFluxPoints}
\title{Estimator for Daily Amounts of Energy and Water Fluxes}
\usage{
dlyEngWtrFluxPoints(
  TEMP,
  PREC,
  BSDF,
  lat,
  elv,
  year = 2000,
  MSMC = 150,
  daily = TRUE,
  mlyOpVar = c("EET", "PET", "AET")
)
}
\arguments{
\item{TEMP}{'numeric' R object with one-year time series of daily mean air temperature (in °C)}

\item{PREC}{'numeric' R object with one-year time series of daily precipitation sum (in mm)}

\item{BSDF}{'numeric' R object with one-year time series of daily fractional sunshine duration (dimensionless)}

\item{lat}{'numeric' vector with the latitude coordinates (in decimal degrees)}

\item{elv}{'numeric' vector with the elevation values (in meters above sea level)}

\item{year}{'numeric' vector with values of the year (using astronomical year numbering)}

\item{MSMC}{'numeric' vector with values of the maximum soil moisture capacity (aka 'bucket size') (in mm)}

\item{daily}{'logical' scalar that indicates whether or not daily values should also be computed.}

\item{mlyOpVar}{'character' vector of at least one length that indicates the bioclimatic variable(s) for which
monthly time series are to be calculated. Valid values are as follows: \cr
(a) \code{'EET'} - monthly amounts of the equilibrium evapotranspiration (in mm); \cr
(b) \code{'PET'} - monthly amounts of the potential evapotranspiration (in mm); \cr
(c) \code{'AET'} - monthly amounts of the actual evapotranspiration (in mm); \cr
(d) \code{'PTC'} - monthly values of the Priestley–Taylor coefficient (dimensionless); \cr
(e) \code{'CWD'} - monthly values of the climatic water deficit (in mm).}
}
\value{
If daily values are also requested (\code{daily = TRUE}), the function returns a list of lists with daily
    and monthly data. If \code{daily = FALSE}, the return object is a list with the monthly values. The character
    vector \code{'mlyOpVar'} determines for which variables are integrated at monthly scale (for explanations see
    'Details'). Daily data is available for the following quantities:

    \itemize{
      \item{\code{H_0_J.m2.dy1}: daily solar irradiation (in J m-2)}
      \item{\code{H_np_J.m2.dy1}: daily positive (daytime) net surface radiation (in J m-2)}
      \item{\code{H_nn_J.m2.dy1}: daily negative (nighttime) net surface radiation (in J m-2)}
      \item{\code{PPFD_mol.m2.dy1}: daily photosynthetically active radiation (in mol m-2)}
      \item{\code{CN_mm.dy1}: daily condensation (in mm)}
      \item{\code{SM_mm.dy1}: daily soil moisture (in mm)}
      \item{\code{RO_mm.dy1}: daily runoff (in mm)}
      \item{\code{EET_mm.dy1}: daily equilibrium evapotranspiration (in mm)}
      \item{\code{PET_mm.dy1}: daily potential evapotranspiration (in mm)}
      \item{\code{AET_mm.dy1}: daily actual evapotranspiration (in mm)}
    }

    Each matrix in the list of daily data consists of 365 or 366 columns, while monthly data are available, of
    course, as 12-column matrices. The former are accessible in the list \code{'dly'}, while the latter can be
    found in the list labelled as \code{'mly'}.
}
\description{
Estimates the daily amounts of energy and water fluxes and the associated monthly bioclimatic
    variables, by using the SPLASH algorithm described by Davis et al. (2017). This version of the algorithm is
    directly suitable for paleoclimate applications because it takes into account the time variability of the
    Earth's orbital elements, and thus changes in the seasonal cycle of insolation.
}
\details{
To estimate the daily radiation, evapotranspiration and soil moisture for an equilibrium year, the SPLASH
    algorithm described by Davis et al. (2017) is implemented with two slight amendments. In accordance with Davis
    et al. (2017), daily insolation (incoming solar radiation at the top of the atmosphere) is estimated by using
    Eq 1.10.3 in Duffie and Beckman (1991), with the remark that orbital parameters of the Earth are not assumed
    to be constant. Temporal variability of orbital parameters is considered through the implementation of the
    procedure as proposed by Berger and Loutre (1991). To simulate seasonal changes in the climatic water balance,
    the simple 'bucket model' proposed by Cramer and Prentice (1988) is applied in accordance with the SPLASH
    v.1.0 model. In this model, the daily value of actual evapotranspiration is estimated as an the analytical
    integral of the minimum of the instantaneous evaporative supply and demand rates over a single day (see Eq 27
    in Davis et  al. (2017)). The SPLASH algorithm is modified in a further aspect: in the 'bucket model', the
    'bucket size' is freely changeable, i.e., it can be specified regionally. Its value is set to 150 mm by
    default, in accordance with Cramer and Prentice (1988). \cr
    The function provides daily estimates for the following key quantities: daily insolation
    (\code{'H_0_J.m2.dy1'}), daily net surface radiation (\code{'H_np_J.m2.dy1'}, and \code{'H_nn_J.m2.dy1'});
    photosynthetic photon flux density (\code{'PPFD_mol.m2.dy1'}); daily condensation, soil moisture and runoff
    (\code{'CN_mm.dy1'}, \code{'SM_mm.dy1'}, and \code{'RO_mm.dy1'}, respectively); and daily equilibrium,
    potential, and actual evapotranspiration (\code{'EET_mm.dy1'}, \code{'PET_mm.dy1'}, and \code{'AET_mm.dy1'}).
    It also integrates daily data for bioclimatic variables relevant to ecoclimatological studies at a monthly
    timescale: monthly equilibrium, potential and actual evapotranspiration (\code{'EET_mo_mm.mo1'},
    \code{'PET _mo_mm.mo1'}, and \code{'AET_mo_mm.mo1'}), monthly Priestley–Taylor coefficient (\code{'PTC_mo'}),
    monthly climatic water deficit (\code{'CWD_mo_mm.mo1'}).
}
\note{
As with any function with a point mode, a set of basic input data is defined here. In this case, they are as
   follows: \code{'TEMP'} (one-year time series of daily mean air temperature), \code{'PREC'} (one-year time series
   of daily precipitation sum), and \code{'BSDF'} (one-year time series of daily mean relative sunshine
   duration). The objects \code{'TEMP'}, \code{'PREC'} and \code{'BSDF'} must be either vectors of length 365 (or
   366) or 365-column (or 366-column) matrices. The first dimensions of these matrices have to be the same length.
   The function automatically converts vectors into single-row matrices during the error handling, and then uses
   these matrices. The first dimensions of these matrices determines the number of rows in the result matrices. In
   the case of arguments that do not affect the course of the calculation procedure or the structure of the
   return object, scalar values (i.e., 'numeric' vector of length 1) may also be allowed. In this case, they are as
   follows: \code{'lat'} (latitude coordinates in decimal degrees), \code{'elv'} (elevation in meters above sea
   level), \code{'year'} (year using astronomical year numbering), and \code{'MSMC'} ('bucket size' in mm). These
   scalars are converted to vectors by the function during the error handling, and these vectors are applied in
   the further calculations. If these data are stored in vectors of length at least 2, their length must be the
   same size of first dimension of the matrices containing the basic data.
}
\examples{
\donttest{
library(graphics)

# Loading mandatory data for the Example 'Points'
data(inp_exPoints)

with(inp_exPoints, {
# Estimates the daily amounts of energy and water fluxes with default settings,
# at a grid cell near Szeged, Hungary (46.3N, 20.2E) (for the normal period 1981-2010)
year <- trunc(mean(seq(1981, 2010)))
wea <- dlyWeaGenPoints(colMeans(temp), colMeans(prec), colMeans(bsdf), year = year)
ewf <- dlyEngWtrFluxPoints(wea$TEMP, wea$PREC, wea$BSDF, lat, lon, elv, year = year)

# Check daily energy and water fluxes
opar <- par(no.readonly = TRUE)
par(mfrow = c(4, 1))
var <- list(t(ewf$dly$H_np_J.m2.dy1) * 1e-6, t(ewf$dly$SM_mm.dy1), t(wea$PREC))
lbl <- list(expression(italic(H[N])~(MJ~m^{-2})), expression(italic(SM[n])~(mm)),
    expression(italic(P[n])~(mm)))
at <- list(seq(0, 16, 4), seq(0, 80, 20), seq(0, 4))
txt <- list("(a)", "(b)", "(c)")
for (i in 1 : length(var)) {
  par(mar = c(1, 5, 1, 1))
  plot(var[[i]], type = "l", lwd = 2, xlab = NA, ylab = NA, axes = FALSE)
  axis(side = 1, las = 1, tck = -0.03, labels = NA, at = seq(-60, 720, 30))
  axis(side = 2, las = 1, tck = -0.03, labels = NA, at = at[[i]])
  axis(side = 2, las = 1, lwd = 0, line = -0.4, cex.axis = 1.6, at = at[[i]])
  mtext(side = 2, lbl[[i]], line = 3, cex = 1.1)
  text(-12, max(at[[i]]) / 4, txt[[i]], pos = 4, cex = 1.7)
}
par(mar = c(2, 5, 1, 1))
plot(t(ewf$dly$PET_mm.dy1), type = "l", lwd = 2, xlab = NA, ylab = NA, axes = FALSE,
  ylim = c(0, max(t(ewf$dly$PET_mm.dy1))))
lines(t(ewf$dly$AET_mm.dy1), lty = 2, lwd = 2, col = "green")
axis(side = 1, las = 1, tck = -0.03, labels = NA, at = seq(-60, 720, 30))
axis(side = 1, las = 1, lwd = 0, line = -0.4, at = seq(-60, 720, 30), cex.axis = 1.6)
axis(side = 2, las = 1, tck = -0.03, labels = NA, at = seq(-1, 6, 1))
axis(side = 2, las = 1, lwd = 0, line = -0.4, cex.axis = 1.6, at = seq(-1, 6, 1))
legend("topright", legend = c(expression(italic(E[n]^{q})), expression(italic(E[n]^{a}))),
    col = c("black", "green"), lty = c(1, 2), cex = 1.6, inset = 0.02,
    adj = c(0.5, 0.5), lwd = c(2, 2), horiz = TRUE, bty = "n", seg.len = 1)
mtext(side = 2, expression(italic(E[n])~(mm)), line = 3, cex = 1.1)
text(-12, 1.5, "(d)", pos = 4, cex = 1.7)
par(opar)

# Check monthly water balance quantities
plot(t(ewf$mly$PET_mo_mm.mo1), type = "l", lwd = 2, ylim = c(0, 1.1 * max(ewf$mly$PET_mo_mm.mo1)),
    xlab = NA, ylab = NA, axes = FALSE)
lines(t(ewf$mly$EET_mo_mm.mo1), lty = 1, lwd = 2, col = "green")
lines(t(ewf$mly$AET_mo_mm.mo1), lty = 2, lwd = 2, col = "blue")
box(lwd = 2)
axis(side = 1, las = 1, tck = -0.02, labels = NA, at = seq(1, 12))
axis(side = 1, las = 1, lwd = 0, line = -0.4, labels = month.abb, at = seq(1, 12), cex.axis = 1.2)
axis(side = 2, las = 1, tck = -0.02, labels = NA, at = seq(-20, 200, 20))
axis(side = 2, las = 1, lwd = 0, line = -0.4, at = seq(-20, 200, 20), cex.axis = 1.2)
mtext(side = 2, expression(list(Evapotranspiration, mm~month^{-1})), line = 2, cex = 1.2)
legend("top", legend = c("Potential", "Equilibrium", "Actual"), col = c("black", "green", "blue"),
    lty = c(1, 1, 2), lwd = c(2, 2, 2), inset = 0.01, x.intersp = 1.1, y.intersp = 2.0,
    horiz = TRUE, bty = "n", cex = 1.2)
})
}

}
\references{
\cite{Berger A, Loutre MF (1991) Insolation values for the climate of the last 10 million years. Quat Sci Rev
    10(4):297-317. \doi{10.1016/0277-3791(91)90033-Q}}

\cite{Cramer W, Prentice IC (1988) Simulation of regional soil moisture deficits on a European scale. Nor J Geogr
    42(2-3):149–151. \doi{10.1080/00291958808552193}}

\cite{Davis TW, Prentice IC, Stocker BD, Thomas RT, Whitley RJ, Wang H, Evans BJ, Gallego-Sala AV, Sykes MT,
    Cramer W (2017) Simple process-led algorithms for simulating habitats (SPLASH v.1.0): robust indices of
    radiation, evapotranspiration and plant-available moisture. Geosci Model Dev 10(2):689–708.
    \doi{10.5194/gmd-10-689-2017}}

\cite{Duffie JA, Beckman WA (1991) Solar Engineering of Thermal Processes. Second Edition. Wiley-Interscience,
    New York, NY}
}
