% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictions.R
\name{predictions}
\alias{predictions}
\title{Adjusted Predictions}
\usage{
predictions(
  model,
  newdata = NULL,
  variables = NULL,
  vcov = TRUE,
  conf_level = 0.95,
  type = NULL,
  by = NULL,
  byfun = NULL,
  wts = NULL,
  transform_post = NULL,
  hypothesis = NULL,
  ...
)
}
\arguments{
\item{model}{Model object}

\item{newdata}{\code{NULL}, data frame, string, or \code{datagrid()} call. Determines the grid of predictors on which we make predictions.
\itemize{
\item \code{NULL} (default): Predictions for each observed value in the original dataset.
\item data frame: Predictions for each row of the \code{newdata} data frame.
\item string:
\itemize{
\item "mean": Predictions at the Mean. Predictions when each predictor is held at its mean or mode.
\item "median": Predictions at the Median. Predictions when each predictor is held at its median or mode.
\item "marginalmeans": Predictions at Marginal Means. See Details section below.
\item "tukey": Predictions at Tukey's 5 numbers.
\item "grid": Predictions on a grid of representative numbers (Tukey's 5 numbers and unique values of categorical predictors).
}
\item \code{\link[=datagrid]{datagrid()}} call to specify a custom grid of regressors. For example:
\itemize{
\item \code{newdata = datagrid(cyl = c(4, 6))}: \code{cyl} variable equal to 4 and 6 and other regressors fixed at their means or modes.
\item See the Examples section and the \code{\link[=datagrid]{datagrid()}} documentation.
}
}}

\item{variables}{Named list of variables with values to create a
counterfactual grid of predictions. The entire dataset replicated
for each unique combination of the variables in this list. See the Examples
section below. Warning: This can use a lot of memory if there are many
variables and values, and when the dataset is large.}

\item{vcov}{Type of uncertainty estimates to report (e.g., for robust standard errors). Acceptable values:
\itemize{
\item FALSE: Do not compute standard errors. This can speed up computation considerably.
\item TRUE: Unit-level standard errors using the default \code{vcov(model)} variance-covariance matrix.
\item String which indicates the kind of uncertainty estimates to return.
\itemize{
\item Heteroskedasticity-consistent: \code{"HC"}, \code{"HC0"}, \code{"HC1"}, \code{"HC2"}, \code{"HC3"}, \code{"HC4"}, \code{"HC4m"}, \code{"HC5"}. See \code{?sandwich::vcovHC}
\item Heteroskedasticity and autocorrelation consistent: \code{"HAC"}
\item Mixed-Models degrees of freedom: "satterthwaite", "kenward-roger"
\item Other: \code{"NeweyWest"}, \code{"KernHAC"}, \code{"OPG"}. See the \code{sandwich} package documentation.
}
\item One-sided formula which indicates the name of cluster variables (e.g., \code{~unit_id}). This formula is passed to the \code{cluster} argument of the \code{sandwich::vcovCL} function.
\item Square covariance matrix
\item Function which returns a covariance matrix (e.g., \code{stats::vcov(model)})
}}

\item{conf_level}{numeric value between 0 and 1. Confidence level to use to build a confidence interval.}

\item{type}{string indicates the type (scale) of the predictions used to
compute marginal effects or contrasts. This can differ based on the model
type, but will typically be a string such as: "response", "link", "probs",
or "zero". When an unsupported string is entered, the model-specific list of
acceptable values is returned in an error message. When \code{type} is \code{NULL}, the
default value is used. This default is the first model-related row in
the \code{marginaleffects:::type_dictionary} dataframe.}

\item{by}{Character vector of variable names over which to compute group-wise estimates.}

\item{byfun}{A function such as \code{mean()} or \code{sum()} used to aggregate
estimates within the subgroups defined by the \code{by} argument. \code{NULL} uses the
\code{mean()} function. Must accept a numeric vector and return a single numeric
value. This is sometimes used to take the sum or mean of predicted
probabilities across outcome or predictor
levels. See examples section.}

\item{wts}{string or numeric: weights to use when computing average
contrasts or marginaleffects. These weights only affect the averaging in
\code{tidy()} or \code{summary()}, and not the unit-level estimates themselves.
\itemize{
\item string: column name of the weights variable in \code{newdata}. When supplying a column name to \code{wts}, it is recommended to supply the original data (including the weights variable) explicitly to \code{newdata}.
\item numeric: vector of length equal to the number of rows in the original data or in \code{newdata} (if supplied).
}}

\item{transform_post}{(experimental) A function applied to unit-level adjusted predictions and confidence intervals just before the function returns results. For bayesian models, this function is applied to individual draws from the posterior distribution, before computing summaries.}

\item{hypothesis}{specify a hypothesis test or custom contrast using a vector, matrix, string, or string formula.
\itemize{
\item String:
\itemize{
\item "pairwise": pairwise differences between estimates in each row.
\item "reference": differences between the estimates in each row and the estimate in the first row.
\item "sequential": difference between an estimate and the estimate in the next row.
\item "revpairwise", "revreference", "revsequential": inverse of the corresponding hypotheses, as described above.
}
\item String formula to specify linear or non-linear hypothesis tests. If the \code{term} column uniquely identifies rows, terms can be used in the formula. Otherwise, use \code{b1}, \code{b2}, etc. to identify the position of each parameter. Examples:
\itemize{
\item \code{hp = drat}
\item \code{hp + drat = 12}
\item \code{b1 + b2 + b3 = 0}
}
\item Numeric vector: Weights to compute a linear combination of (custom contrast between) estimates. Length equal to the number of rows generated by the same function call, but without the \code{hypothesis} argument.
\item Numeric matrix: Each column is a vector of weights, as describe above, used to compute a distinct linear combination of (contrast between) estimates. The column names of the matrix are used as labels in the output.
\item See the Examples section below and the vignette: https://vincentarelbundock.github.io/marginaleffects/articles/hypothesis.html
}}

\item{...}{Additional arguments are passed to the \code{predict()} method
supplied by the modeling package.These arguments are particularly useful
for mixed-effects or bayesian models (see the online vignettes on the
\code{marginaleffects} website). Available arguments can vary from model to
model, depending on the range of supported arguments by each modeling
package. See the "Model-Specific Arguments" section of the
\code{?marginaleffects} documentation for a non-exhaustive list of available
arguments.}
}
\value{
A \code{data.frame} with one row per observation and several columns:
\itemize{
\item \code{rowid}: row number of the \code{newdata} data frame
\item \code{type}: prediction type, as defined by the \code{type} argument
\item \code{group}: (optional) value of the grouped outcome (e.g., categorical outcome models)
\item \code{predicted}: predicted outcome
\item \code{std.error}: standard errors computed by the \code{insight::get_predicted} function or, if unavailable, via \code{marginaleffects} delta method functionality.
\item \code{conf.low}: lower bound of the confidence interval (or equal-tailed interval for bayesian models)
\item \code{conf.high}: upper bound of the confidence interval (or equal-tailed interval for bayesian models)
}
}
\description{
Outcome predicted by a fitted model on a specified scale for a given
combination of values of the predictor variables, such as their observed
values, their means, or factor levels (a.k.a. "reference grid"). The
\code{tidy()} and \code{summary()} functions can be used to aggregate the output of
\code{predictions()}. To learn more, read the predictions vignette, visit the
package website, or scroll down this page for a full list of vignettes:
\itemize{
\item \url{https://vincentarelbundock.github.io/marginaleffects/articles/predictions.html}
\item \url{https://vincentarelbundock.github.io/marginaleffects/}
}
}
\details{
The \code{newdata} argument, the \code{tidy()} function, and \code{datagrid()} function can be used to control the kind of predictions to report:
\itemize{
\item Average Predictions
\item Predictions at the Mean
\item Predictions at User-Specified values (aka Predictions at Representative values).
}

When possible, \code{predictions()} delegates the computation of confidence
intervals to the \code{insight::get_predicted()} function, which uses back
transformation to produce adequate confidence intervals on the scale
specified by the \code{type} argument. When this is not possible, \code{predictions()}
uses the Delta Method to compute standard errors around adjusted
predictions, and builds symmetric confidence intervals. These naive symmetric
intervals may not always be appropriate. For instance, they may stretch beyond
the bounds of a binary response variables.
}
\section{Vignettes and documentation}{


Vignettes:
\itemize{
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/predictions.html}{Adjusted Predictions}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/contrasts.html}{Contrasts}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/marginaleffects.html}{Marginal Effects}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/marginalmeans.html}{Marginal Means}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/hypothesis.html}{Hypothesis Tests and Custom Contrasts using the Delta Method}
}

Case studies:
\itemize{
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/brms.html}{Bayesian Analyses with \code{brms}}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/gformula.html}{Causal Inference with the g-Formula}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/elasticity.html}{Elasticity}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/gam.html}{Generalized Additive Models}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/lme4.html}{Mixed effects models}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/mlogit.html}{Multinomial Logit and Discrete Choice Models}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/multiple_imputation.html}{Multiple Imputation}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/python.html}{Python NumPyro models in \code{marginaleffects}}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/logistic_contrasts.html}{Unit-level contrasts in logistic regressions}
}

Tips and technical notes:
\itemize{
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/supported_models.html}{69 Supported Classes of Models}
\item \href{https://vincentarelbundock.github.io/marginaleffects/reference/index.html}{Index of Functions and Documentation}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/sandwich.html}{Standard Errors and Confidence Intervals}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/modelsummary.html}{Tables and Plots}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/performance.html}{Performance}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/alternative_software.html}{Alternative Software}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/faq.html}{Frequently Asked Questions}
}
}

\section{Model-Specific Arguments}{


Some model types allow model-specific arguments to modify the nature of
marginal effects, predictions, marginal means, and contrasts.\tabular{llll}{
   Package \tab Class \tab Argument \tab Documentation \cr
   \code{brms} \tab \code{brmsfit} \tab \code{ndraws} \tab \link[brms:posterior_predict.brmsfit]{brms::posterior_predict} \cr
    \tab  \tab \code{re_formula} \tab  \cr
   \code{lme4} \tab \code{merMod} \tab \code{include_random} \tab \link[insight:get_predicted]{insight::get_predicted} \cr
    \tab  \tab \code{re.form} \tab \link[lme4:predict.merMod]{lme4::predict.merMod} \cr
    \tab  \tab \code{allow.new.levels} \tab \link[lme4:predict.merMod]{lme4::predict.merMod} \cr
   \code{glmmTMB} \tab \code{glmmTMB} \tab \code{re.form} \tab \link[glmmTMB:predict.glmmTMB]{glmmTMB::predict.glmmTMB} \cr
    \tab  \tab \code{allow.new.levels} \tab \link[glmmTMB:predict.glmmTMB]{glmmTMB::predict.glmmTMB} \cr
    \tab  \tab \code{zitype} \tab \link[glmmTMB:predict.glmmTMB]{glmmTMB::predict.glmmTMB} \cr
   \code{mgcv} \tab \code{bam} \tab \code{exclude} \tab \link[mgcv:predict.bam]{mgcv::predict.bam} \cr
   \code{robustlmm} \tab \code{rlmerMod} \tab \code{re.form} \tab \link[robustlmm:rlmerMod-class]{robustlmm::predict.rlmerMod} \cr
    \tab  \tab \code{allow.new.levels} \tab \link[robustlmm:rlmerMod-class]{robustlmm::predict.rlmerMod} \cr
}
}

\examples{
# Adjusted Prediction for every row of the original dataset
mod <- lm(mpg ~ hp + factor(cyl), data = mtcars)
pred <- predictions(mod)
head(pred)

# Adjusted Predictions at User-Specified Values of the Regressors
predictions(mod, newdata = datagrid(hp = c(100, 120), cyl = 4))

# Average Adjusted Predictions (AAP)
library(dplyr)
mod <- lm(mpg ~ hp * am * vs, mtcars)

pred <- predictions(mod)
summary(pred)

predictions(mod, by = "am")

# Conditional Adjusted Predictions
plot_cap(mod, condition = "hp")

# Counterfactual predictions with the `variables` argument
# the `mtcars` dataset has 32 rows

mod <- lm(mpg ~ hp + am, data = mtcars)
p <- predictions(mod)
head(p)
nrow(p)

# counterfactual predictions obtained by replicating the entire for different
# values of the predictors
p <- predictions(mod, variables = list(hp = c(90, 110)))
nrow(p)


# hypothesis test: is the prediction in the 1st row equal to the prediction in the 2nd row
mod <- lm(mpg ~ wt + drat, data = mtcars)

predictions(
    mod,
    newdata = datagrid(wt = 2:3),
    hypothesis = "b1 = b2")

# same hypothesis test using row indices
predictions(
    mod,
    newdata = datagrid(wt = 2:3),
    hypothesis = "b1 - b2 = 0")

# same hypothesis test using numeric vector of weights
predictions(
    mod,
    newdata = datagrid(wt = 2:3),
    hypothesis = c(1, -1))

# two custom contrasts using a matrix of weights
lc <- matrix(c(
    1, -1,
    2, 3),
    ncol = 2)
predictions(
    mod,
    newdata = datagrid(wt = 2:3),
    hypothesis = lc)


# `by` argument
mod <- lm(mpg ~ hp * am * vs, data = mtcars)
predictions(mod, by = c("am", "vs")) 

library(nnet)
nom <- multinom(factor(gear) ~ mpg + am * vs, data = mtcars, trace = FALSE)

# first 5 raw predictions
predictions(nom, type = "probs") |> head()

# average predictions
predictions(nom, type = "probs", by = "group") |> summary()

by <- data.frame(
    group = c("3", "4", "5"),
    by = c("3,4", "3,4", "5"))

predictions(nom, type = "probs", by = by)

# sum of predicted probabilities for combined response levels
mod <- multinom(factor(cyl) ~ mpg + am, data = mtcars, trace = FALSE)
by <- data.frame(
    by = c("4,6", "4,6", "8"),
    group = as.character(c(4, 6, 8)))
predictions(mod, newdata = "mean", byfun = sum, by = by)

}
