\name{multiModel}
\alias{multiModel}
\title{Test multiple models of character transition at a single node}
\description{
  Compares up to five different models of character evolution: two models of no transition (Brownian motion and a single-
  optimum Ornstein-Uhlenbeck model); one model allowing transition at a single node in a whole-tree context; and two models 
  that split the tree into two subtrees and treat character evolution separately in the two trees (the \sQuote{censored} approach -- see
  discussion below).
}
\usage{
multiModel(phy, dat, node, models = c("whole.brown", "whole.ou1", "whole.ou2", "part.brown", "part.ou"))
}
\arguments{
  \item{phy}{ An \pkg{ape}-style tree }
  \item{dat}{ A vector of data, with names corresponding to tips in \code{phy} }
  \item{node}{ A single node, defined by the tips that are descendent from it }
  \item{models}{ The vector of models, where \dQuote{whole} indicates a model in which the tree is treated as a single entity, and 
                  \dQuote{part} indicates a model in which the tree is subdivided and treated as two entities }
}
\details{
  This function is useful after you have identified nodes at which significant character transitions are likely to 
  have occurred. Its use is to identify whether shifts at a given node are compatible with a single Ornstein-Uhlenbeck
  (O-U) model with a change in equilibrium value at the node; a two-tree model, with each tree evolving under a 
  separate Brownian motion or O-U model; or a no-change model, either Brownian motion or no-change O-U model. 
  Details and discussion of biological interpretation of these models is in Hipp 2007.
}
\value{
  A list with two items:
  \item{IC}{A matrix of information criterion statistics, generated by \code{informationCriterion}}
  \item{modelMatrix}{A matrix of model parameters for the whole-tree models, the partial-trees models, and summed-partial-trees models}
}
\references{  
  Hipp, A.L. (2007) 
  Non-uniform processes of chromosome evolution in sedges (Carex: Cyperaceae). 
  \emph{Evolution} 61:2175--2194.

  O'Meara, B. C., C. Ane, M. J. Sanderson, and P. C. Wainwright (2006) 
  Testing for different rates of continuous trait evolution using likelihood. 
  \emph{Evolution} 60:922--933.
  
  O'Meara, B. C. (2009)
  \emph{brownie} v 1.0 (MATLAB) and v 2.0 (stand-alone). \url{http://www.brianomeara.info/brownie/}
}
\examples{
  library(maticce)
  data(carex)
  attach(carex)
  mm2 <- multiModel(carex$ovales.tree, ovales.data, ovales.nodes[[2]]) # compares five different models of character change at node 2
  mm2
  layout(matrix(1:4,2,2))
  pie(mm2$IC$AICwi, labels = mm2$IC$name, col = rainbow(length(mm2$IC$name)), main = "AIC weights")
  pie(mm2$IC$AICcwi, labels = mm2$IC$name, col = rainbow(length(mm2$IC$name)), main = "AICc weights")
  pie(mm2$IC$BICwi, labels = mm2$IC$name, col = rainbow(length(mm2$IC$name)), main = "BIC weights")
  noChangeAICwi <- sum(mm2$IC$AICwi[1:2])
  changeAICwi <- sum(mm2$IC$AICwi[3:5])
  barplot(c(noChangeAICwi, changeAICwi), ylim = c(0,1), names.arg = c('no transition', 'transition'), 
    main = 'Transition vs. no-transition')
}
\author{Andrew L. Hipp \email{ahipp@mortonarb.org}}
\note{
  In \pkg{ouch} version 2.x, the \code{ouchtree} function rescales trees to depth = 1. The alpha and sigma parameters
  are thus not directly comparable between subtrees, as they were in previous versions of \pkg{ouch}, and this function as currently implemented
  should not be used to test for shifts in evolutionary rate within a tree (cf. O'Meara et al. 2006). Such a test can be performed
  in \pkg{geiger}, an \pkg{R} package, or \acronym{brownie}, which is available as a stand-alone application or a set of \acronym{MATLAB} modules.
}
\seealso{ \code{\link{carex}} for an example}