#' Meta-cluster calculations
#'
#' Generate matrix of pairwise cluster-solution similarities by Adjusted Rand
#'  index calculations.
#'
#' @param solutions_matrix solutions_matrix containing cluster solutions to
#' calculate pairwise ARIs for.
#'
#' @param processes Specify number of processes used to complete calculations
#'  * `1` (default) Sequential processing
#'  * `2` or higher: Parallel processing will use the
#'    `future.apply::future_apply` to distribute the calculations across
#'    the specified number of CPU cores. If higher than the number of
#'    available cores, a warning will be printed and the maximum number of
#'    cores will be used.
#'  * `max`: All available cores will be used.
#' Note that no progress indicator is available during multi-core processing.
#'
#' @param verbose If TRUE, print progress to console.
#'
#' @return om_aris ARIs between clustering solutions of an solutions matrix
#'
#' @export
calc_aris <- function(solutions_matrix,
                      processes = 1,
                      verbose = FALSE) {
    ###########################################################################
    # Prepare dataframe containing 1 cluster solution per row
    ###########################################################################
    # Row id and subjectkey columns
    subjects <- subs(solutions_matrix)
    # Only subject label cols
    subjects_no_id <- as.matrix(subjects[, 2:length(subjects)])
    # The skeleton of the inter-cluster similarity matrix
    aris <- matrix(1, nrow(subjects), nrow(subjects))
    ###########################################################################
    # Indices of all pairwise comparisons to calculate ARIs for
    ###########################################################################
    pairwise_indices <- utils::combn(nrow(aris), 2)
    ###########################################################################
    # Run calculations (sequentially or in parallel)
    ###########################################################################
    if (processes == 1) {
        for (col in seq_len(ncol(pairwise_indices))) {
            if (verbose) {
                if (col %% 100 == 0) {
                    progress <- 100 * col / ncol(pairwise_indices)
                    print(paste0(progress, "% completed..."))
                }
            }
            v1 <- pairwise_indices[1, col]
            v2 <- pairwise_indices[2, col]
            ari <- mclust::adjustedRandIndex(
                subjects_no_id[v1, ],
                subjects_no_id[v2, ]
            )
            aris[v1, v2] <- ari
            aris[v2, v1] <- ari
        }
        colnames(aris) <- solutions_matrix$"row_id"
        rownames(aris) <- solutions_matrix$"row_id"
        if (verbose) {
            print("100% completed.")
        }
        return(aris)
    } else {
        max_cores <- future::availableCores()
        if (processes == "max") {
            processes <- max_cores
        } else if (processes > max_cores) {
            warning(
                paste0(
                    "Requested processes exceed available cores.",
                    " Defaulting to the max available (", max_cores, ")."
                )
            )
            processes <- max_cores
        }
        # Parallelized ARI calculations
        future::plan(future::multisession, workers = processes)
        ari_vector <- future.apply::future_apply(
            pairwise_indices,
            MARGIN = 2,
            FUN = function(col) {
                mclust::adjustedRandIndex(
                    subjects_no_id[col[1], ],
                    subjects_no_id[col[2], ]
                )
            }
        )
        future::plan(future::sequential)
        #######################################################################
        # Formatting of results to symmetric matrix
        #######################################################################
        aris[lower.tri(aris, diag = FALSE)] <- ari_vector
        aris <- t(aris)
        aris[lower.tri(aris)] <- t(aris)[lower.tri(aris)]
        colnames(aris) <- solutions_matrix$"row_id"
        rownames(aris) <- solutions_matrix$"row_id"
        return(aris)
    }
}

#' Extract representative solutions from a matrix of ARIs
#'
#' Following clustering with `batch_snf`, a matrix of pairwise ARIs that show
#' how related each cluster solution is to each other can be generated by the
#' `calc_aris` function. Partitioning of the ARI matrix can be done by
#' visual inspection of `adjusted_rand_index_heatmap()` or by
#' `shiny_annotator`. Given the indices of meta cluster boundaries, this
#' function will return a single representative solution from each meta cluster
#' based on maximum average ARI to all other solutions within that meta
#' cluster.
#'
#' @param aris Matrix of adjusted rand indices from `calc_aris()`
#'
#' @param split_vector A vector of partition indices.
#'
#' @param order Numeric vector indicating row ordering of settings matrix.
#'
#' @param solutions_matrix Output of `batch_snf` containing cluster solutions.
#'
#' @param filter_fn Optional function to filter the meta-cluster by prior to
#' maximum average ARI determination. This can be useful if you are explicitly
#' trying to select a solution that meets a certain condition, such as only
#' picking from the 4 cluster solutions within a meta cluster. An example
#' valid function could be `fn <- function(x) x[x$"nclust" == 4, ]`.
#'
#' @return A "data.frame" class object corresponding to a subset of the
#' provided solutions matrix in which only one row is present per meta cluster.
#'
#' @export
get_representative_solutions <- function(aris,
                                         split_vector,
                                         order,
                                         solutions_matrix,
                                         filter_fn = NULL) {
    ###########################################################################
    # Re-sort the solutions matrix based on the aris
    ###########################################################################
    order <- unlist(order)
    aris <- data.frame(aris[order, order])
    solutions_matrix <- solutions_matrix[order, ]
    ###########################################################################
    # Extract and assign meta cluster labels
    ###########################################################################
    mc_labels <- label_splits(split_vector, nrow(solutions_matrix))
    mcs <- unique(mc_labels)
    solutions_matrix$"label" <- mc_labels
    aris$"label" <- mc_labels
    ###########################################################################
    # Iterate through the meta clusters and keep the representative solution
    ###########################################################################
    rep_solutions <- data.frame()
    for (mc in mcs) {
        # Subset to just those solutions and ARIs within the MC
        mc_sm <- solutions_matrix[solutions_matrix$"label" == mc, ]
        mc_ari <- aris[aris$"label" == mc, ]
        mc_ari$"label" <- NULL
        # The most representative solution based on total ARI within MC
        mc_sm$"total_aris" <- rowSums(mc_ari)
        if (!is.null(filter_fn)) {
            mc_sm <- filter_fn(mc_sm)
        }
        rep_mc <- which(mc_sm$"total_aris" == max(mc_sm$"total_aris"))[1]
        rep_solution <- mc_sm[rep_mc, ]
        rep_solution$"total_aris" <- NULL
        rep_solutions <- rbind(rep_solutions, rep_solution)
    }
    ###########################################################################
    # Assign labels to the representative solutions
    ###########################################################################
    return(rep_solutions)
}
