#' Prints the summary of results.
#'
#' Prints the summary of the results of \code{\link{summary.miclust}}.
#'
#' @param x object of class \code{summary.miclust} obtained with the method \code{\link{summary.miclust}}.
#' @param digits digits for the print method. Default is 2.
#' @param \dots further arguments for the print method.
#' @return a print of the summary of the results generated by \code{\link{summary.miclust}}.
#' @seealso \code{\link{miclust}}, \code{\link{summary.miclust}}.
#' @importFrom stats median
#' @export
print.summary.miclust <- function(x, digits = 2, ...) {
  ### control "x"
  if (!inherits(x, "summary.miclust"))
    stop("Argument 'x' must be of class 'summary.miclust'.")

  nusedimp <- length(x$usedimp)

  ####################
  if (nusedimp == 0)
   {
    cat("\n")
    cat("   Results for complete cases (",
        x$completecasesperc,
        "% of cases) and for ",
        x$k,
        " clusters:\n", sep = "")
    cat("---------------------------------------------------------------------\n")
    cat("Selected variables:\n")
    print(x$selectedvariables)
    cat("\n")
    cat("Cluster vector:\n")
    print(x$cluster)
    cat("\n")
    cat("Within-cluster summary:\n", sep = "")
    print(x$summarybycluster, digits = digits)
  }

  ####################

  if (nusedimp > 0) {
    cat("\n")
    cat("Results using:\n")
    cat("   ", nusedimp, "imputed data sets for the cluster analysis\n")
    cat("   ", x$m, "imputed data sets for the descriptive summary\n")
    cat("   ", x$k, "as the final number of clusters\n")
    cat("-----------------------------------------------------------\n")
    cat("\n")
    if (x$search != "none") {
      cat("Presence of the variables in the subset of selected variables:\n")
      print(x$selectedvarspresence)
      cat("\n")
      cat("Selected variables:\n")
      print(x$selectedvariables)
      cat("\n")
     }
    cat("Cohen's kappa between-imputations distribution (",
        length(x$kappas),
        " comparisons):\n",
        sep = "")
    print(x$kappadistribution, digits = digits)
    cat("\n")
    cat("Between-imputation clusters size distribution (",
        1 + length(x$kappas),
        " imputations):\n",
        sep = "")
    print(x$clusterssize, digits = digits)
    cat("\n")
    cat("Probability of assignment to the cluster distribution (",
        1 + length(x$kappas),
        " imputations):\n",
        sep = "")
    print(x$allocationprobabilities, digits = digits)
    cat("\n")
    cat("Within-cluster summary (",
        1 + length(x$kappas),
        " imputations):\n", sep = "")
    print(x$summarybycluster, digits = digits)
  }
}
