% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/findNextCATItem.R
\name{findNextItem}
\alias{findNextItem}
\title{Find next CAT item}
\usage{
findNextItem(x, person = NULL, test = NULL, design = NULL,
  criteria = NULL, objective = NULL, subset = NULL, all_index = FALSE,
  ...)
}
\arguments{
\item{x}{an object of class 'mirtCAT_design' returned from the \code{\link{mirtCAT}} function
when passing \code{design_elements = TRUE}}

\item{person}{(required when \code{x} is missing) internal person object. To be 
used when \code{customNextItem} function has been defined}

\item{test}{(required when \code{x} is missing) internal test object. To be 
used when \code{customNextItem} function has been defined}

\item{design}{(required when \code{x} is missing) internal design object. To be 
used when \code{customNextItem} function has been defined}

\item{criteria}{item selection criteria (see \code{\link{mirtCAT}}'s \code{criteria} input). 
If not specified the value from \code{extract.mirtCAT(design, 'criteria')} will be used}

\item{objective}{a vector of values used as the optimization criteria to be passed to 
\code{lp(objective.in)}. This is typically the vector of criteria values returned from
\code{\link{computeCriteria}}, however supplying other
criteria are possible (e.g., to minimize the number of items administered simply pass a vector
of -1's)}

\item{subset}{an integer vector indicating which items should be included in the optimal search;
the default \code{NULL} includes all possible items. To allow only the first 10 items to be 
selected from this can be modified to \code{subset = 1:10}. This is useful when administering 
a multi-unidimensional CAT session where unidimensional blocks should be clustered together 
for smoother presentation. Useful when using the \code{customNextItem} function in 
\code{\link{mirtCAT}}}

\item{all_index}{logical; return all items instead of just the most optimal? 
When \code{TRUE} a vector of items is returned instead of the most optimal, 
where the items are sorted according to how
well they fit the criteria (e.g., the first element is the most optimal, followed by the second
most optimal, and so on). Note that this does not work for some selection criteria (e.g.,
'seq' or 'random')}

\item{...}{additional arguments to be passed to \code{\link{lp}}}
}
\value{
typically returns an integer value indicating the index of the next item to be selected or a
  value of \code{NA} to indicate that the test should be terminated. However, see the arguments for 
  further returned object descriptions
}
\description{
A function that returns the next item in the computerized adaptive, optimal assembly, or shadow test. 
This should be used in conjunction with the \code{\link{updateDesign}} function and \code{customNextItem}. 
The raw input forms can be used when a \code{customNextItem} function has been 
defined in \code{\link{mirtCAT}}.
}
\details{
When a numeric \code{objective} is supplied the next item in the computerized adaptive test is found via 
an integer solver through searching for a maximum. The raw input forms can be used
when a \code{customNextItem} function has been defined in \code{\link{mirtCAT}}, and requires 
the definition of a \code{constr_fun} (see the associated element in \code{\link{mirtCAT}} for details,
as well as the examples below). Can be used to for 'Optimal Test Assembly', 
as well as 'Shadow Testing' designs (van der Linden, 2005),
by using the \code{\link{lp}} function. When \code{objective} is not supplied the result follows the 
typical maximum criteria of more standard adaptive tests.
}
\examples{
\dontrun{
# test defined in mirtCAT help file, first example
CATdesign <- mirtCAT(df, mod, criteria = 'MI', design_elements = TRUE)

# returns number 1 in this case, since that's the starting item
findNextItem(CATdesign)

# determine next item if item 1 and item 10 were answered correctly, and Theta = 0.5
CATdesign <- updateDesign(CATdesign, items = c(1, 10), responses = c(1, 1), Theta = 0.5)
findNextItem(CATdesign)
findNextItem(CATdesign, all_index = TRUE) # all items rank in terms of most optimal

# alternatively, update the Theta using the internal ReferenceClass method
Person$help('Update.thetas') # internal help file for class 'Person'
CATdesign$person$Update.thetas(CATdesign$design, CATdesign$test) 
findNextItem(CATdesign)


#-------------------------------------------------------------
## Integer programming example (e.g., shadow testing)

# find maximum information subject to constraints
#  sum(xi) <= 5               ### 5 or fewer items
#  x1 + x2 <= 1               ### items 1 and 2 can't be together
#  x4 == 0                    ### item 4 not included
#  x5 + x6 == 1               ### item 5 or 6 must be included, but not both

# constraint function
constr_fun <- function(person, test, design){

  # left hand side constrains 
  #    - 1 row per constraint, and ncol must equal number of items
  mo <- extract.mirtCAT(test, 'mo')
  nitems <- extract.mirt(mo, 'nitems')
  lhs <- matrix(0, 4, nitems)
  lhs[1,] <- 1
  lhs[2,c(1,2)] <- 1
  lhs[3, 4] <- 1
  lhs[4, c(5,6)] <- 1
  
  # relationship direction
  dirs <- c("<=", "<=", '==', '==')
  
  #right hand side
  rhs <- c(5, 1, 0, 1)

  #all together
  constraints <- data.frame(lhs, dirs, rhs)
  constraints
}

#### CATdesign <- mirtCAT(..., design_elements = TRUE, 
###                       design = list(constr_fun=constr_fun))

#' # MI criteria value associated with each respective item
objective <- computeCriteria(CATdesign, criteria = 'MI')

# most optimal item, given constraints
findNextItem(CATdesign, objective=objective)

# all the items which solve the problem
findNextItemp(CATdesign, objective=objective, all_index = TRUE)

## within a customNextItem() definition the above code would look like
# customNextItem <- function(person, design, test){
#   objective <- computeCriteria(person=person, design=design, test=test, 
#                                criteria = 'MI') 
#   item <- findNextItem(person=person, design=design, test=test,
#                        objective=objective)
#   item
# }

}
}
\author{
Phil Chalmers \email{rphilip.chalmers@gmail.com}
}
\references{
van der Linden, W. J. (2005). Linear models for optimal test design. Springer.
}
\seealso{
\code{\link{mirtCAT}}, \code{\link{updateDesign}}, \code{\link{extract.mirtCAT}}
}

